#include "fd_staked_rate_limit.h"
#include "fd_util.h"

/* Mock tick counter for deterministic testing.
   Assume 2.4 GHz CPU for testing (2.4 ticks per ns). */

static long   mock_tick        = 0L;
static double mock_tick_per_ns = 2.4;

static long
mock_tickcount( void ) {
  return mock_tick;
}

static void
advance_time_ms( long ms ) {
  mock_tick += (long)((double)ms * 1e6 * mock_tick_per_ns);
}

static void
advance_time_sec( long sec ) {
  mock_tick += (long)((double)sec * 1e9 * mock_tick_per_ns);
}

/* Test unstaked rate limiting (IPs not in the staked set use global limit) */

static void
test_unstaked_rate_limit( void ) {
  FD_LOG_NOTICE(( "Testing unstaked rate limiting" ));

  fd_staked_rate_limit_t rl[1];
  mock_tick = (long)(1e9 * mock_tick_per_ns);

  /* 100 pps unstaked (global), 10 pps per staked IP */
  fd_staked_rate_limit_init( rl, 100UL, 100UL, 10UL, 10UL, mock_tick_per_ns, mock_tickcount() );

  uint ip1 = 0x01020304U;
  uint ip2 = 0x05060708U;

  /* Both IPs are unstaked, they share the global unstaked limit of 100 */
  for( ulong i = 0UL; i < 50UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, ip1 ) == 1 );
  }
  for( ulong i = 0UL; i < 50UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, ip2 ) == 1 );
  }

  /* 101st packet should be blocked (global unstaked limit exhausted) */
  FD_TEST( fd_staked_rate_limit_check( rl, ip1 ) == 0 );
  FD_TEST( fd_staked_rate_limit_check( rl, ip2 ) == 0 );

  /* Advance time by 100ms (should add 10 tokens to global) */
  advance_time_ms( 100L );
  fd_staked_rate_limit_refresh( rl, mock_tickcount() );

  /* Should allow 10 more packets total */
  for( ulong i = 0UL; i < 10UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, ip1 ) == 1 );
  }
  FD_TEST( fd_staked_rate_limit_check( rl, ip1 ) == 0 );

  FD_LOG_NOTICE(( "Unstaked rate limiting: PASS" ));
}

/* Test staked IP rate limiting (each staked IP gets its own limit) */

static void
test_staked_rate_limit( void ) {
  FD_LOG_NOTICE(( "Testing staked rate limiting" ));

  fd_staked_rate_limit_t rl[1];
  mock_tick = (long)(2e9 * mock_tick_per_ns);

  /* 100 pps unstaked, 10 pps per staked IP */
  fd_staked_rate_limit_init( rl, 100UL, 100UL, 10UL, 10UL, mock_tick_per_ns, mock_tickcount() );

  uint staked_ip1 = 0x0A0B0C0DU;
  uint staked_ip2 = 0x0E0F1011U;

  /* Add staked IPs */
  FD_TEST( fd_staked_rate_limit_ip_add( rl, staked_ip1 ) == 1 );
  FD_TEST( fd_staked_rate_limit_ip_add( rl, staked_ip2 ) == 1 );

  /* Each staked IP should have its own 10 token limit */
  for( ulong i = 0UL; i < 10UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, staked_ip1 ) == 1 );
  }
  FD_TEST( fd_staked_rate_limit_check( rl, staked_ip1 ) == 0 );

  /* staked_ip2 should still have all its tokens */
  for( ulong i = 0UL; i < 10UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, staked_ip2 ) == 1 );
  }
  FD_TEST( fd_staked_rate_limit_check( rl, staked_ip2 ) == 0 );

  /* Unstaked limit should still be full (100 tokens) */
  uint unstaked_ip = 0xAABBCCDDU;
  for( ulong i = 0UL; i < 100UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, unstaked_ip ) == 1 );
  }
  FD_TEST( fd_staked_rate_limit_check( rl, unstaked_ip ) == 0 );

  /* Advance time by 1 second (should refill all buckets) */
  advance_time_sec( 1L );
  fd_staked_rate_limit_refresh( rl, mock_tickcount() );

  /* All should have tokens again */
  FD_TEST( fd_staked_rate_limit_check( rl, staked_ip1 ) == 1 );
  FD_TEST( fd_staked_rate_limit_check( rl, staked_ip2 ) == 1 );
  FD_TEST( fd_staked_rate_limit_check( rl, unstaked_ip ) == 1 );

  FD_LOG_NOTICE(( "Staked rate limiting: PASS" ));
}

/* Test staked vs unstaked isolation */

static void
test_staked_unstaked_isolation( void ) {
  FD_LOG_NOTICE(( "Testing staked vs unstaked isolation" ));

  fd_staked_rate_limit_t rl[1];
  mock_tick = (long)(3e9 * mock_tick_per_ns);

  /* Small limits: 20 pps unstaked, 5 pps per staked */
  fd_staked_rate_limit_init( rl, 20UL, 20UL, 5UL, 5UL, mock_tick_per_ns, mock_tickcount() );

  uint staked_ip   = 0x12345678U;
  uint unstaked_ip = 0x87654321U;

  FD_TEST( fd_staked_rate_limit_ip_add( rl, staked_ip ) == 1 );

  /* Exhaust the staked IP's tokens */
  for( ulong i = 0UL; i < 5UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 1 );
  }
  FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 0 );

  /* Unstaked limit should still be full - staked IP did NOT consume from global */
  for( ulong i = 0UL; i < 20UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, unstaked_ip ) == 1 );
  }
  FD_TEST( fd_staked_rate_limit_check( rl, unstaked_ip ) == 0 );

  /* Staked IP still can't send (its bucket is empty) */
  FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 0 );

  FD_LOG_NOTICE(( "Staked vs unstaked isolation: PASS" ));
}

/* Test burst tolerance */

static void
test_burst( void ) {
  FD_LOG_NOTICE(( "Testing burst tolerance" ));

  fd_staked_rate_limit_t rl[1];
  mock_tick = (long)(4e9 * mock_tick_per_ns);

  /* 10 pps with burst of 50 for both tiers */
  fd_staked_rate_limit_init( rl, 10UL, 50UL, 10UL, 50UL, mock_tick_per_ns, mock_tickcount() );

  uint staked_ip = 0xDEADBEEFU;
  FD_TEST( fd_staked_rate_limit_ip_add( rl, staked_ip ) == 1 );

  /* Should allow burst of 50 for staked IP */
  for( ulong i = 0UL; i < 50UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 1 );
  }
  FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 0 );

  /* Wait 1 sec (adds 10 tokens, capped at 50) */
  advance_time_sec( 1L );
  fd_staked_rate_limit_refresh( rl, mock_tickcount() );

  for( ulong i = 0UL; i < 10UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 1 );
  }
  FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 0 );

  /* Wait 10 sec (would add 100 tokens, capped at 50) */
  advance_time_sec( 10L );
  fd_staked_rate_limit_refresh( rl, mock_tickcount() );

  for( ulong i = 0UL; i < 50UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 1 );
  }
  FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 0 );

  FD_LOG_NOTICE(( "Burst tolerance: PASS" ));
}

/* Test adding duplicate IPs */

static void
test_duplicate_ip( void ) {
  FD_LOG_NOTICE(( "Testing duplicate IP handling" ));

  fd_staked_rate_limit_t rl[1];
  mock_tick = (long)(5e9 * mock_tick_per_ns);

  fd_staked_rate_limit_init( rl, 100UL, 100UL, 10UL, 10UL, mock_tick_per_ns, mock_tickcount() );

  uint ip = 0x11223344U;

  /* Add same IP twice - should succeed (dedup) */
  FD_TEST( fd_staked_rate_limit_ip_add( rl, ip ) == 1 );
  FD_TEST( fd_staked_rate_limit_ip_add( rl, ip ) == 1 );
  FD_TEST( rl->staked_ips_len == 1 );

  for( ulong i = 0UL; i < 10UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, ip ) == 1 );
  }
  FD_TEST( fd_staked_rate_limit_check( rl, ip ) == 0 );

  FD_LOG_NOTICE(( "Duplicate IP handling: PASS" ));
}

/* Test unlimited rates */

static void
test_unlimited( void ) {
  FD_LOG_NOTICE(( "Testing unlimited rates" ));

  fd_staked_rate_limit_t rl[1];
  mock_tick = (long)(6e9 * mock_tick_per_ns);

  uint staked_ip   = 0xAABBCCDDU;
  uint unstaked_ip = 0x11111111U;

  /* Case 1: Unlimited unstaked, limited staked */
  fd_staked_rate_limit_init( rl, 0UL, 0UL, 10UL, 10UL, mock_tick_per_ns, mock_tickcount() );
  FD_TEST( fd_staked_rate_limit_ip_add( rl, staked_ip ) == 1 );

  for( ulong i = 0UL; i < 1000UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, unstaked_ip ) == 1 );
  }
  for( ulong i = 0UL; i < 10UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 1 );
  }
  FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 0 );

  /* Case 2: Limited unstaked, unlimited staked */
  fd_staked_rate_limit_init( rl, 10UL, 10UL, 0UL, 0UL, mock_tick_per_ns, mock_tickcount() );
  FD_TEST( fd_staked_rate_limit_ip_add( rl, staked_ip ) == 1 );

  for( ulong i = 0UL; i < 1000UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 1 );
  }
  for( ulong i = 0UL; i < 10UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, unstaked_ip ) == 1 );
  }
  FD_TEST( fd_staked_rate_limit_check( rl, unstaked_ip ) == 0 );

  /* Case 3: Both unlimited */
  fd_staked_rate_limit_init( rl, 0UL, 0UL, 0UL, 0UL, mock_tick_per_ns, mock_tickcount() );
  FD_TEST( fd_staked_rate_limit_ip_add( rl, staked_ip ) == 1 );

  for( ulong i = 0UL; i < 1000UL; i++ ) {
    FD_TEST( fd_staked_rate_limit_check( rl, unstaked_ip ) == 1 );
    FD_TEST( fd_staked_rate_limit_check( rl, staked_ip ) == 1 );
  }

  FD_LOG_NOTICE(( "Unlimited rates: PASS" ));
}

/* Test clearing and rebuilding staked IPs */

static void
test_clear_and_rebuild( void ) {
  FD_LOG_NOTICE(( "Testing clear and rebuild" ));

  fd_staked_rate_limit_t rl[1];
  mock_tick = (long)(7e9 * mock_tick_per_ns);

  fd_staked_rate_limit_init( rl, 100UL, 100UL, 10UL, 10UL, mock_tick_per_ns, mock_tickcount() );

  uint ip1 = 0x11111111U;
  uint ip2 = 0x22222222U;

  /* Add IP1 as staked */
  FD_TEST( fd_staked_rate_limit_ip_add( rl, ip1 ) == 1 );

  FD_TEST( fd_staked_rate_limit_contains_ip( rl, ip1 ) == 1 );
  FD_TEST( fd_staked_rate_limit_contains_ip( rl, ip2 ) == 0 );

  /* Clear and add IP2 instead */
  fd_staked_rate_limit_clear_ips( rl );
  FD_TEST( fd_staked_rate_limit_ip_add( rl, ip2 ) == 1 );

  FD_TEST( fd_staked_rate_limit_contains_ip( rl, ip1 ) == 0 );
  FD_TEST( fd_staked_rate_limit_contains_ip( rl, ip2 ) == 1 );

  FD_LOG_NOTICE(( "Clear and rebuild: PASS" ));
}

/* Test max capacity */

static void
test_max_staked_ips( void ) {
  FD_LOG_NOTICE(( "Testing max staked IPs capacity" ));

  fd_staked_rate_limit_t rl[1];
  mock_tick = (long)(8e9 * mock_tick_per_ns);

  fd_staked_rate_limit_init( rl, 100UL, 100UL, 10UL, 10UL, mock_tick_per_ns, mock_tickcount() );

  /* Fill to capacity */
  for( uint i = 0U; i < FD_RATE_LIMIT_STAKED_IPS_MAX; i++ ) {
    FD_TEST( fd_staked_rate_limit_ip_add( rl, i + 1U ) == 1 );
  }
  FD_TEST( rl->staked_ips_len == FD_RATE_LIMIT_STAKED_IPS_MAX );

  /* Next add should fail */
  FD_TEST( fd_staked_rate_limit_ip_add( rl, 0xFFFFFFFFU ) == 0 );
  FD_TEST( rl->staked_ips_len == FD_RATE_LIMIT_STAKED_IPS_MAX );

  /* All added IPs should be findable */
  for( uint i = 0U; i < FD_RATE_LIMIT_STAKED_IPS_MAX; i++ ) {
    FD_TEST( fd_staked_rate_limit_contains_ip( rl, i + 1U ) == 1 );
  }

  /* The one that failed should not be found */
  FD_TEST( fd_staked_rate_limit_contains_ip( rl, 0xFFFFFFFFU ) == 0 );

  FD_LOG_NOTICE(( "Max staked IPs capacity: PASS" ));
}

/* Benchmark refresh performance */

static void
bench_refresh( void ) {
  FD_LOG_NOTICE(( "Benchmarking refresh" ));

  fd_staked_rate_limit_t rl[1];
  double tick_per_ns = mock_tick_per_ns;

  fd_staked_rate_limit_init( rl, 1000000UL, 2000000UL, 10000UL, 20000UL, tick_per_ns, fd_tickcount() );

  for( uint i = 0U; i < 100U; i++ ) {
    fd_staked_rate_limit_ip_add( rl, 0xC0A80001U + i );
  }

  uint         base_ip = 0xC0A80001U;
  volatile int sink    = 0;

  for( ulong i = 0UL; i < 100000UL; i++ ) {
    sink += fd_staked_rate_limit_check( rl, (uint)(base_ip + (i % 100U)) );
  }

  ulong iter       = 100000UL;
  long  start_wall = fd_log_wallclock();
  long  now        = fd_tickcount();

  for( ulong i = 0UL; i < iter; i++ ) {
    now += (long)(1e6 * tick_per_ns);
    fd_staked_rate_limit_refresh( rl, now );

    for( ulong j = 0UL; j < 10UL; j++ ) {
      sink += fd_staked_rate_limit_check( rl, (uint)(base_ip + ((i + j) % 100U)) );
    }
  }

  long   elapsed_ns     = fd_log_wallclock() - start_wall;
  double ns_per_refresh = (double)elapsed_ns / (double)iter;

  FD_LOG_NOTICE(( "Refresh: %.2f ns/refresh (%lu refreshes in %ld ns, sink=%d)",
                  ns_per_refresh, iter, elapsed_ns, sink ));
}

/* Benchmark hot path performance */

static void
bench_hot_path( void ) {
  FD_LOG_NOTICE(( "Benchmarking hot path" ));

  fd_staked_rate_limit_t rl[1];
  double tick_per_ns = mock_tick_per_ns;

  fd_staked_rate_limit_init( rl, 11000000UL, 11000000UL, 11000000UL, 11000000UL, tick_per_ns, fd_tickcount() );

  /* Add 7 staked IPs (realistic validator set size) */
  uint staked_ips[7];
  for( uint i = 0U; i < 7U; i++ ) {
    staked_ips[i] = 0xDEAD0000U + i;
    fd_staked_rate_limit_ip_add( rl, staked_ips[i] );
  }

  uint unstaked_ip = 0xBEEF0000U;

  /* Warmup */
  for( ulong i = 0UL; i < 1000UL; i++ ) {
    for( uint j = 0U; j < 7U; j++ ) {
      fd_staked_rate_limit_check( rl, staked_ips[j] );
    }
    fd_staked_rate_limit_check( rl, unstaked_ip );
  }

  /* Benchmark staked IP lookup - loop over all 7 staked IPs */
  ulong        iter  = 5500000UL;
  long         start = fd_log_wallclock();
  volatile int sink  = 0;

  for( ulong i = 0UL; i < iter; i++ ) {
    for( uint j = 0U; j < 7U; j++ ) {
      sink += fd_staked_rate_limit_check( rl, staked_ips[j] );
    }
  }

  long   elapsed_ns         = fd_log_wallclock() - start;
  double ns_per_check_staked = (double)elapsed_ns / (double)(iter * 7UL);

  /* Benchmark unstaked IP lookup */
  fd_staked_rate_limit_init( rl, 11000000UL, 11000000UL, 11000000UL, 11000000UL, tick_per_ns, fd_tickcount() );
  for( uint i = 0U; i < 7U; i++ ) {
    fd_staked_rate_limit_ip_add( rl, 0xDEAD0000U + i );
  }

  start = fd_log_wallclock();
  for( ulong i = 0UL; i < iter; i++ ) {
    sink += fd_staked_rate_limit_check( rl, (uint)(unstaked_ip + i) );
  }

  elapsed_ns = fd_log_wallclock() - start;
  double ns_per_check_unstaked = (double)elapsed_ns / (double)iter;

  FD_LOG_NOTICE(( "Hot path staked: %.2f ns/check, unstaked: %.2f ns/check (sink=%d)",
                  ns_per_check_staked, ns_per_check_unstaked, sink ));
}

int
main( int     argc,
      char ** argv ) {
  fd_boot( &argc, &argv );

  test_unstaked_rate_limit();
  test_staked_rate_limit();
  test_staked_unstaked_isolation();
  test_burst();
  test_duplicate_ip();
  test_unlimited();
  test_clear_and_rebuild();
  test_max_staked_ips();
  bench_refresh();
  bench_hot_path();

  FD_LOG_NOTICE(( "All tests PASS" ));
  fd_halt();
  return 0;
}
