/* Rate limiter for XDP packet processing

   Two-tiered rate limiting:
   - Unstaked (global): total packets/sec across all unstaked IPs
   - Per-staked-IP: packets/sec per staked IP

   Design for XDP hot path:
   - Uses tick-based accounting
   - Open-addressed hash table for staked IP lookup
   - Manual refresh in housekeeping keeps hot path minimal
   - Contact infos can be loaded to use the per-IP limit */

#ifndef HEADER_fd_src_util_fd_staked_rate_limit_h
#define HEADER_fd_src_util_fd_staked_rate_limit_h

#include "fd_util_base.h"
#include "../disco/fd_disco.h"

#include <stdint.h>
#include <string.h>

#define FD_RATE_LIMIT_STAKED_IPS_MAX  512U

/* Hash table size */
#define FD_RATE_LIMIT_MAP_SIZE        (FD_RATE_LIMIT_STAKED_IPS_MAX * 8)
#define FD_RATE_LIMIT_MAP_MASK        (FD_RATE_LIMIT_MAP_SIZE - 1U)

/* Sentinel value for empty hash slots (0 is invalid IP anyway) */
#define FD_RATE_LIMIT_IP_EMPTY        0U

/* Main rate limiter state.

   All token counts are signed integers. Negative means depleted.
   Tokens represent packets (1 token = 1 packet).

   Each packet consumes from either the unstaked limit (global) or
   the staked per-source-IP bucket.

   Hash table layout:
   - map_ip[i]: IP address at slot i (0 = empty)
   - map_tokens[i]: token count for IP at slot i */

struct __attribute__((aligned(64))) fd_staked_rate_limit {
  uint32_t map_ip    [ FD_RATE_LIMIT_MAP_SIZE ];
  long     map_tokens[ FD_RATE_LIMIT_MAP_SIZE ];

  long     unstaked_tokens;
  long     last_refresh_tick;
  double   unstaked_tokens_per_tick;  /* 0.0 = unlimited */
  long     unstaked_capacity;
  double   ip_tokens_per_tick;        /* 0.0 = unlimited */
  long     ip_capacity;
  uint32_t staked_ips_len;
};

typedef struct fd_staked_rate_limit fd_staked_rate_limit_t;

/* Fast hash for IPv4 addresses using xorshift multiply.
   Good distribution, very fast on modern CPUs. */

static inline uint32_t
fd_rate_limit_hash( uint32_t ip ) {
  uint32_t h = ip;
  h ^= h >> 16;
  h *= 0x85ebca6bU;
  h ^= h >> 13;
  h *= 0xc2b2ae35U;
  h ^= h >> 16;
  return h;
}

/* fd_staked_rate_limit_clear_ips clears all staked IPs from the rate limiter. */

static inline void
fd_staked_rate_limit_clear_ips( fd_staked_rate_limit_t * rl ) {
  rl->staked_ips_len = 0;
  memset( rl->map_ip,     0, sizeof(rl->map_ip)     );
  memset( rl->map_tokens, 0, sizeof(rl->map_tokens) );
}

/* fd_staked_rate_limit_init initializes rate limiter.

   unstaked_rate_per_sec: Max packets/sec for unstaked IPs (0 = unlimited)
   unstaked_burst:        Max burst size for unstaked (tokens)
   ip_rate_per_sec:       Max packets/sec per staked IP (0 = unlimited)
   ip_burst:              Max burst size per staked IP (tokens)
   tick_per_ns:           Tick rate (e.g. ~2.4 for 2.4 GHz CPU)
   now_tick:              Current tick (from fd_tickcount())

   Burst size should typically be >= rate to allow for some burstiness.
   Common pattern: burst = rate * 2 for 2 seconds of burst tolerance. */

static inline void
fd_staked_rate_limit_init( fd_staked_rate_limit_t * rl,
                           ulong                    unstaked_rate_per_sec,
                           ulong                    unstaked_burst,
                           ulong                    ip_rate_per_sec,
                           ulong                    ip_burst,
                           double                   tick_per_ns,
                           long                     now_tick ) {

  double ticks_per_sec = tick_per_ns * 1e9;
  FD_TEST((ticks_per_sec != 0));

  if (unstaked_rate_per_sec == 0UL) {
    rl->unstaked_tokens_per_tick = 0.0;
    rl->unstaked_capacity = LONG_MAX;
    rl->unstaked_tokens = LONG_MAX;
  } else {
    rl->unstaked_tokens_per_tick = (double)unstaked_rate_per_sec / ticks_per_sec;
    rl->unstaked_capacity = (long)unstaked_burst;
    rl->unstaked_tokens = (long)unstaked_burst;
  }

  if (ip_rate_per_sec == 0UL) {
    rl->ip_tokens_per_tick = 0.0;
    rl->ip_capacity = LONG_MAX;
  } else {
    rl->ip_tokens_per_tick = (double)ip_rate_per_sec / ticks_per_sec;
    rl->ip_capacity = (long)ip_burst;
  }

  rl->last_refresh_tick = now_tick;

  fd_staked_rate_limit_clear_ips( rl );
}

FD_FN_PURE static inline long *
fd_staked_rate_limit_lookup( fd_staked_rate_limit_t * rl,
                             uint32_t                 ip ) {
  uint32_t slot = fd_rate_limit_hash( ip ) & FD_RATE_LIMIT_MAP_MASK;

  for( uint32_t probe = 0; probe < FD_RATE_LIMIT_MAP_SIZE; probe++ ) {
    uint32_t cur_ip = rl->map_ip[ slot ];
    if( FD_LIKELY( cur_ip == ip ) ) return &rl->map_tokens[ slot ];
    if( FD_LIKELY( cur_ip == FD_RATE_LIMIT_IP_EMPTY ) ) return NULL;
    slot = (slot + 1U) & FD_RATE_LIMIT_MAP_MASK;
  }

  return NULL;
}

/* fd_staked_rate_limit_contains_ip checks if ip is in the staked set.
   Returns 1 if found, 0 if not found. */

static inline int
fd_staked_rate_limit_contains_ip( fd_staked_rate_limit_t const * rl,
                                  uint32_t                       ip ) {
  uint32_t slot = fd_rate_limit_hash( ip ) & FD_RATE_LIMIT_MAP_MASK;

  for( uint32_t probe = 0; probe < FD_RATE_LIMIT_MAP_SIZE; probe++ ) {
    uint32_t cur_ip = rl->map_ip[ slot ];
    if( cur_ip == ip ) return 1;
    if( cur_ip == FD_RATE_LIMIT_IP_EMPTY ) return 0;
    slot = (slot + 1U) & FD_RATE_LIMIT_MAP_MASK;
  }

  return 0;
}

/* fd_staked_rate_limit_refresh replenishes tokens based on elapsed ticks.
   Call periodically (e.g. every 1-10ms) from housekeeping. */

FD_FN_UNUSED static inline void
fd_staked_rate_limit_refresh( fd_staked_rate_limit_t * rl,
                              long                     now_tick ) {

  long elapsed_ticks = now_tick - rl->last_refresh_tick;
  if( FD_UNLIKELY( elapsed_ticks <= 0L ) ) return;

  rl->last_refresh_tick = now_tick;

  /* Replenish unstaked tokens */
  if( FD_LIKELY( rl->unstaked_tokens_per_tick > 0.0 ) ) {
    long tokens_to_add = (long)(rl->unstaked_tokens_per_tick * (double)elapsed_ticks);
    rl->unstaked_tokens = (tokens_to_add > rl->unstaked_capacity - rl->unstaked_tokens) ? rl->unstaked_capacity : rl->unstaked_tokens + tokens_to_add;
  }

  /* Replenish staked per-IP tokens */
  if( FD_LIKELY( rl->ip_tokens_per_tick > 0.0 ) ) {
    long tokens_to_add = (long)(rl->ip_tokens_per_tick * (double)elapsed_ticks);

    for( uint32_t i = 0; i < FD_RATE_LIMIT_MAP_SIZE; i++ ) {
      if( rl->map_ip[ i ] != FD_RATE_LIMIT_IP_EMPTY ) {
        rl->map_tokens[ i ] = (tokens_to_add > rl->ip_capacity - rl->map_tokens[ i ]) ? rl->ip_capacity : rl->map_tokens[ i ] + tokens_to_add;
      }
    }
  }
}

/* fd_staked_rate_limit_check checks if packet should be allowed (HOT PATH).
   Returns 1 if allowed, 0 if should be dropped.
   ip_addr: IPv4 address in big-endian format */

FD_FN_UNUSED static inline int
fd_staked_rate_limit_check( fd_staked_rate_limit_t * rl,
                            uint                     ip_addr ) {

  long * tokens = fd_staked_rate_limit_lookup( rl, ip_addr );

  if( FD_LIKELY( tokens != NULL ) ) {
    /* IP has its own token bucket */
    int ok = *tokens > 0L;
    *tokens -= ok;
    return ok;
  }

  /* IP is unstaked, use global bucket */
  int ok = rl->unstaked_tokens > 0L;
  rl->unstaked_tokens -= ok;
  return ok;
}

static inline int
fd_staked_rate_limit_ip_add( fd_staked_rate_limit_t * rl,
                             uint32_t                 ip ) {

  if( ip == FD_RATE_LIMIT_IP_EMPTY ) return 0;  /* 0 is reserved sentinel */
  if( rl->staked_ips_len >= FD_RATE_LIMIT_STAKED_IPS_MAX ) return 0;

  uint32_t slot = fd_rate_limit_hash( ip ) & FD_RATE_LIMIT_MAP_MASK;

  for( uint32_t probe = 0; probe < FD_RATE_LIMIT_MAP_SIZE; probe++ ) {
    uint32_t cur_ip = rl->map_ip[ slot ];

    if( cur_ip == ip ) {
      /* Already present */
      return 1;
    }

    if( cur_ip == FD_RATE_LIMIT_IP_EMPTY ) {
      /* Found empty slot, insert */
      rl->map_ip[ slot ]     = ip;
      rl->map_tokens[ slot ] = rl->ip_capacity;
      rl->staked_ips_len++;
      return 1;
    }

    slot = (slot + 1U) & FD_RATE_LIMIT_MAP_MASK;
  }

  return 0;  /* Table full (shouldn't happen with proper sizing) */
}

/* fd_staked_rate_limit_load_contacts loads new contact infos.
   Preserves token counts for IPs that were already in the set. */

static inline void
fd_staked_rate_limit_load_contacts( fd_staked_rate_limit_t *     rl,
                                    fd_shred_dest_wire_t const * dest,
                                    ulong                        dest_cnt ) {

  /* Copy previous map so we can preserve token counts */
  uint32_t prev_ip    [ FD_RATE_LIMIT_MAP_SIZE ];
  long     prev_tokens[ FD_RATE_LIMIT_MAP_SIZE ];
  fd_memcpy( prev_ip,     rl->map_ip,     sizeof(prev_ip)     );
  fd_memcpy( prev_tokens, rl->map_tokens, sizeof(prev_tokens) );

  fd_staked_rate_limit_clear_ips( rl );

  /* Iterate over all IPs that need to be in the set now */
  for( ulong i = 0UL; i < dest_cnt; i++ ) {
    if( dest[ i ].stake == 0UL ) continue;
    if( dest[ i ].ip4_addr == FD_RATE_LIMIT_IP_EMPTY ) continue;

    if( FD_UNLIKELY( !fd_staked_rate_limit_ip_add( rl, dest[ i ].ip4_addr ) ) ) {
      FD_LOG_WARNING(( "Staked IP rate limiter full: %u", rl->staked_ips_len ));
      continue;
    }

    /* Check if it existed previously and restore tokens */
    uint32_t slot = fd_rate_limit_hash( dest[ i ].ip4_addr ) & FD_RATE_LIMIT_MAP_MASK;
    for( uint32_t probe = 0; probe < FD_RATE_LIMIT_MAP_SIZE; probe++ ) {
      if( prev_ip[ slot ] == dest[ i ].ip4_addr ) {
        /* Found in old map, restore tokens to new slot */
        long * tokens = fd_staked_rate_limit_lookup( rl, dest[ i ].ip4_addr );
        if( tokens ) *tokens = prev_tokens[ slot ];
        break;
      }
      if( prev_ip[ slot ] == FD_RATE_LIMIT_IP_EMPTY ) break;
      slot = (slot + 1U) & FD_RATE_LIMIT_MAP_MASK;
    }
  }
}

#endif /* HEADER_fd_src_util_fd_staked_rate_limit_h */
