// fd_ip_wl.h — single-header
#pragma once
#include <stdint.h>
#include <stdbool.h>
#include <string.h>
#include <stdio.h>

// maximum IPs accepted from config
#define FD_WL_MAX 512
#define FD_WL_BUCKET_BITS 10
#define FD_WL_BUCKETS (1u << FD_WL_BUCKET_BITS)

// IP whitelist container
typedef struct fd_ip_wl {
    uint32_t keys[FD_WL_MAX];                 // raw loaded keys (big-endian)
    uint32_t keys_bucketed[FD_WL_MAX];        // contiguous per-bucket layout
    uint32_t index[FD_WL_BUCKETS + 1];        // prefix index per bucket
    uint32_t len;                             // number of loaded keys
    uint8_t  built;                           // 1 after build()
} fd_ip_wl;

static inline void fd_ip_wl_init(fd_ip_wl *w) {
    w->len = 0;
    w->built = 0;
    // zero out arrays
    memset(w->keys, 0, sizeof w->keys);
    memset(w->keys_bucketed, 0, sizeof w->keys_bucketed);
    memset(w->index, 0, sizeof w->index);
}

// Parse dotted IPv4 "a.b.c.d" -> big-endian u32
static inline bool fd_ip_wl_ipv4_dotted_to_be(const char *s, uint32_t *out_be) {
    // tolerant space handling
    unsigned long a=0,b=0,c=0,d=0; char tail;
    int n = sscanf(s, " %lu . %lu . %lu . %lu %c", &a,&b,&c,&d,&tail);
    if (n < 4) return false;
    if (a>255 || b>255 || c>255 || d>255) return false;
    *out_be = ((uint32_t)a<<24) | ((uint32_t)b<<16) | ((uint32_t)c<<8) | (uint32_t)d;
    return true;
}

static inline bool fd_ip_wl_add(fd_ip_wl *w, uint32_t key_be) {
    // at capacity
    if (w->len >= FD_WL_MAX) return false;
    // dedup
    for (uint32_t i = 0; i < w->len; ++i)
        // already present, treat as success
        if (w->keys[i] == key_be) return true;
    w->keys[w->len++] = key_be;
    w->built = 0;
    return true;
}

// Build bucket index and compacted layout
static inline void fd_ip_wl_build(fd_ip_wl *w) {
    // count per bucket
    uint32_t counts[FD_WL_BUCKETS];
    memset(counts, 0, sizeof counts);

    for (uint32_t i = 0; i < w->len; ++i) {
        uint32_t b = w->keys[i] >> (32 - FD_WL_BUCKET_BITS);
        counts[b]++;
    }

    // prefix sum -> index[]
    uint32_t sum = 0;
    for (uint32_t b = 0; b < FD_WL_BUCKETS; ++b) {
        w->index[b] = sum;
        sum += counts[b];
    }
    w->index[FD_WL_BUCKETS] = sum; // == w->len

    // fill contiguous per-bucket array
    uint32_t curs[FD_WL_BUCKETS];
    memcpy(curs, w->index, FD_WL_BUCKETS * sizeof(uint32_t));

    for (uint32_t i = 0; i < w->len; ++i) {
        uint32_t k = w->keys[i];
        uint32_t b = k >> (32 - FD_WL_BUCKET_BITS);
        w->keys_bucketed[curs[b]++] = k;
    }
    w->built = 1;
}

// Membership test
FD_FN_UNUSED
static bool fd_ip_wl_contains(const fd_ip_wl *w, uint32_t key_be) {
    if (!w->built) return false;
    const uint32_t b = key_be >> (32 - FD_WL_BUCKET_BITS);
    uint32_t start = w->index[b];
    uint32_t end   = w->index[b + 1];
    if ( FD_UNLIKELY ( start == end ))  return false;

    uint32_t acc = 0;
    for (uint32_t i = start; i < end; ++i) {
        acc |= (w->keys_bucketed[i] == key_be);
    }
    return acc != 0;
}

// Clear
static inline void fd_ip_wl_clear(fd_ip_wl *w) {
    w->len = 0;
    w->built = 0;
}

static inline uint32_t fd_ip_wl_count(const fd_ip_wl *w) { return w->len; }
