#include "fd_netconn_ring.h"
#include "../../util/fd_util.h"
#include "../../util/tile/fd_tile.h"

/* Test basic push/pop operations */
static void
test_basic( void ) {
  uchar mem[ FD_NETCONN_RING_FOOTPRINT( 4 ) ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
  fd_netconn_ring_t * ring = fd_netconn_ring_new( mem, 4 );

  FD_TEST( ring );
  FD_TEST( fd_netconn_ring_avail( ring ) == 0 );

  uchar payload[] = { 0xDE, 0xAD, 0xBE, 0xEF };
  uint  src_ip    = 0x01020304;
  ushort src_port = 1234;

  FD_TEST( fd_netconn_ring_push( ring, src_ip, src_port, payload, sizeof(payload) ) == 1 );
  FD_TEST( fd_netconn_ring_avail( ring ) == 1 );

  uint   out_ip;
  ushort out_port;
  uchar  out_payload[ FD_NETCONN_RING_MTU ];

  ulong sz = fd_netconn_ring_pop( ring, &out_ip, &out_port, out_payload );
  FD_TEST( sz == sizeof(payload) );
  FD_TEST( out_ip == src_ip );
  FD_TEST( out_port == src_port );
  FD_TEST( memcmp( out_payload, payload, sizeof(payload) ) == 0 );
  FD_TEST( fd_netconn_ring_avail( ring ) == 0 );

  /* Pop from empty ring */
  FD_TEST( fd_netconn_ring_pop( ring, &out_ip, &out_port, out_payload ) == 0 );
}

/* Test ring drops when full */
static void
test_drop_on_full( void ) {
  uchar mem[ FD_NETCONN_RING_FOOTPRINT( 4 ) ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
  fd_netconn_ring_t * ring = fd_netconn_ring_new( mem, 4 );

  uchar payload[] = { 0x11 };

  /* Fill ring to capacity */
  for( int i = 0; i < 4; i++ ) {
    FD_TEST( fd_netconn_ring_push( ring, (uint)i, (ushort)i, payload, 1 ) == 1 );
  }
  FD_TEST( fd_netconn_ring_avail( ring ) == 4 );

  /* Next push should fail (drop) */
  FD_TEST( fd_netconn_ring_push( ring, 99, 99, payload, 1 ) == 0 );
  FD_TEST( fd_netconn_ring_avail( ring ) == 4 );

  /* Verify original data intact */
  uint   out_ip;
  ushort out_port;
  uchar  out_payload[ FD_NETCONN_RING_MTU ];

  for( int i = 0; i < 4; i++ ) {
    FD_TEST( fd_netconn_ring_pop( ring, &out_ip, &out_port, out_payload ) == 1 );
    FD_TEST( out_ip == (uint)i );
    FD_TEST( out_port == (ushort)i );
  }
}

/* Test wraparound behavior */
static void
test_wraparound( void ) {
  uchar mem[ FD_NETCONN_RING_FOOTPRINT( 4 ) ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
  fd_netconn_ring_t * ring = fd_netconn_ring_new( mem, 4 );

  uchar payload[] = { 0x33 };
  uint   out_ip;
  ushort out_port;
  uchar  out_payload[ FD_NETCONN_RING_MTU ];

  /* Push and pop many times to test wraparound */
  for( int round = 0; round < 100; round++ ) {
    for( int i = 0; i < 3; i++ ) {
      FD_TEST( fd_netconn_ring_push( ring, (uint)(round * 10 + i), (ushort)i, payload, 1 ) == 1 );
    }
    for( int i = 0; i < 3; i++ ) {
      FD_TEST( fd_netconn_ring_pop( ring, &out_ip, &out_port, out_payload ) == 1 );
      FD_TEST( out_ip == (uint)(round * 10 + i) );
    }
    FD_TEST( fd_netconn_ring_avail( ring ) == 0 );
  }
}

/* Test payload size clamping to MTU */
static void
test_mtu_clamp( void ) {
  uchar mem[ FD_NETCONN_RING_FOOTPRINT( 4 ) ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
  fd_netconn_ring_t * ring = fd_netconn_ring_new( mem, 4 );

  uchar big_payload[ FD_NETCONN_RING_MTU + 100 ];
  memset( big_payload, 0xAA, sizeof(big_payload) );

  FD_TEST( fd_netconn_ring_push( ring, 1, 2, big_payload, (ushort)sizeof(big_payload) ) == 1 );

  uint   out_ip;
  ushort out_port;
  uchar  out_payload[ FD_NETCONN_RING_MTU ];

  ulong sz = fd_netconn_ring_pop( ring, &out_ip, &out_port, out_payload );
  FD_TEST( sz == FD_NETCONN_RING_MTU );
}

/* Test interleaved push/pop */
static void
test_interleaved( void ) {
  uchar mem[ FD_NETCONN_RING_FOOTPRINT( 8 ) ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
  fd_netconn_ring_t * ring = fd_netconn_ring_new( mem, 8 );

  uchar payload[] = { 0x44 };
  uint   out_ip;
  ushort out_port;
  uchar  out_payload[ FD_NETCONN_RING_MTU ];

  /* Push 5, pop 2, push 3, pop 6 */
  for( int i = 0; i < 5; i++ ) {
    FD_TEST( fd_netconn_ring_push( ring, (uint)i, 0, payload, 1 ) == 1 );
  }
  FD_TEST( fd_netconn_ring_avail( ring ) == 5 );

  FD_TEST( fd_netconn_ring_pop( ring, &out_ip, &out_port, out_payload ) == 1 );
  FD_TEST( out_ip == 0 );
  FD_TEST( fd_netconn_ring_pop( ring, &out_ip, &out_port, out_payload ) == 1 );
  FD_TEST( out_ip == 1 );
  FD_TEST( fd_netconn_ring_avail( ring ) == 3 );

  for( int i = 5; i < 8; i++ ) {
    FD_TEST( fd_netconn_ring_push( ring, (uint)i, 0, payload, 1 ) == 1 );
  }
  FD_TEST( fd_netconn_ring_avail( ring ) == 6 );

  for( int i = 2; i < 8; i++ ) {
    FD_TEST( fd_netconn_ring_pop( ring, &out_ip, &out_port, out_payload ) == 1 );
    FD_TEST( out_ip == (uint)i );
  }
  FD_TEST( fd_netconn_ring_avail( ring ) == 0 );
}

/* Test footprint and alignment */
static void
test_footprint( void ) {
  FD_TEST( fd_netconn_ring_align() == 64UL );
  FD_TEST( fd_netconn_ring_footprint( 1 ) > sizeof(fd_netconn_ring_t) );
  FD_TEST( fd_netconn_ring_footprint( 4 ) == sizeof(fd_netconn_ring_t) + 4 * sizeof(fd_netconn_pkt_t) );
}

/* Test full detection and skip functionality */
static void
test_full_and_skip( void ) {
  uchar mem[ FD_NETCONN_RING_FOOTPRINT( 4 ) ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
  fd_netconn_ring_t * ring = fd_netconn_ring_new( mem, 4 );

  uchar payload[ 8 ] = { 1, 2, 3, 4, 5, 6, 7, 8 };
  uint   out_ip;
  ushort out_port;
  uchar  out_payload[ FD_NETCONN_RING_MTU ];

  /* Empty ring is not full */
  FD_TEST( !fd_netconn_ring_full( ring ) );
  FD_TEST( fd_netconn_ring_avail( ring ) == 0 );

  /* Fill the ring */
  for( uint i = 0; i < 4; i++ ) {
    FD_TEST( fd_netconn_ring_push( ring, i, (ushort)i, payload, 8 ) );
  }

  /* Ring should now be full */
  FD_TEST( fd_netconn_ring_full( ring ) );
  FD_TEST( fd_netconn_ring_avail( ring ) == 4 );
  FD_TEST( !fd_netconn_ring_push( ring, 99, 99, payload, 8 ) ); /* Push fails when full */

  /* Skip first two packets */
  FD_TEST( fd_netconn_ring_skip( ring ) == 1 );
  FD_TEST( fd_netconn_ring_avail( ring ) == 3 );
  FD_TEST( !fd_netconn_ring_full( ring ) );

  FD_TEST( fd_netconn_ring_skip( ring ) == 1 );
  FD_TEST( fd_netconn_ring_avail( ring ) == 2 );

  /* Pop should get packet with ip=2 (skipped 0 and 1) */
  ulong sz = fd_netconn_ring_pop( ring, &out_ip, &out_port, out_payload );
  FD_TEST( sz == 8 );
  FD_TEST( out_ip == 2 );
  FD_TEST( out_port == 2 );

  /* Pop last packet */
  sz = fd_netconn_ring_pop( ring, &out_ip, &out_port, out_payload );
  FD_TEST( sz == 8 );
  FD_TEST( out_ip == 3 );

  /* Ring is now empty */
  FD_TEST( fd_netconn_ring_avail( ring ) == 0 );
  FD_TEST( !fd_netconn_ring_full( ring ) );

  /* Skip on empty ring returns 0 */
  FD_TEST( fd_netconn_ring_skip( ring ) == 0 );
}

/* Stress test with many operations */
static void
test_stress( void ) {
  uchar mem[ FD_NETCONN_RING_FOOTPRINT( 1024 ) ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
  fd_netconn_ring_t * ring = fd_netconn_ring_new( mem, 1024 );

  fd_rng_t _rng[1];
  fd_rng_t * rng = fd_rng_join( fd_rng_new( _rng, 42U, 0UL ) );

  uchar payload[ 256 ];
  uint   out_ip;
  ushort out_port;
  uchar  out_payload[ FD_NETCONN_RING_MTU ];

  ulong pushed = 0;
  ulong popped = 0;

  for( int i = 0; i < 100000; i++ ) {
    if( fd_rng_ulong_roll( rng, 2 ) == 0 ) {
      ushort sz = (ushort)(1 + fd_rng_ulong_roll( rng, 255 ));
      if( fd_netconn_ring_push( ring, (uint)i, (ushort)i, payload, sz ) ) {
        pushed++;
      }
    } else {
      if( fd_netconn_ring_pop( ring, &out_ip, &out_port, out_payload ) ) {
        popped++;
      }
    }
  }

  /* Drain remaining */
  while( fd_netconn_ring_pop( ring, &out_ip, &out_port, out_payload ) ) {
    popped++;
  }

  FD_TEST( pushed == popped );
  FD_TEST( fd_netconn_ring_avail( ring ) == 0 );

  fd_rng_delete( fd_rng_leave( rng ) );
}

/* Concurrent SPSC test - producer and consumer on different threads */

#define CONCURRENT_RING_DEPTH 1024UL
#define CONCURRENT_MSG_CNT    1000000UL

static uchar concurrent_ring_mem[ FD_NETCONN_RING_FOOTPRINT( CONCURRENT_RING_DEPTH ) ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
static fd_netconn_ring_t * concurrent_ring;
static volatile int        producer_done;
static volatile ulong      producer_pushed;
static volatile ulong      producer_dropped;

static int
consumer_main( int     argc,
               char ** argv ) {
  (void)argc; (void)argv;

  uint   out_ip;
  ushort out_port;
  uchar  out_payload[ FD_NETCONN_RING_MTU ];

  ulong consumed  = 0UL;
  ulong expected  = 0UL;
  ulong bad_seq   = 0UL;
  ulong bad_data  = 0UL;

  while( !FD_VOLATILE_CONST( producer_done ) || fd_netconn_ring_avail( concurrent_ring ) ) {
    ulong sz = fd_netconn_ring_pop( concurrent_ring, &out_ip, &out_port, out_payload );
    if( sz == 0 ) {
      FD_SPIN_PAUSE();
      continue;
    }

    /* Verify sequence number is monotonically increasing (packets can be dropped but not reordered) */
    if( out_ip < (uint)expected ) {
      bad_seq++;
    }
    expected = (ulong)out_ip + 1UL;

    /* Verify payload integrity - first 8 bytes contain sequence XOR pattern */
    if( sz >= 8 ) {
      ulong * payload_seq = (ulong *)out_payload;
      if( *payload_seq != (out_ip ^ 0xDEADBEEFCAFEBABEUL) ) {
        bad_data++;
      }
    }

    /* Verify port matches low bits of sequence */
    if( out_port != (ushort)(out_ip & 0xFFFF) ) {
      bad_data++;
    }

    consumed++;
  }

  FD_TEST( bad_seq  == 0 );
  FD_TEST( bad_data == 0 );

  /* Verify we consumed exactly what was successfully pushed (dropped items never entered the ring) */
  ulong pushed  = FD_VOLATILE_CONST( producer_pushed );
  ulong dropped = FD_VOLATILE_CONST( producer_dropped );
  (void)dropped; /* Just for verification that pushed + dropped == CONCURRENT_MSG_CNT */
  FD_TEST( consumed == pushed );

  return 0;
}

static void
test_concurrent( void ) {
  concurrent_ring = fd_netconn_ring_new( concurrent_ring_mem, CONCURRENT_RING_DEPTH );
  FD_TEST( concurrent_ring );

  producer_done    = 0;
  producer_pushed  = 0;
  producer_dropped = 0;

  /* Start consumer on tile 1 */
  fd_tile_exec_t * consumer = fd_tile_exec_new( 1UL, consumer_main, 0, NULL );
  FD_TEST( consumer );

  fd_log_sleep( (long)1e7 ); /* 10ms - let consumer start */

  /* Producer loop on main thread */
  uchar payload[ 64 ];
  ulong pushed  = 0;
  ulong dropped = 0;

  for( ulong seq = 0; seq < CONCURRENT_MSG_CNT; seq++ ) {
    /* Encode sequence in payload for integrity check */
    ulong * payload_seq = (ulong *)payload;
    *payload_seq = seq ^ 0xDEADBEEFCAFEBABEUL;

    int ok = fd_netconn_ring_push( concurrent_ring,
                                   (uint)seq,
                                   (ushort)(seq & 0xFFFF),
                                   payload,
                                   sizeof(payload) );
    if( ok ) {
      pushed++;
    } else {
      dropped++;
    }

    /* Occasionally yield to let consumer catch up */
    if( (seq & 0xFFF) == 0 ) FD_SPIN_PAUSE();
  }

  FD_COMPILER_MFENCE();
  FD_VOLATILE( producer_pushed )  = pushed;
  FD_VOLATILE( producer_dropped ) = dropped;
  FD_VOLATILE( producer_done )    = 1;
  FD_COMPILER_MFENCE();

  /* Wait for consumer to finish */
  FD_TEST( !fd_tile_exec_delete( consumer, NULL ) );

  FD_LOG_NOTICE(( "concurrent: pushed %lu, dropped %lu (%.2f%% drop rate)",
                  pushed, dropped, 100.0 * (double)dropped / (double)(pushed + dropped) ));
}

int
main( int     argc,
      char ** argv ) {
  fd_boot( &argc, &argv );

  test_footprint();
  FD_LOG_NOTICE(( "OK: footprint" ));

  test_basic();
  FD_LOG_NOTICE(( "OK: basic" ));

  test_drop_on_full();
  FD_LOG_NOTICE(( "OK: drop_on_full" ));

  test_wraparound();
  FD_LOG_NOTICE(( "OK: wraparound" ));

  test_mtu_clamp();
  FD_LOG_NOTICE(( "OK: mtu_clamp" ));

  test_interleaved();
  FD_LOG_NOTICE(( "OK: interleaved" ));

  test_stress();
  FD_LOG_NOTICE(( "OK: stress" ));

  test_full_and_skip();
  FD_LOG_NOTICE(( "OK: full_and_skip" ));

  if( fd_tile_cnt() >= 2UL ) {
    test_concurrent();
    FD_LOG_NOTICE(( "OK: concurrent" ));
  } else {
    FD_LOG_NOTICE(( "SKIP: concurrent (need 2+ tiles)" ));
  }

  FD_LOG_NOTICE(( "pass" ));
  fd_halt();
  return 0;
}
