/* fd_netconn_ring.h - SPSC ring buffer for packet handoff.
   Drops new packets when full to ensure safe concurrent access. */

#ifndef HEADER_fd_src_disco_net_connector_fd_netconn_ring_h
#define HEADER_fd_src_disco_net_connector_fd_netconn_ring_h

#include "../../util/fd_util_base.h"
#include "../../util/bits/fd_bits.h"

#define FD_NETCONN_RING_MTU 1232UL

struct fd_netconn_pkt_hdr {
  uint   src_ip;
  ushort src_port;
  ushort payload_sz;
};

typedef struct fd_netconn_pkt_hdr fd_netconn_pkt_hdr_t;

struct  __attribute__((aligned(64))) fd_netconn_pkt {
  fd_netconn_pkt_hdr_t hdr;
  uchar payload[ FD_NETCONN_RING_MTU ];
};

typedef struct fd_netconn_pkt fd_netconn_pkt_t;

/* Cache-line separated head/tail to avoid false sharing */
struct __attribute__((aligned(64))) fd_netconn_ring {
  ulong head;
  uchar _pad0[56];
  ulong tail;
  uchar _pad1[56];
  ulong depth;
  ulong mask;
  uchar _pad2[48];
  fd_netconn_pkt_t entries[];
};

typedef struct fd_netconn_ring fd_netconn_ring_t;

/* Compile-time footprint for static array declarations.
   Usage: uchar mem[ FD_NETCONN_RING_FOOTPRINT( depth ) ] __attribute__((aligned(64))); */
#define FD_NETCONN_RING_FOOTPRINT( depth ) \
  (sizeof(fd_netconn_ring_t) + (depth) * sizeof(fd_netconn_pkt_t))

#define FD_NETCONN_RING_ALIGN 64UL

FD_PROTOTYPES_BEGIN

FD_FN_CONST static inline ulong
fd_netconn_ring_align( void ) {
  return 64UL;
}

FD_FN_CONST static inline ulong
fd_netconn_ring_footprint( ulong depth ) {
  return sizeof( fd_netconn_ring_t ) + depth * sizeof( fd_netconn_pkt_t );
}

static inline fd_netconn_ring_t *
fd_netconn_ring_new( void * mem, ulong depth ) {
  fd_netconn_ring_t * ring = (fd_netconn_ring_t *)mem;
  ring->head  = 0UL;
  ring->tail  = 0UL;
  ring->depth = depth;
  ring->mask  = depth - 1UL;
  FD_COMPILER_MFENCE();
  return ring;
}

static inline int
fd_netconn_ring_full( fd_netconn_ring_t * ring ) {
  ulong head = ring->head;
  FD_COMPILER_MFENCE();
  ulong tail = ring->tail;

  return head - tail >= ring->depth;
}

/* Producer: returns 1 on success, 0 if dropped (ring full) */
static inline int
fd_netconn_ring_push( fd_netconn_ring_t * ring,
                      uint                src_ip,
                      ushort              src_port,
                      uchar const *       payload,
                      ushort              payload_sz ) {
  ulong head = ring->head;
  FD_COMPILER_MFENCE();
  ulong tail = ring->tail;

  if( FD_UNLIKELY( head - tail >= ring->depth ) ) return 0;

  fd_netconn_pkt_t * entry = &ring->entries[ head & ring->mask ];
  entry->hdr.src_ip     = src_ip;
  entry->hdr.src_port   = src_port;
  entry->hdr.payload_sz = (ushort)fd_ushort_min( payload_sz, FD_NETCONN_RING_MTU );
  fd_memcpy( entry->payload, payload, entry->hdr.payload_sz );

  FD_COMPILER_MFENCE();
  ring->head = head + 1UL;
  return 1;
}

/* Consumer: approximate packet count */
static inline ulong
fd_netconn_ring_avail( fd_netconn_ring_t const * ring ) {
  ulong tail = ring->tail;
  FD_COMPILER_MFENCE();
  ulong head = ring->head;
  return head - tail;
}

/* Consumer: pop next packet, returns payload size or 0 if empty */
static inline ulong
fd_netconn_ring_pop( fd_netconn_ring_t * ring,
                     uint *              out_src_ip,
                     ushort *            out_src_port,
                     uchar *             out_payload ) {
  ulong tail = ring->tail;
  FD_COMPILER_MFENCE();
  ulong head = ring->head;

  if( FD_UNLIKELY( head == tail ) ) return 0UL;

  fd_netconn_pkt_t const * entry = &ring->entries[ tail & ring->mask ];
  *out_src_ip   = entry->hdr.src_ip;
  *out_src_port = entry->hdr.src_port;
  ulong payload_sz = entry->hdr.payload_sz;
  fd_memcpy( out_payload, entry->payload, payload_sz );

  FD_COMPILER_MFENCE();
  ring->tail = tail + 1UL;
  return payload_sz;
}

/* Consumer: skip next packet, returns 1 if packet was discarded */
static inline ulong
fd_netconn_ring_skip( fd_netconn_ring_t * ring ) {
  ulong tail = ring->tail;
  FD_COMPILER_MFENCE();
  ulong head = ring->head;

  if( FD_UNLIKELY( head == tail ) ) return 0UL;

  FD_COMPILER_MFENCE();
  ring->tail = tail + 1UL;
  return 1;
}

FD_PROTOTYPES_END

#endif /* HEADER_fd_src_disco_net_connector_fd_netconn_ring_h */
