#include "../tiles.h"
#include "../fd_disco_base.h"
#include "../net/fd_net_tile.h"
#include "../metrics/fd_metrics.h"
#include "../metrics/generated/fd_metrics_netcon.h"
#include "../../util/net/fd_ip4.h"
#include "../../util/net/fd_udp.h"
#include "fd_net_connector.h"

#include "generated/fd_net_connector_tile_seccomp.h"

#include <linux/unistd.h>

#define MAX_NET_INS (32UL)

typedef struct {
  ulong              in_cnt;
  fd_net_rx_bounds_t in_bounds[ MAX_NET_INS ];
  uchar              buffer[ FD_NET_MTU ];

  ulong received[ FD_NET_CONNECTOR_RING_CNT ];
  ulong dropped [ FD_NET_CONNECTOR_RING_CNT ];

  uchar ring_mem_gossip_staked    [ FD_NET_CONNECTOR_RING_FOOTPRINT_STAKED   ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
  uchar ring_mem_gossip_unstaked  [ FD_NET_CONNECTOR_RING_FOOTPRINT_UNSTAKED ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
  uchar ring_mem_repair_staked    [ FD_NET_CONNECTOR_RING_FOOTPRINT_STAKED   ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
  uchar ring_mem_repair_unstaked  [ FD_NET_CONNECTOR_RING_FOOTPRINT_UNSTAKED ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
  uchar ring_mem_ancestor_unstaked[ FD_NET_CONNECTOR_RING_FOOTPRINT_UNSTAKED ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));
  uchar ring_mem_ancestor_staked  [ FD_NET_CONNECTOR_RING_FOOTPRINT_STAKED   ] __attribute__((aligned(FD_NETCONN_RING_ALIGN)));

  fd_netconn_ring_t * rings[ FD_NET_CONNECTOR_RING_CNT ];
} fd_net_connector_ctx_t;

static fd_net_connector_ctx_t * fd_net_connector_global_ctx;

FD_FN_CONST static inline ulong
scratch_align( void ) {
  return 128UL;
}

FD_FN_PURE static inline ulong
scratch_footprint( fd_topo_tile_t const * tile ) {
  (void)tile;
  ulong l = FD_LAYOUT_INIT;
  l = FD_LAYOUT_APPEND( l, 128UL, sizeof( fd_net_connector_ctx_t ) );
  return FD_LAYOUT_FINI( l, scratch_align() );
}

static inline void
metrics_write( fd_net_connector_ctx_t * ctx ) {
  FD_MCNT_ENUM_COPY( NETCON, RECEIVED, ctx->received );
  FD_MCNT_ENUM_COPY( NETCON, DROPPED,  ctx->dropped );

  ulong depths[ FD_NET_CONNECTOR_RING_CNT ];
  for( ulong i = 0; i < FD_NET_CONNECTOR_RING_CNT; i++ ) {
    depths[ i ] = fd_netconn_ring_avail( ctx->rings[ i ] );
  }
  FD_MGAUGE_ENUM_COPY( NETCON, DEPTH, depths );
}

static int
before_frag( fd_net_connector_ctx_t * ctx,
             ulong                    in_idx,
             ulong                    seq,
             ulong                    sig ) {
  (void)ctx; (void)in_idx; (void)seq;
  ulong proto = fd_disco_netmux_sig_proto( sig );
  return ( proto < DST_PROTO_NETCONN_GOSSIP_STAKED || proto > DST_PROTO_NETCONN_ANCESTOR_STAKED );
}

static void
during_frag( fd_net_connector_ctx_t * ctx,
             ulong                    in_idx,
             ulong                    seq    FD_PARAM_UNUSED,
             ulong                    sig    FD_PARAM_UNUSED,
             ulong                    chunk,
             ulong                    sz,
             ulong                    ctl ) {
  void const * src = fd_net_rx_translate_frag( &ctx->in_bounds[ in_idx ], chunk, ctl, sz );
  fd_memcpy( ctx->buffer, src, sz );
}

static void
after_frag( fd_net_connector_ctx_t * ctx,
            ulong                    in_idx,
            ulong                    seq,
            ulong                    sig,
            ulong                    sz,
            ulong                    tsorig,
            ulong                    tspub,
            fd_stem_context_t *      stem ) {
  (void)in_idx; (void)seq; (void)tsorig; (void)tspub; (void)stem;

  ulong proto  = fd_disco_netmux_sig_proto( sig );
  ulong hdr_sz = fd_disco_netmux_sig_hdr_sz( sig );
  uint  src_ip = fd_disco_netmux_sig_ip( sig );

  if( FD_UNLIKELY( hdr_sz > sz || sz < 42UL ) ) return;

  fd_ip4_hdr_t const * ip_hdr = (fd_ip4_hdr_t const *)(ctx->buffer + 14UL);
  ulong ip_hdr_len = FD_IP4_GET_LEN( *ip_hdr );
  if( FD_UNLIKELY( 14UL + ip_hdr_len + 8UL > sz ) ) return;

  fd_udp_hdr_t const * udp_hdr = (fd_udp_hdr_t const *)(ctx->buffer + 14UL + ip_hdr_len);
  ushort src_port = udp_hdr->net_sport;

  uchar const * payload    = ctx->buffer + hdr_sz;
  ulong         payload_sz = fd_ulong_min( sz - hdr_sz, FD_NET_CONNECTOR_MTU );

  uint ring_idx = (uint)(proto - DST_PROTO_NETCONN_GOSSIP_STAKED);
  if( FD_UNLIKELY( ring_idx >= FD_NET_CONNECTOR_RING_CNT ) ) return;

  fd_netconn_ring_t * ring = ctx->rings[ ring_idx ];
  ctx->received[ ring_idx ]++;

  if( FD_UNLIKELY( !fd_netconn_ring_push( ring, src_ip, src_port, payload, (ushort)payload_sz ) ) ) {
    ctx->dropped[ ring_idx ]++;
  }
}

static void
privileged_init( fd_topo_t *      topo,
                 fd_topo_tile_t * tile ) {
  (void)topo;
  (void)tile;
}

static void
unprivileged_init( fd_topo_t *      topo,
                   fd_topo_tile_t * tile ) {
  void * scratch = fd_topo_obj_laddr( topo, tile->tile_obj_id );
  FD_SCRATCH_ALLOC_INIT( l, scratch );
  fd_net_connector_ctx_t * ctx = FD_SCRATCH_ALLOC_APPEND( l, 128UL, sizeof( fd_net_connector_ctx_t ) );
  memset( ctx, 0, sizeof( fd_net_connector_ctx_t ) );

  ctx->in_cnt = tile->in_cnt;
  if( FD_UNLIKELY( ctx->in_cnt > MAX_NET_INS ) ) FD_LOG_ERR(( "too many inputs" ));

  for( ulong i = 0UL; i < ctx->in_cnt; i++ ) {
    fd_net_rx_bounds_init( &ctx->in_bounds[ i ], topo->links[ tile->in_link_id[ i ] ].dcache );
  }

  ctx->rings[ FD_NET_CONNECTOR_RING_GOSSIP_STAKED ]     = fd_netconn_ring_new( ctx->ring_mem_gossip_staked,     FD_NET_CONNECTOR_RING_DEPTH_STAKED );
  ctx->rings[ FD_NET_CONNECTOR_RING_GOSSIP_UNSTAKED ]   = fd_netconn_ring_new( ctx->ring_mem_gossip_unstaked,   FD_NET_CONNECTOR_RING_DEPTH_UNSTAKED );
  ctx->rings[ FD_NET_CONNECTOR_RING_REPAIR_STAKED ]     = fd_netconn_ring_new( ctx->ring_mem_repair_staked,     FD_NET_CONNECTOR_RING_DEPTH_STAKED );
  ctx->rings[ FD_NET_CONNECTOR_RING_REPAIR_UNSTAKED ]   = fd_netconn_ring_new( ctx->ring_mem_repair_unstaked,   FD_NET_CONNECTOR_RING_DEPTH_UNSTAKED );
  ctx->rings[ FD_NET_CONNECTOR_RING_ANCESTOR_UNSTAKED ] = fd_netconn_ring_new( ctx->ring_mem_ancestor_unstaked, FD_NET_CONNECTOR_RING_DEPTH_UNSTAKED );
  ctx->rings[ FD_NET_CONNECTOR_RING_ANCESTOR_STAKED ]   = fd_netconn_ring_new( ctx->ring_mem_ancestor_staked,   FD_NET_CONNECTOR_RING_DEPTH_STAKED );

  FD_COMPILER_MFENCE();
  FD_VOLATILE( fd_net_connector_global_ctx ) = ctx;

  ulong scratch_top = FD_SCRATCH_ALLOC_FINI( l, 1UL );
  if( FD_UNLIKELY( scratch_top > (ulong)scratch + scratch_footprint( tile ) ) ) FD_LOG_ERR(( "scratch overflow" ));
}


/* FFI for Agave - staked ring has priority */

void
fd_ext_netconn_wait_ready( void ) {
  FD_COMPILER_MFENCE();
  while( !FD_VOLATILE_CONST( fd_net_connector_global_ctx ) ) FD_SPIN_PAUSE();
}

ulong
fd_ext_netconn_poll_gossip( uchar * out_payload, uint * out_src_ip, ushort * out_src_port ) {
  fd_net_connector_ctx_t * ctx = FD_VOLATILE_CONST( fd_net_connector_global_ctx );
  if( FD_UNLIKELY( !ctx ) ) return 0UL;
  ulong sz = fd_netconn_ring_pop( ctx->rings[ FD_NET_CONNECTOR_RING_GOSSIP_STAKED ], out_src_ip, out_src_port, out_payload );
  if (FD_UNLIKELY( sz )) {
    /* gossip unstaked full, skip packet */
    if (FD_UNLIKELY( fd_netconn_ring_full(ctx->rings[ FD_NET_CONNECTOR_RING_GOSSIP_UNSTAKED ]) )) {
      fd_netconn_ring_skip( ctx->rings[ FD_NET_CONNECTOR_RING_GOSSIP_UNSTAKED ] );
    }
    return sz;
  }

  return fd_netconn_ring_pop( ctx->rings[ FD_NET_CONNECTOR_RING_GOSSIP_UNSTAKED ], out_src_ip, out_src_port, out_payload );
}

ulong
fd_ext_netconn_poll_repair( uchar * out_payload, uint * out_src_ip, ushort * out_src_port ) {
  fd_net_connector_ctx_t * ctx = FD_VOLATILE_CONST( fd_net_connector_global_ctx );
  if( FD_UNLIKELY( !ctx ) ) return 0UL;
  ulong sz = fd_netconn_ring_pop( ctx->rings[ FD_NET_CONNECTOR_RING_REPAIR_STAKED ], out_src_ip, out_src_port, out_payload );
  return sz ? sz : fd_netconn_ring_pop( ctx->rings[ FD_NET_CONNECTOR_RING_REPAIR_UNSTAKED ], out_src_ip, out_src_port, out_payload );
}

ulong
fd_ext_netconn_poll_ancestor( uchar * out_payload, uint * out_src_ip, ushort * out_src_port ) {
  fd_net_connector_ctx_t * ctx = FD_VOLATILE_CONST( fd_net_connector_global_ctx );
  if( FD_UNLIKELY( !ctx ) ) return 0UL;
  ulong sz = fd_netconn_ring_pop( ctx->rings[ FD_NET_CONNECTOR_RING_ANCESTOR_STAKED ], out_src_ip, out_src_port, out_payload );
  return sz ? sz : fd_netconn_ring_pop( ctx->rings[ FD_NET_CONNECTOR_RING_ANCESTOR_UNSTAKED ], out_src_ip, out_src_port, out_payload );
}

ulong fd_ext_netconn_gossip_avail( void ) {
  fd_net_connector_ctx_t * ctx = FD_VOLATILE_CONST( fd_net_connector_global_ctx );
  if( FD_UNLIKELY( !ctx ) ) return 0UL;
  return fd_netconn_ring_avail( ctx->rings[ FD_NET_CONNECTOR_RING_GOSSIP_STAKED ] ) +
         fd_netconn_ring_avail( ctx->rings[ FD_NET_CONNECTOR_RING_GOSSIP_UNSTAKED ] );
}

ulong fd_ext_netconn_repair_avail( void ) {
  fd_net_connector_ctx_t * ctx = FD_VOLATILE_CONST( fd_net_connector_global_ctx );
  if( FD_UNLIKELY( !ctx ) ) return 0UL;
  return fd_netconn_ring_avail( ctx->rings[ FD_NET_CONNECTOR_RING_REPAIR_STAKED ] ) +
         fd_netconn_ring_avail( ctx->rings[ FD_NET_CONNECTOR_RING_REPAIR_UNSTAKED ] );
}

ulong fd_ext_netconn_ancestor_avail( void ) {
  fd_net_connector_ctx_t * ctx = FD_VOLATILE_CONST( fd_net_connector_global_ctx );
  if( FD_UNLIKELY( !ctx ) ) return 0UL;
  return fd_netconn_ring_avail( ctx->rings[ FD_NET_CONNECTOR_RING_ANCESTOR_STAKED ] ) +
         fd_netconn_ring_avail( ctx->rings[ FD_NET_CONNECTOR_RING_ANCESTOR_UNSTAKED ] );
}


static ulong
populate_allowed_seccomp( fd_topo_t const *      topo,
                          fd_topo_tile_t const * tile,
                          ulong                  out_cnt,
                          struct sock_filter *   out ) {
  (void)topo;
  (void)tile;

  populate_sock_filter_policy_fd_net_connector_tile( out_cnt, out, (uint)fd_log_private_logfile_fd() );
  return sock_filter_policy_fd_net_connector_tile_instr_cnt;
}

static ulong
populate_allowed_fds( fd_topo_t const *      topo,
                      fd_topo_tile_t const * tile,
                      ulong                  out_fds_cnt,
                      int *                  out_fds ) {
  (void)topo;
  (void)tile;

  if( FD_UNLIKELY( out_fds_cnt < 2UL ) ) FD_LOG_ERR(( "out_fds_cnt %lu", out_fds_cnt ));

  ulong out_cnt = 0UL;
  out_fds[ out_cnt++ ] = 2; /* stderr */
  if( FD_LIKELY( -1 != fd_log_private_logfile_fd() ) )
    out_fds[ out_cnt++ ] = fd_log_private_logfile_fd(); /* logfile */
  return out_cnt;
}

#define STEM_BURST (1UL)

#define STEM_CALLBACK_CONTEXT_TYPE  fd_net_connector_ctx_t
#define STEM_CALLBACK_CONTEXT_ALIGN 128UL

#define STEM_CALLBACK_METRICS_WRITE metrics_write
#define STEM_CALLBACK_BEFORE_FRAG   before_frag
#define STEM_CALLBACK_DURING_FRAG   during_frag
#define STEM_CALLBACK_AFTER_FRAG    after_frag

#include "../stem/fd_stem.c"

fd_topo_run_tile_t fd_tile_net_connector = {
  .name                     = "netcon",
  .populate_allowed_seccomp = populate_allowed_seccomp,
  .populate_allowed_fds     = populate_allowed_fds,
  .scratch_align            = scratch_align,
  .scratch_footprint        = scratch_footprint,
  .privileged_init          = privileged_init,
  .unprivileged_init        = unprivileged_init,
  .run                      = stem_run,
};
