use super::SystemInstruction;
use core::{fmt, marker::PhantomData};
use serde::{
    de::{
        self,
        value::{MapAccessDeserializer, SeqAccessDeserializer},
        Deserialize, EnumAccess, MapAccess, SeqAccess, Unexpected, VariantAccess, Visitor,
    },
    ser::{Serialize, SerializeStructVariant},
    Deserializer, Serializer,
};
use serde_derive::Deserialize;
use solana_pubkey::Pubkey;

const SESSION_WRAP_DISCRIMINATOR: u32 = 4_000_000;
const INTENT_TRANSFER_DISCRIMINATOR: u32 = 4_000_001;

const SESSION_WRAP_DISCRIMINATOR_U64: u64 = SESSION_WRAP_DISCRIMINATOR as u64;
const INTENT_TRANSFER_DISCRIMINATOR_U64: u64 = INTENT_TRANSFER_DISCRIMINATOR as u64;

const VARIANTS: &[&str] = &[
    "CreateAccount",
    "Assign",
    "Transfer",
    "CreateAccountWithSeed",
    "AdvanceNonceAccount",
    "WithdrawNonceAccount",
    "InitializeNonceAccount",
    "AuthorizeNonceAccount",
    "Allocate",
    "AllocateWithSeed",
    "AssignWithSeed",
    "TransferWithSeed",
    "UpgradeNonceAccount",
    "SessionWrap",
    "IntentTransfer",
];

#[derive(Debug)]
enum Field {
    CreateAccount,
    Assign,
    Transfer,
    CreateAccountWithSeed,
    AdvanceNonceAccount,
    WithdrawNonceAccount,
    InitializeNonceAccount,
    AuthorizeNonceAccount,
    Allocate,
    AllocateWithSeed,
    AssignWithSeed,
    TransferWithSeed,
    UpgradeNonceAccount,
    SessionWrap,
    IntentTransfer,
}

impl<'de> Deserialize<'de> for Field {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        struct FieldVisitor;

        impl<'de> Visitor<'de> for FieldVisitor {
            type Value = Field;

            fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
                formatter.write_str("variant identifier")
            }

            fn visit_u64<E>(self, value: u64) -> Result<Self::Value, E>
            where
                E: de::Error,
            {
                match value {
                    0 => Ok(Field::CreateAccount),
                    1 => Ok(Field::Assign),
                    2 => Ok(Field::Transfer),
                    3 => Ok(Field::CreateAccountWithSeed),
                    4 => Ok(Field::AdvanceNonceAccount),
                    5 => Ok(Field::WithdrawNonceAccount),
                    6 => Ok(Field::InitializeNonceAccount),
                    7 => Ok(Field::AuthorizeNonceAccount),
                    8 => Ok(Field::Allocate),
                    9 => Ok(Field::AllocateWithSeed),
                    10 => Ok(Field::AssignWithSeed),
                    11 => Ok(Field::TransferWithSeed),
                    12 => Ok(Field::UpgradeNonceAccount),
                    SESSION_WRAP_DISCRIMINATOR_U64 => Ok(Field::SessionWrap),
                    INTENT_TRANSFER_DISCRIMINATOR_U64 => Ok(Field::IntentTransfer),
                    _ => Err(de::Error::invalid_value(
                        Unexpected::Unsigned(value),
                        &"variant index 0 <= i < 13 or 4_000_000 <= i < 4_000_002",
                    )),
                }
            }

            fn visit_str<E>(self, value: &str) -> Result<Self::Value, E>
            where
                E: de::Error,
            {
                match value {
                    "CreateAccount" => Ok(Field::CreateAccount),
                    "Assign" => Ok(Field::Assign),
                    "Transfer" => Ok(Field::Transfer),
                    "CreateAccountWithSeed" => Ok(Field::CreateAccountWithSeed),
                    "AdvanceNonceAccount" => Ok(Field::AdvanceNonceAccount),
                    "WithdrawNonceAccount" => Ok(Field::WithdrawNonceAccount),
                    "InitializeNonceAccount" => Ok(Field::InitializeNonceAccount),
                    "AuthorizeNonceAccount" => Ok(Field::AuthorizeNonceAccount),
                    "Allocate" => Ok(Field::Allocate),
                    "AllocateWithSeed" => Ok(Field::AllocateWithSeed),
                    "AssignWithSeed" => Ok(Field::AssignWithSeed),
                    "TransferWithSeed" => Ok(Field::TransferWithSeed),
                    "UpgradeNonceAccount" => Ok(Field::UpgradeNonceAccount),
                    "SessionWrap" => Ok(Field::SessionWrap),
                    "IntentTransfer" => Ok(Field::IntentTransfer),
                    _ => Err(de::Error::unknown_variant(value, VARIANTS)),
                }
            }

            fn visit_bytes<E>(self, value: &[u8]) -> Result<Self::Value, E>
            where
                E: de::Error,
            {
                match value {
                    b"CreateAccount" => Ok(Field::CreateAccount),
                    b"Assign" => Ok(Field::Assign),
                    b"Transfer" => Ok(Field::Transfer),
                    b"CreateAccountWithSeed" => Ok(Field::CreateAccountWithSeed),
                    b"AdvanceNonceAccount" => Ok(Field::AdvanceNonceAccount),
                    b"WithdrawNonceAccount" => Ok(Field::WithdrawNonceAccount),
                    b"InitializeNonceAccount" => Ok(Field::InitializeNonceAccount),
                    b"AuthorizeNonceAccount" => Ok(Field::AuthorizeNonceAccount),
                    b"Allocate" => Ok(Field::Allocate),
                    b"AllocateWithSeed" => Ok(Field::AllocateWithSeed),
                    b"AssignWithSeed" => Ok(Field::AssignWithSeed),
                    b"TransferWithSeed" => Ok(Field::TransferWithSeed),
                    b"UpgradeNonceAccount" => Ok(Field::UpgradeNonceAccount),
                    b"SessionWrap" => Ok(Field::SessionWrap),
                    b"IntentTransfer" => Ok(Field::IntentTransfer),
                    _ => {
                        let value = &String::from_utf8_lossy(value);
                        Err(de::Error::unknown_variant(value, VARIANTS))
                    }
                }
            }
        }

        deserializer.deserialize_identifier(FieldVisitor)
    }
}

struct StructVariantVisitor<T> {
    name: &'static str,
    marker: PhantomData<T>,
}

impl<'de, T> Visitor<'de> for StructVariantVisitor<T>
where
    T: Deserialize<'de>,
{
    type Value = T;

    fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        formatter.write_str(self.name)
    }

    fn visit_seq<A>(self, seq: A) -> Result<Self::Value, A::Error>
    where
        A: SeqAccess<'de>,
    {
        Deserialize::deserialize(SeqAccessDeserializer::new(seq))
    }

    fn visit_map<A>(self, map: A) -> Result<Self::Value, A::Error>
    where
        A: MapAccess<'de>,
    {
        Deserialize::deserialize(MapAccessDeserializer::new(map))
    }
}

fn deserialize_struct_variant<'de, A, T>(
    access: A,
    fields: &'static [&'static str],
    name: &'static str,
) -> Result<T, A::Error>
where
    A: VariantAccess<'de>,
    T: Deserialize<'de>,
{
    access.struct_variant(
        fields,
        StructVariantVisitor {
            name,
            marker: PhantomData,
        },
    )
}

#[derive(Deserialize)]
struct CreateAccountFields {
    lamports: u64,
    space: u64,
    owner: Pubkey,
}

#[derive(Deserialize)]
struct AssignFields {
    owner: Pubkey,
}

#[derive(Deserialize)]
struct TransferFields {
    lamports: u64,
}

#[derive(Deserialize)]
struct CreateAccountWithSeedFields {
    base: Pubkey,
    seed: String,
    lamports: u64,
    space: u64,
    owner: Pubkey,
}

#[derive(Deserialize)]
struct AllocateFields {
    space: u64,
}

#[derive(Deserialize)]
struct AllocateWithSeedFields {
    base: Pubkey,
    seed: String,
    space: u64,
    owner: Pubkey,
}

#[derive(Deserialize)]
struct AssignWithSeedFields {
    base: Pubkey,
    seed: String,
    owner: Pubkey,
}

#[derive(Deserialize)]
struct TransferWithSeedFields {
    lamports: u64,
    from_seed: String,
    from_owner: Pubkey,
}

#[derive(Deserialize)]
struct SessionWrapFields {
    lamports: u64,
}

#[derive(Deserialize)]
struct IntentTransferFields {
    lamports: u64,
}

struct SystemInstructionVisitor;

impl<'de> Visitor<'de> for SystemInstructionVisitor {
    type Value = SystemInstruction;

    fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        formatter.write_str("enum SystemInstruction")
    }

    fn visit_enum<A>(self, data: A) -> Result<Self::Value, A::Error>
    where
        A: EnumAccess<'de>,
    {
        match data.variant()? {
            (Field::CreateAccount, variant) => {
                let fields: CreateAccountFields = deserialize_struct_variant(
                    variant,
                    &["lamports", "space", "owner"],
                    "struct variant SystemInstruction::CreateAccount",
                )?;
                Ok(SystemInstruction::CreateAccount {
                    lamports: fields.lamports,
                    space: fields.space,
                    owner: fields.owner,
                })
            }
            (Field::Assign, variant) => {
                let fields: AssignFields = deserialize_struct_variant(
                    variant,
                    &["owner"],
                    "struct variant SystemInstruction::Assign",
                )?;
                Ok(SystemInstruction::Assign {
                    owner: fields.owner,
                })
            }
            (Field::Transfer, variant) => {
                let fields: TransferFields = deserialize_struct_variant(
                    variant,
                    &["lamports"],
                    "struct variant SystemInstruction::Transfer",
                )?;
                Ok(SystemInstruction::Transfer {
                    lamports: fields.lamports,
                })
            }
            (Field::CreateAccountWithSeed, variant) => {
                let fields: CreateAccountWithSeedFields = deserialize_struct_variant(
                    variant,
                    &["base", "seed", "lamports", "space", "owner"],
                    "struct variant SystemInstruction::CreateAccountWithSeed",
                )?;
                Ok(SystemInstruction::CreateAccountWithSeed {
                    base: fields.base,
                    seed: fields.seed,
                    lamports: fields.lamports,
                    space: fields.space,
                    owner: fields.owner,
                })
            }
            (Field::AdvanceNonceAccount, variant) => {
                variant.unit_variant()?;
                Ok(SystemInstruction::AdvanceNonceAccount)
            }
            (Field::WithdrawNonceAccount, variant) => Ok(SystemInstruction::WithdrawNonceAccount(
                variant.newtype_variant()?,
            )),
            (Field::InitializeNonceAccount, variant) => Ok(
                SystemInstruction::InitializeNonceAccount(variant.newtype_variant()?),
            ),
            (Field::AuthorizeNonceAccount, variant) => Ok(
                SystemInstruction::AuthorizeNonceAccount(variant.newtype_variant()?),
            ),
            (Field::Allocate, variant) => {
                let fields: AllocateFields = deserialize_struct_variant(
                    variant,
                    &["space"],
                    "struct variant SystemInstruction::Allocate",
                )?;
                Ok(SystemInstruction::Allocate {
                    space: fields.space,
                })
            }
            (Field::AllocateWithSeed, variant) => {
                let fields: AllocateWithSeedFields = deserialize_struct_variant(
                    variant,
                    &["base", "seed", "space", "owner"],
                    "struct variant SystemInstruction::AllocateWithSeed",
                )?;
                Ok(SystemInstruction::AllocateWithSeed {
                    base: fields.base,
                    seed: fields.seed,
                    space: fields.space,
                    owner: fields.owner,
                })
            }
            (Field::AssignWithSeed, variant) => {
                let fields: AssignWithSeedFields = deserialize_struct_variant(
                    variant,
                    &["base", "seed", "owner"],
                    "struct variant SystemInstruction::AssignWithSeed",
                )?;
                Ok(SystemInstruction::AssignWithSeed {
                    base: fields.base,
                    seed: fields.seed,
                    owner: fields.owner,
                })
            }
            (Field::TransferWithSeed, variant) => {
                let fields: TransferWithSeedFields = deserialize_struct_variant(
                    variant,
                    &["lamports", "from_seed", "from_owner"],
                    "struct variant SystemInstruction::TransferWithSeed",
                )?;
                Ok(SystemInstruction::TransferWithSeed {
                    lamports: fields.lamports,
                    from_seed: fields.from_seed,
                    from_owner: fields.from_owner,
                })
            }
            (Field::UpgradeNonceAccount, variant) => {
                variant.unit_variant()?;
                Ok(SystemInstruction::UpgradeNonceAccount)
            }
            (Field::SessionWrap, variant) => {
                let fields: SessionWrapFields = deserialize_struct_variant(
                    variant,
                    &["lamports"],
                    "struct variant SystemInstruction::SessionWrap",
                )?;
                Ok(SystemInstruction::SessionWrap {
                    lamports: fields.lamports,
                })
            }
            (Field::IntentTransfer, variant) => {
                let fields: IntentTransferFields = deserialize_struct_variant(
                    variant,
                    &["lamports"],
                    "struct variant SystemInstruction::IntentTransfer",
                )?;
                Ok(SystemInstruction::IntentTransfer {
                    lamports: fields.lamports,
                })
            }
        }
    }
}

impl<'de> Deserialize<'de> for SystemInstruction {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        deserializer.deserialize_enum("SystemInstruction", VARIANTS, SystemInstructionVisitor)
    }
}

impl Serialize for SystemInstruction {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        match *self {
            SystemInstruction::CreateAccount {
                ref lamports,
                ref space,
                ref owner,
            } => {
                let mut state = serializer.serialize_struct_variant(
                    "SystemInstruction",
                    0,
                    "CreateAccount",
                    3,
                )?;
                SerializeStructVariant::serialize_field(&mut state, "lamports", lamports)?;
                SerializeStructVariant::serialize_field(&mut state, "space", space)?;
                SerializeStructVariant::serialize_field(&mut state, "owner", owner)?;
                SerializeStructVariant::end(state)
            }
            SystemInstruction::Assign { ref owner } => {
                let mut state =
                    serializer.serialize_struct_variant("SystemInstruction", 1, "Assign", 1)?;
                SerializeStructVariant::serialize_field(&mut state, "owner", owner)?;
                SerializeStructVariant::end(state)
            }
            SystemInstruction::Transfer { ref lamports } => {
                let mut state =
                    serializer.serialize_struct_variant("SystemInstruction", 2, "Transfer", 1)?;
                SerializeStructVariant::serialize_field(&mut state, "lamports", lamports)?;
                SerializeStructVariant::end(state)
            }
            SystemInstruction::CreateAccountWithSeed {
                ref base,
                ref seed,
                ref lamports,
                ref space,
                ref owner,
            } => {
                let mut state = serializer.serialize_struct_variant(
                    "SystemInstruction",
                    3,
                    "CreateAccountWithSeed",
                    5,
                )?;
                SerializeStructVariant::serialize_field(&mut state, "base", base)?;
                SerializeStructVariant::serialize_field(&mut state, "seed", seed)?;
                SerializeStructVariant::serialize_field(&mut state, "lamports", lamports)?;
                SerializeStructVariant::serialize_field(&mut state, "space", space)?;
                SerializeStructVariant::serialize_field(&mut state, "owner", owner)?;
                SerializeStructVariant::end(state)
            }
            SystemInstruction::AdvanceNonceAccount => {
                serializer.serialize_unit_variant("SystemInstruction", 4, "AdvanceNonceAccount")
            }
            SystemInstruction::WithdrawNonceAccount(ref amount) => serializer
                .serialize_newtype_variant("SystemInstruction", 5, "WithdrawNonceAccount", amount),
            SystemInstruction::InitializeNonceAccount(ref pubkey) => serializer
                .serialize_newtype_variant(
                    "SystemInstruction",
                    6,
                    "InitializeNonceAccount",
                    pubkey,
                ),
            SystemInstruction::AuthorizeNonceAccount(ref pubkey) => serializer
                .serialize_newtype_variant("SystemInstruction", 7, "AuthorizeNonceAccount", pubkey),
            SystemInstruction::Allocate { ref space } => {
                let mut state =
                    serializer.serialize_struct_variant("SystemInstruction", 8, "Allocate", 1)?;
                SerializeStructVariant::serialize_field(&mut state, "space", space)?;
                SerializeStructVariant::end(state)
            }
            SystemInstruction::AllocateWithSeed {
                ref base,
                ref seed,
                ref space,
                ref owner,
            } => {
                let mut state = serializer.serialize_struct_variant(
                    "SystemInstruction",
                    9,
                    "AllocateWithSeed",
                    4,
                )?;
                SerializeStructVariant::serialize_field(&mut state, "base", base)?;
                SerializeStructVariant::serialize_field(&mut state, "seed", seed)?;
                SerializeStructVariant::serialize_field(&mut state, "space", space)?;
                SerializeStructVariant::serialize_field(&mut state, "owner", owner)?;
                SerializeStructVariant::end(state)
            }
            SystemInstruction::AssignWithSeed {
                ref base,
                ref seed,
                ref owner,
            } => {
                let mut state = serializer.serialize_struct_variant(
                    "SystemInstruction",
                    10,
                    "AssignWithSeed",
                    3,
                )?;
                SerializeStructVariant::serialize_field(&mut state, "base", base)?;
                SerializeStructVariant::serialize_field(&mut state, "seed", seed)?;
                SerializeStructVariant::serialize_field(&mut state, "owner", owner)?;
                SerializeStructVariant::end(state)
            }
            SystemInstruction::TransferWithSeed {
                ref lamports,
                ref from_seed,
                ref from_owner,
            } => {
                let mut state = serializer.serialize_struct_variant(
                    "SystemInstruction",
                    11,
                    "TransferWithSeed",
                    3,
                )?;
                SerializeStructVariant::serialize_field(&mut state, "lamports", lamports)?;
                SerializeStructVariant::serialize_field(&mut state, "from_seed", from_seed)?;
                SerializeStructVariant::serialize_field(&mut state, "from_owner", from_owner)?;
                SerializeStructVariant::end(state)
            }
            SystemInstruction::UpgradeNonceAccount => {
                serializer.serialize_unit_variant("SystemInstruction", 12, "UpgradeNonceAccount")
            }
            SystemInstruction::SessionWrap { ref lamports } => {
                let mut state = serializer.serialize_struct_variant(
                    "SystemInstruction",
                    SESSION_WRAP_DISCRIMINATOR,
                    "SessionWrap",
                    1,
                )?;
                SerializeStructVariant::serialize_field(&mut state, "lamports", lamports)?;
                SerializeStructVariant::end(state)
            }
            SystemInstruction::IntentTransfer { ref lamports } => {
                let mut state = serializer.serialize_struct_variant(
                    "SystemInstruction",
                    INTENT_TRANSFER_DISCRIMINATOR,
                    "IntentTransfer",
                    1,
                )?;
                SerializeStructVariant::serialize_field(&mut state, "lamports", lamports)?;
                SerializeStructVariant::end(state)
            }
        }
    }
}

#[cfg(feature = "bincode")]
#[cfg(test)]
mod tests {
    use super::*;
    use solana_system_interface_base::instruction::SystemInstruction as SystemInstructionBase;

    impl TryInto<SystemInstructionBase> for SystemInstruction {
        type Error = ();

        fn try_into(self) -> Result<SystemInstructionBase, Self::Error> {
            match self {
                SystemInstruction::CreateAccount {
                    lamports,
                    space,
                    owner,
                } => Ok(SystemInstructionBase::CreateAccount {
                    lamports,
                    space,
                    owner,
                }),
                SystemInstruction::Assign { owner } => Ok(SystemInstructionBase::Assign { owner }),
                SystemInstruction::Transfer { lamports } => {
                    Ok(SystemInstructionBase::Transfer { lamports })
                }
                SystemInstruction::CreateAccountWithSeed {
                    base,
                    seed,
                    lamports,
                    space,
                    owner,
                } => Ok(SystemInstructionBase::CreateAccountWithSeed {
                    base,
                    seed,
                    lamports,
                    space,
                    owner,
                }),
                SystemInstruction::AdvanceNonceAccount => {
                    Ok(SystemInstructionBase::AdvanceNonceAccount)
                }
                SystemInstruction::WithdrawNonceAccount(amount) => {
                    Ok(SystemInstructionBase::WithdrawNonceAccount(amount))
                }
                SystemInstruction::InitializeNonceAccount(pubkey) => {
                    Ok(SystemInstructionBase::InitializeNonceAccount(pubkey))
                }
                SystemInstruction::AuthorizeNonceAccount(pubkey) => {
                    Ok(SystemInstructionBase::AuthorizeNonceAccount(pubkey))
                }
                SystemInstruction::Allocate { space } => {
                    Ok(SystemInstructionBase::Allocate { space })
                }
                SystemInstruction::AllocateWithSeed {
                    base,
                    seed,
                    space,
                    owner,
                } => Ok(SystemInstructionBase::AllocateWithSeed {
                    base,
                    seed,
                    space,
                    owner,
                }),
                SystemInstruction::AssignWithSeed { base, seed, owner } => {
                    Ok(SystemInstructionBase::AssignWithSeed { base, seed, owner })
                }
                SystemInstruction::TransferWithSeed {
                    lamports,
                    from_seed,
                    from_owner,
                } => Ok(SystemInstructionBase::TransferWithSeed {
                    lamports,
                    from_seed,
                    from_owner,
                }),
                SystemInstruction::UpgradeNonceAccount => {
                    Ok(SystemInstructionBase::UpgradeNonceAccount)
                }
                SystemInstruction::SessionWrap { .. }
                | SystemInstruction::IntentTransfer { .. } => Err(()),
            }
        }
    }

    macro_rules! test_variant {
        ($name:ident, $instruction:expr) => {
            mod $name {
                use super::*;
                test_roundtrip!($name, $instruction);
                test_backward_compatible!($name, $instruction);
            }
        };
    }

    macro_rules! test_roundtrip {
        ($name:ident, $instruction:expr) => {
            #[test]
            fn test_roundtrip() {
                let instruction = $instruction;
                let bytes = bincode::serialize(&instruction).unwrap();
                let deserialized =
                    bincode::deserialize_from::<_, SystemInstruction>(&bytes[..]).unwrap();
                assert_eq!(deserialized, instruction);
            }
        };
    }

    macro_rules! test_backward_compatible {
        ($name:ident, $instruction:expr) => {
            #[test]
            fn test_backward_compatible() {
                let instruction = $instruction;
                let bytes = bincode::serialize::<SystemInstructionBase>(
                    &instruction.clone().try_into().unwrap(),
                )
                .unwrap();
                let deserialized =
                    bincode::deserialize_from::<_, SystemInstruction>(&bytes[..]).unwrap();
                assert_eq!(deserialized, instruction);
            }
        };
    }

    test_variant!(
        create_account,
        SystemInstruction::CreateAccount {
            lamports: 1,
            space: 2,
            owner: Pubkey::new_unique()
        }
    );

    test_variant!(
        assign,
        SystemInstruction::Assign {
            owner: Pubkey::new_unique()
        }
    );

    test_variant!(transfer, SystemInstruction::Transfer { lamports: 42 });

    test_variant!(
        create_account_with_seed,
        SystemInstruction::CreateAccountWithSeed {
            base: Pubkey::new_unique(),
            seed: "test_seed".to_string(),
            lamports: 100,
            space: 200,
            owner: Pubkey::new_unique()
        }
    );

    test_variant!(
        advance_nonce_account,
        SystemInstruction::AdvanceNonceAccount
    );

    test_variant!(
        withdraw_nonce_account,
        SystemInstruction::WithdrawNonceAccount(500)
    );

    test_variant!(
        initialize_nonce_account,
        SystemInstruction::InitializeNonceAccount(Pubkey::new_unique())
    );

    test_variant!(
        authorize_nonce_account,
        SystemInstruction::AuthorizeNonceAccount(Pubkey::new_unique())
    );

    test_variant!(allocate, SystemInstruction::Allocate { space: 1024 });

    test_variant!(
        allocate_with_seed,
        SystemInstruction::AllocateWithSeed {
            base: Pubkey::new_unique(),
            seed: "alloc_seed".to_string(),
            space: 512,
            owner: Pubkey::new_unique()
        }
    );

    test_variant!(
        assign_with_seed,
        SystemInstruction::AssignWithSeed {
            base: Pubkey::new_unique(),
            seed: "assign_seed".to_string(),
            owner: Pubkey::new_unique()
        }
    );

    test_variant!(
        transfer_with_seed,
        SystemInstruction::TransferWithSeed {
            lamports: 999,
            from_seed: "from_seed".to_string(),
            from_owner: Pubkey::new_unique()
        }
    );

    test_variant!(
        upgrade_nonce_account,
        SystemInstruction::UpgradeNonceAccount
    );

    mod session_wrap {
        use super::*;
        test_roundtrip!(
            session_wrap,
            SystemInstruction::SessionWrap { lamports: 777 }
        );
    }

    mod intent_transfer {
        use super::*;
        test_roundtrip!(
            intent_transfer,
            SystemInstruction::IntentTransfer { lamports: 888 }
        );
    }
}
