#![feature(test)]

extern crate test;
use {solana_sdk::big_mod_exp::big_mod_exp, test::Bencher};

#[bench]
fn bench_big_num_exponentiation_exp_3_mod_0512_bits_odd(b: &mut Bencher) {
    let exponent = [3];

    // a random base
    let base = [
        243, 125, 228, 56, 107, 91, 133, 57, 46, 184, 164, 236, 176, 173, 36, 149, 58, 238, 150,
        32, 181, 248, 42, 134, 92, 170, 70, 16, 109, 212, 16, 28, 195, 174, 187, 226, 140, 22, 3,
        31, 96, 234, 110, 254, 106, 215, 101, 164, 190, 88, 14, 112, 151, 78, 205, 151, 254, 225,
        153, 125, 109, 4, 68, 87,
    ];

    // a random modulus
    let modulus = [
        249, 242, 76, 142, 109, 239, 3, 168, 130, 45, 156, 105, 209, 72, 218, 93, 86, 112, 88, 215,
        43, 194, 59, 35, 44, 86, 2, 252, 132, 113, 24, 4, 109, 98, 68, 209, 53, 191, 213, 162, 221,
        114, 213, 66, 58, 254, 152, 79, 82, 222, 79, 76, 1, 68, 255, 3, 218, 218, 83, 98, 85, 108,
        65, 85,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_3_mod_1024_bits_odd(b: &mut Bencher) {
    let exponent = [3];

    // a random base
    let base = [
        212, 177, 76, 117, 205, 210, 60, 63, 64, 203, 108, 135, 17, 114, 163, 95, 224, 106, 191,
        205, 112, 55, 118, 224, 234, 35, 97, 95, 254, 115, 29, 29, 155, 136, 100, 24, 208, 77, 141,
        136, 62, 201, 198, 76, 142, 13, 77, 111, 231, 196, 136, 198, 18, 7, 42, 155, 127, 177, 176,
        250, 30, 83, 243, 231, 197, 161, 33, 122, 254, 152, 1, 214, 70, 166, 155, 142, 77, 132,
        159, 158, 102, 62, 53, 226, 255, 64, 220, 68, 201, 221, 248, 237, 88, 135, 64, 229, 111,
        183, 2, 52, 48, 1, 45, 146, 26, 132, 64, 31, 77, 137, 131, 245, 191, 166, 68, 1, 216, 211,
        177, 196, 122, 75, 212, 7, 183, 80, 240, 66,
    ];
    // a random modulus
    let modulus = [
        237, 223, 117, 18, 201, 74, 224, 241, 183, 182, 90, 141, 137, 224, 33, 142, 133, 75, 192,
        99, 45, 75, 185, 134, 226, 65, 105, 202, 253, 125, 129, 38, 135, 120, 49, 248, 112, 4, 211,
        189, 0, 156, 21, 44, 227, 62, 38, 6, 32, 45, 254, 31, 108, 151, 172, 166, 18, 46, 3, 141,
        22, 176, 57, 160, 40, 104, 39, 68, 167, 233, 192, 157, 33, 200, 231, 220, 195, 161, 137,
        235, 28, 4, 117, 223, 173, 4, 38, 143, 50, 16, 254, 176, 146, 13, 195, 210, 247, 134, 71,
        226, 101, 7, 52, 150, 227, 221, 149, 152, 52, 84, 142, 243, 197, 230, 134, 182, 126, 183,
        122, 82, 62, 74, 173, 42, 233, 38, 13, 241,
    ];
    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_3_mod_2048_bits_odd(b: &mut Bencher) {
    let exponent = [3];

    // a random base
    let base = [
        236, 66, 86, 50, 109, 186, 105, 32, 149, 81, 7, 31, 151, 116, 93, 59, 84, 247, 239, 100,
        174, 245, 37, 174, 204, 240, 52, 226, 252, 105, 16, 88, 180, 24, 90, 223, 60, 226, 147,
        165, 255, 74, 249, 183, 1, 80, 160, 48, 111, 90, 220, 16, 237, 217, 251, 30, 208, 24, 127,
        170, 237, 244, 15, 89, 205, 254, 128, 49, 245, 233, 139, 239, 4, 126, 248, 212, 173, 103,
        222, 225, 118, 93, 219, 91, 26, 90, 142, 91, 210, 149, 227, 222, 255, 227, 182, 233, 176,
        83, 210, 143, 174, 92, 46, 19, 242, 117, 6, 125, 163, 93, 116, 63, 71, 236, 139, 17, 192,
        239, 106, 133, 145, 158, 46, 238, 107, 80, 54, 80, 231, 138, 236, 44, 55, 13, 193, 159,
        144, 85, 138, 204, 84, 126, 66, 40, 104, 232, 113, 216, 165, 184, 198, 20, 234, 225, 170,
        174, 90, 101, 253, 231, 80, 252, 28, 148, 89, 64, 65, 26, 143, 60, 158, 116, 12, 14, 46,
        210, 99, 233, 187, 212, 44, 36, 47, 227, 123, 195, 45, 115, 12, 123, 16, 164, 92, 52, 229,
        65, 127, 114, 213, 116, 210, 2, 149, 144, 217, 131, 146, 67, 66, 91, 199, 46, 58, 5, 185,
        247, 73, 170, 6, 45, 109, 0, 191, 233, 95, 239, 241, 30, 61, 119, 54, 4, 164, 214, 202,
        251, 139, 28, 22, 219, 100, 233, 195, 151, 237, 183, 41, 153, 42, 82, 208, 222, 21, 160,
        100, 100,
    ];

    // a random modulus
    let modulus = [
        155, 66, 179, 54, 45, 180, 207, 15, 110, 66, 217, 170, 218, 229, 14, 147, 163, 227, 26, 27,
        56, 162, 176, 213, 136, 239, 229, 242, 214, 53, 97, 19, 91, 195, 133, 126, 130, 1, 54, 143,
        78, 210, 176, 236, 152, 95, 92, 140, 158, 72, 151, 225, 83, 120, 44, 192, 72, 12, 100, 19,
        76, 249, 175, 180, 3, 217, 241, 47, 99, 8, 101, 17, 7, 154, 235, 191, 239, 243, 156, 137,
        147, 6, 248, 70, 44, 52, 4, 159, 137, 14, 79, 178, 247, 112, 241, 56, 240, 45, 22, 250, 99,
        99, 79, 10, 147, 188, 219, 89, 129, 60, 204, 149, 6, 112, 52, 85, 204, 62, 164, 85, 59,
        200, 11, 239, 196, 157, 53, 128, 223, 221, 90, 234, 112, 74, 195, 52, 133, 189, 35, 110,
        66, 222, 150, 19, 121, 107, 23, 171, 46, 167, 10, 253, 119, 247, 214, 175, 239, 40, 45, 24,
        115, 2, 150, 243, 44, 187, 160, 142, 68, 56, 172, 77, 143, 142, 53, 216, 228, 216, 239,
        176, 186, 96, 11, 147, 160, 127, 107, 192, 246, 173, 95, 144, 190, 167, 93, 172, 81, 89,
        163, 86, 111, 48, 30, 172, 32, 33, 34, 224, 191, 214, 244, 161, 233, 222, 113, 112, 76,
        163, 71, 99, 138, 92, 127, 203, 253, 201, 164, 231, 61, 59, 98, 165, 238, 23, 196, 10, 177,
        253, 110, 149, 31, 57, 212, 43, 16, 24, 241, 163, 72, 81, 140, 115, 37, 155, 94, 39,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_3_mod_4096_bits_odd(b: &mut Bencher) {
    let exponent = [3];

    // a random base
    let base = [
        167, 62, 26, 222, 69, 55, 136, 13, 139, 152, 251, 52, 101, 99, 181, 160, 55, 137, 73, 135,
        48, 71, 232, 168, 183, 14, 161, 145, 163, 22, 31, 14, 111, 65, 93, 52, 253, 42, 115, 167,
        211, 27, 143, 95, 20, 20, 179, 231, 241, 169, 110, 220, 42, 178, 222, 110, 97, 186, 76,
        210, 43, 205, 210, 166, 58, 173, 28, 10, 192, 102, 35, 122, 240, 244, 186, 197, 211, 39,
        230, 59, 140, 42, 221, 90, 81, 107, 88, 210, 1, 222, 215, 116, 7, 190, 251, 226, 87, 131,
        53, 137, 108, 245, 194, 62, 173, 26, 124, 253, 192, 10, 143, 10, 56, 237, 191, 216, 13, 6,
        22, 37, 130, 226, 241, 58, 157, 230, 247, 125, 170, 124, 167, 184, 148, 204, 247, 5, 181,
        73, 81, 195, 64, 72, 33, 187, 100, 173, 35, 86, 99, 228, 124, 83, 64, 1, 176, 178, 194,
        236, 110, 202, 70, 204, 59, 147, 232, 246, 152, 93, 124, 222, 192, 136, 149, 14, 36, 225,
        122, 228, 16, 215, 49, 80, 30, 230, 235, 158, 29, 86, 209, 67, 172, 163, 79, 22, 167, 113,
        154, 5, 157, 187, 237, 144, 76, 77, 45, 177, 15, 21, 19, 7, 181, 125, 232, 9, 138, 111,
        163, 207, 196, 197, 162, 67, 66, 81, 36, 3, 68, 68, 36, 213, 243, 42, 41, 12, 173, 71, 197,
        51, 217, 122, 98, 142, 79, 177, 34, 6, 160, 189, 158, 248, 115, 238, 226, 10, 71, 179, 92,
        87, 14, 100, 13, 220, 105, 238, 238, 0, 229, 100, 96, 29, 18, 155, 53, 247, 10, 146, 252,
        2, 219, 96, 206, 225, 171, 59, 135, 177, 90, 174, 245, 3, 222, 95, 35, 1, 31, 107, 53, 99,
        117, 93, 207, 148, 62, 12, 140, 211, 0, 122, 39, 66, 21, 71, 197, 137, 97, 142, 240, 61,
        69, 25, 121, 165, 53, 40, 218, 248, 112, 213, 8, 91, 217, 236, 173, 227, 103, 250, 133,
        143, 0, 227, 117, 159, 69, 8, 52, 147, 0, 175, 255, 76, 129, 47, 230, 234, 58, 233, 128, 2,
        184, 190, 104, 241, 69, 95, 74, 176, 112, 75, 236, 81, 100, 222, 111, 171, 126, 16, 55, 26,
        1, 63, 15, 24, 147, 176, 50, 20, 23, 176, 176, 182, 72, 208, 105, 138, 160, 243, 113, 123,
        106, 145, 39, 176, 0, 192, 137, 187, 182, 9, 5, 16, 72, 2, 121, 54, 224, 231, 208, 116,
        144, 55, 134, 20, 235, 123, 80, 105, 37, 196, 166, 51, 26, 151, 84, 52, 120, 162, 194, 97,
        212, 242, 153, 205, 112, 158, 151, 244, 242, 79, 219, 128, 99, 246, 32, 38, 132, 34, 131,
        28, 117, 162, 68, 107, 21, 25, 68, 66, 203, 7, 179, 46, 119, 8, 31, 36, 239, 179, 140, 51,
        221, 151, 200, 68, 183, 132, 36, 171, 247, 80, 191, 125, 22, 210, 204, 83, 200, 103, 97,
        184, 61, 225, 127, 73, 104, 159, 172, 185, 202, 116, 154, 28, 150, 6,
    ];

    // a random modulus
    let modulus = [
        186, 147, 169, 139, 184, 41, 41, 80, 108, 1, 29, 237, 140, 15, 147, 98, 8, 192, 145, 125,
        112, 68, 139, 38, 31, 131, 18, 213, 128, 28, 239, 126, 50, 54, 178, 48, 110, 200, 255, 10,
        206, 143, 168, 51, 43, 18, 24, 80, 166, 137, 197, 251, 215, 163, 248, 230, 16, 81, 164, 40,
        118, 207, 12, 149, 30, 55, 8, 135, 56, 9, 118, 228, 18, 190, 224, 181, 78, 70, 141, 109,
        114, 123, 200, 223, 241, 143, 249, 55, 171, 184, 16, 113, 63, 95, 194, 141, 196, 90, 118,
        203, 194, 73, 229, 187, 126, 241, 137, 9, 58, 68, 205, 95, 6, 254, 191, 224, 240, 85, 254,
        200, 6, 210, 24, 1, 32, 131, 185, 210, 109, 71, 116, 58, 213, 187, 115, 210, 107, 65, 116,
        172, 131, 209, 45, 65, 144, 7, 82, 199, 187, 97, 202, 5, 47, 159, 80, 235, 221, 89, 102,
        236, 123, 243, 25, 133, 28, 19, 140, 117, 77, 214, 127, 208, 94, 18, 24, 166, 38, 101, 165,
        144, 183, 109, 78, 100, 128, 67, 215, 253, 248, 244, 242, 182, 219, 26, 135, 24, 192, 149,
        247, 65, 206, 203, 28, 66, 205, 95, 65, 190, 163, 68, 24, 142, 228, 152, 231, 136, 169,
        197, 37, 124, 201, 235, 127, 51, 7, 197, 221, 161, 243, 212, 204, 128, 76, 52, 58, 3, 60,
        158, 162, 140, 51, 233, 167, 231, 190, 227, 111, 82, 210, 48, 79, 232, 80, 202, 114, 161,
        230, 250, 197, 122, 65, 90, 91, 110, 7, 106, 55, 170, 156, 118, 133, 101, 248, 234, 156,
        106, 228, 193, 33, 74, 82, 188, 205, 42, 2, 187, 75, 213, 33, 67, 206, 15, 163, 25, 72, 32,
        192, 92, 239, 40, 15, 116, 155, 120, 213, 50, 142, 47, 161, 30, 44, 213, 100, 217, 213,
        159, 190, 230, 163, 63, 152, 119, 190, 42, 154, 81, 39, 70, 179, 101, 169, 74, 37, 143, 49,
        134, 99, 130, 101, 87, 111, 240, 38, 240, 164, 134, 162, 124, 109, 161, 181, 192, 42, 57,
        240, 133, 18, 215, 175, 164, 45, 150, 161, 30, 104, 217, 68, 3, 5, 57, 170, 148, 24, 116,
        96, 172, 23, 139, 23, 230, 127, 182, 178, 167, 100, 46, 182, 190, 176, 4, 245, 103, 158,
        50, 96, 24, 27, 107, 135, 46, 122, 17, 225, 183, 0, 58, 3, 243, 168, 84, 64, 34, 29, 39,
        17, 227, 240, 30, 150, 28, 76, 11, 86, 53, 143, 25, 164, 215, 164, 70, 71, 232, 195, 95,
        221, 117, 135, 3, 241, 40, 207, 123, 122, 148, 118, 56, 65, 218, 8, 88, 124, 219, 91, 236,
        194, 245, 37, 246, 54, 202, 36, 151, 160, 40, 75, 92, 175, 241, 161, 113, 53, 220, 67, 249,
        7, 118, 77, 214, 122, 148, 44, 164, 249, 239, 57, 102, 45, 136, 93, 4, 76, 51, 186, 156,
        101, 82, 217, 116, 211, 255, 92, 183, 101, 50, 73, 38, 79, 219, 63, 140, 174, 217,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_3_mod_0512_bits_even(b: &mut Bencher) {
    let exponent = [3];

    // a random base
    let base = [
        243, 125, 228, 56, 107, 91, 133, 57, 46, 184, 164, 236, 176, 173, 36, 149, 58, 238, 150,
        32, 181, 248, 42, 134, 92, 170, 70, 16, 109, 212, 16, 28, 195, 174, 187, 226, 140, 22, 3,
        31, 96, 234, 110, 254, 106, 215, 101, 164, 190, 88, 14, 112, 151, 78, 205, 151, 254, 225,
        153, 125, 109, 4, 68, 87,
    ];

    // a random modulus
    let modulus = [
        249, 242, 76, 142, 109, 239, 3, 168, 130, 45, 156, 105, 209, 72, 218, 93, 86, 112, 88, 215,
        43, 194, 59, 35, 44, 86, 2, 252, 132, 113, 24, 4, 109, 98, 68, 209, 53, 191, 213, 162, 221,
        114, 213, 66, 58, 254, 152, 79, 82, 222, 79, 76, 1, 68, 255, 3, 218, 218, 83, 98, 85, 108,
        65, 86,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_3_mod_1024_bits_even(b: &mut Bencher) {
    let exponent = [3];

    // a random base
    let base = [
        212, 177, 76, 117, 205, 210, 60, 63, 64, 203, 108, 135, 17, 114, 163, 95, 224, 106, 191,
        205, 112, 55, 118, 224, 234, 35, 97, 95, 254, 115, 29, 29, 155, 136, 100, 24, 208, 77, 141,
        136, 62, 201, 198, 76, 142, 13, 77, 111, 231, 196, 136, 198, 18, 7, 42, 155, 127, 177, 176,
        250, 30, 83, 243, 231, 197, 161, 33, 122, 254, 152, 1, 214, 70, 166, 155, 142, 77, 132,
        159, 158, 102, 62, 53, 226, 255, 64, 220, 68, 201, 221, 248, 237, 88, 135, 64, 229, 111,
        183, 2, 52, 48, 1, 45, 146, 26, 132, 64, 31, 77, 137, 131, 245, 191, 166, 68, 1, 216, 211,
        177, 196, 122, 75, 212, 7, 183, 80, 240, 66,
    ];
    // a random modulus
    let modulus = [
        237, 223, 117, 18, 201, 74, 224, 241, 183, 182, 90, 141, 137, 224, 33, 142, 133, 75, 192,
        99, 45, 75, 185, 134, 226, 65, 105, 202, 253, 125, 129, 38, 135, 120, 49, 248, 112, 4, 211,
        189, 0, 156, 21, 44, 227, 62, 38, 6, 32, 45, 254, 31, 108, 151, 172, 166, 18, 46, 3, 141,
        22, 176, 57, 160, 40, 104, 39, 68, 167, 233, 192, 157, 33, 200, 231, 220, 195, 161, 137,
        235, 28, 4, 117, 223, 173, 4, 38, 143, 50, 16, 254, 176, 146, 13, 195, 210, 247, 134, 71,
        226, 101, 7, 52, 150, 227, 221, 149, 152, 52, 84, 142, 243, 197, 230, 134, 182, 126, 183,
        122, 82, 62, 74, 173, 42, 233, 38, 13, 242,
    ];
    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_3_mod_2048_bits_even(b: &mut Bencher) {
    let exponent = [3];

    // a random base
    let base = [
        236, 66, 86, 50, 109, 186, 105, 32, 149, 81, 7, 31, 151, 116, 93, 59, 84, 247, 239, 100,
        174, 245, 37, 174, 204, 240, 52, 226, 252, 105, 16, 88, 180, 24, 90, 223, 60, 226, 147,
        165, 255, 74, 249, 183, 1, 80, 160, 48, 111, 90, 220, 16, 237, 217, 251, 30, 208, 24, 127,
        170, 237, 244, 15, 89, 205, 254, 128, 49, 245, 233, 139, 239, 4, 126, 248, 212, 173, 103,
        222, 225, 118, 93, 219, 91, 26, 90, 142, 91, 210, 149, 227, 222, 255, 227, 182, 233, 176,
        83, 210, 143, 174, 92, 46, 19, 242, 117, 6, 125, 163, 93, 116, 63, 71, 236, 139, 17, 192,
        239, 106, 133, 145, 158, 46, 238, 107, 80, 54, 80, 231, 138, 236, 44, 55, 13, 193, 159,
        144, 85, 138, 204, 84, 126, 66, 40, 104, 232, 113, 216, 165, 184, 198, 20, 234, 225, 170,
        174, 90, 101, 253, 231, 80, 252, 28, 148, 89, 64, 65, 26, 143, 60, 158, 116, 12, 14, 46,
        210, 99, 233, 187, 212, 44, 36, 47, 227, 123, 195, 45, 115, 12, 123, 16, 164, 92, 52, 229,
        65, 127, 114, 213, 116, 210, 2, 149, 144, 217, 131, 146, 67, 66, 91, 199, 46, 58, 5, 185,
        247, 73, 170, 6, 45, 109, 0, 191, 233, 95, 239, 241, 30, 61, 119, 54, 4, 164, 214, 202,
        251, 139, 28, 22, 219, 100, 233, 195, 151, 237, 183, 41, 153, 42, 82, 208, 222, 21, 160,
        100, 100,
    ];

    // a random modulus
    let modulus = [
        155, 66, 179, 54, 45, 180, 207, 15, 110, 66, 217, 170, 218, 229, 14, 147, 163, 227, 26, 27,
        56, 162, 176, 213, 136, 239, 229, 242, 214, 53, 97, 19, 91, 195, 133, 126, 130, 1, 54, 143,
        78, 210, 176, 236, 152, 95, 92, 140, 158, 72, 151, 225, 83, 120, 44, 192, 72, 12, 100, 19,
        76, 249, 175, 180, 3, 217, 241, 47, 99, 8, 101, 17, 7, 154, 235, 191, 239, 243, 156, 137,
        147, 6, 248, 70, 44, 52, 4, 159, 137, 14, 79, 178, 247, 112, 241, 56, 240, 45, 22, 250, 99,
        99, 79, 10, 147, 188, 219, 89, 129, 60, 204, 149, 6, 112, 52, 85, 204, 62, 164, 85, 59,
        200, 11, 239, 196, 157, 53, 128, 223, 221, 90, 234, 112, 74, 195, 52, 133, 189, 35, 110,
        66, 222, 150, 19, 121, 107, 23, 171, 46, 167, 10, 253, 119, 247, 214, 175, 239, 40, 45, 24,
        115, 2, 150, 243, 44, 187, 160, 142, 68, 56, 172, 77, 143, 142, 53, 216, 228, 216, 239,
        176, 186, 96, 11, 147, 160, 127, 107, 192, 246, 173, 95, 144, 190, 167, 93, 172, 81, 89,
        163, 86, 111, 48, 30, 172, 32, 33, 34, 224, 191, 214, 244, 161, 233, 222, 113, 112, 76,
        163, 71, 99, 138, 92, 127, 203, 253, 201, 164, 231, 61, 59, 98, 165, 238, 23, 196, 10, 177,
        253, 110, 149, 31, 57, 212, 43, 16, 24, 241, 163, 72, 81, 140, 115, 37, 155, 94, 40,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_3_mod_4096_bits_even(b: &mut Bencher) {
    let exponent = [3];

    // a random base
    let base = [
        167, 62, 26, 222, 69, 55, 136, 13, 139, 152, 251, 52, 101, 99, 181, 160, 55, 137, 73, 135,
        48, 71, 232, 168, 183, 14, 161, 145, 163, 22, 31, 14, 111, 65, 93, 52, 253, 42, 115, 167,
        211, 27, 143, 95, 20, 20, 179, 231, 241, 169, 110, 220, 42, 178, 222, 110, 97, 186, 76,
        210, 43, 205, 210, 166, 58, 173, 28, 10, 192, 102, 35, 122, 240, 244, 186, 197, 211, 39,
        230, 59, 140, 42, 221, 90, 81, 107, 88, 210, 1, 222, 215, 116, 7, 190, 251, 226, 87, 131,
        53, 137, 108, 245, 194, 62, 173, 26, 124, 253, 192, 10, 143, 10, 56, 237, 191, 216, 13, 6,
        22, 37, 130, 226, 241, 58, 157, 230, 247, 125, 170, 124, 167, 184, 148, 204, 247, 5, 181,
        73, 81, 195, 64, 72, 33, 187, 100, 173, 35, 86, 99, 228, 124, 83, 64, 1, 176, 178, 194,
        236, 110, 202, 70, 204, 59, 147, 232, 246, 152, 93, 124, 222, 192, 136, 149, 14, 36, 225,
        122, 228, 16, 215, 49, 80, 30, 230, 235, 158, 29, 86, 209, 67, 172, 163, 79, 22, 167, 113,
        154, 5, 157, 187, 237, 144, 76, 77, 45, 177, 15, 21, 19, 7, 181, 125, 232, 9, 138, 111,
        163, 207, 196, 197, 162, 67, 66, 81, 36, 3, 68, 68, 36, 213, 243, 42, 41, 12, 173, 71, 197,
        51, 217, 122, 98, 142, 79, 177, 34, 6, 160, 189, 158, 248, 115, 238, 226, 10, 71, 179, 92,
        87, 14, 100, 13, 220, 105, 238, 238, 0, 229, 100, 96, 29, 18, 155, 53, 247, 10, 146, 252,
        2, 219, 96, 206, 225, 171, 59, 135, 177, 90, 174, 245, 3, 222, 95, 35, 1, 31, 107, 53, 99,
        117, 93, 207, 148, 62, 12, 140, 211, 0, 122, 39, 66, 21, 71, 197, 137, 97, 142, 240, 61,
        69, 25, 121, 165, 53, 40, 218, 248, 112, 213, 8, 91, 217, 236, 173, 227, 103, 250, 133,
        143, 0, 227, 117, 159, 69, 8, 52, 147, 0, 175, 255, 76, 129, 47, 230, 234, 58, 233, 128, 2,
        184, 190, 104, 241, 69, 95, 74, 176, 112, 75, 236, 81, 100, 222, 111, 171, 126, 16, 55, 26,
        1, 63, 15, 24, 147, 176, 50, 20, 23, 176, 176, 182, 72, 208, 105, 138, 160, 243, 113, 123,
        106, 145, 39, 176, 0, 192, 137, 187, 182, 9, 5, 16, 72, 2, 121, 54, 224, 231, 208, 116,
        144, 55, 134, 20, 235, 123, 80, 105, 37, 196, 166, 51, 26, 151, 84, 52, 120, 162, 194, 97,
        212, 242, 153, 205, 112, 158, 151, 244, 242, 79, 219, 128, 99, 246, 32, 38, 132, 34, 131,
        28, 117, 162, 68, 107, 21, 25, 68, 66, 203, 7, 179, 46, 119, 8, 31, 36, 239, 179, 140, 51,
        221, 151, 200, 68, 183, 132, 36, 171, 247, 80, 191, 125, 22, 210, 204, 83, 200, 103, 97,
        184, 61, 225, 127, 73, 104, 159, 172, 185, 202, 116, 154, 28, 150, 6,
    ];

    // a random modulus
    let modulus = [
        186, 147, 169, 139, 184, 41, 41, 80, 108, 1, 29, 237, 140, 15, 147, 98, 8, 192, 145, 125,
        112, 68, 139, 38, 31, 131, 18, 213, 128, 28, 239, 126, 50, 54, 178, 48, 110, 200, 255, 10,
        206, 143, 168, 51, 43, 18, 24, 80, 166, 137, 197, 251, 215, 163, 248, 230, 16, 81, 164, 40,
        118, 207, 12, 149, 30, 55, 8, 135, 56, 9, 118, 228, 18, 190, 224, 181, 78, 70, 141, 109,
        114, 123, 200, 223, 241, 143, 249, 55, 171, 184, 16, 113, 63, 95, 194, 141, 196, 90, 118,
        203, 194, 73, 229, 187, 126, 241, 137, 9, 58, 68, 205, 95, 6, 254, 191, 224, 240, 85, 254,
        200, 6, 210, 24, 1, 32, 131, 185, 210, 109, 71, 116, 58, 213, 187, 115, 210, 107, 65, 116,
        172, 131, 209, 45, 65, 144, 7, 82, 199, 187, 97, 202, 5, 47, 159, 80, 235, 221, 89, 102,
        236, 123, 243, 25, 133, 28, 19, 140, 117, 77, 214, 127, 208, 94, 18, 24, 166, 38, 101, 165,
        144, 183, 109, 78, 100, 128, 67, 215, 253, 248, 244, 242, 182, 219, 26, 135, 24, 192, 149,
        247, 65, 206, 203, 28, 66, 205, 95, 65, 190, 163, 68, 24, 142, 228, 152, 231, 136, 169,
        197, 37, 124, 201, 235, 127, 51, 7, 197, 221, 161, 243, 212, 204, 128, 76, 52, 58, 3, 60,
        158, 162, 140, 51, 233, 167, 231, 190, 227, 111, 82, 210, 48, 79, 232, 80, 202, 114, 161,
        230, 250, 197, 122, 65, 90, 91, 110, 7, 106, 55, 170, 156, 118, 133, 101, 248, 234, 156,
        106, 228, 193, 33, 74, 82, 188, 205, 42, 2, 187, 75, 213, 33, 67, 206, 15, 163, 25, 72, 32,
        192, 92, 239, 40, 15, 116, 155, 120, 213, 50, 142, 47, 161, 30, 44, 213, 100, 217, 213,
        159, 190, 230, 163, 63, 152, 119, 190, 42, 154, 81, 39, 70, 179, 101, 169, 74, 37, 143, 49,
        134, 99, 130, 101, 87, 111, 240, 38, 240, 164, 134, 162, 124, 109, 161, 181, 192, 42, 57,
        240, 133, 18, 215, 175, 164, 45, 150, 161, 30, 104, 217, 68, 3, 5, 57, 170, 148, 24, 116,
        96, 172, 23, 139, 23, 230, 127, 182, 178, 167, 100, 46, 182, 190, 176, 4, 245, 103, 158,
        50, 96, 24, 27, 107, 135, 46, 122, 17, 225, 183, 0, 58, 3, 243, 168, 84, 64, 34, 29, 39,
        17, 227, 240, 30, 150, 28, 76, 11, 86, 53, 143, 25, 164, 215, 164, 70, 71, 232, 195, 95,
        221, 117, 135, 3, 241, 40, 207, 123, 122, 148, 118, 56, 65, 218, 8, 88, 124, 219, 91, 236,
        194, 245, 37, 246, 54, 202, 36, 151, 160, 40, 75, 92, 175, 241, 161, 113, 53, 220, 67, 249,
        7, 118, 77, 214, 122, 148, 44, 164, 249, 239, 57, 102, 45, 136, 93, 4, 76, 51, 186, 156,
        101, 82, 217, 116, 211, 255, 92, 183, 101, 50, 73, 38, 79, 219, 63, 140, 174, 218,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_65537_mod_0512_bits_odd(b: &mut Bencher) {
    let exponent = [1, 0, 1];

    // a random base
    let base = [
        243, 125, 228, 56, 107, 91, 133, 57, 46, 184, 164, 236, 176, 173, 36, 149, 58, 238, 150,
        32, 181, 248, 42, 134, 92, 170, 70, 16, 109, 212, 16, 28, 195, 174, 187, 226, 140, 22, 3,
        31, 96, 234, 110, 254, 106, 215, 101, 164, 190, 88, 14, 112, 151, 78, 205, 151, 254, 225,
        153, 125, 109, 4, 68, 87,
    ];

    // a random modulus
    let modulus = [
        249, 242, 76, 142, 109, 239, 3, 168, 130, 45, 156, 105, 209, 72, 218, 93, 86, 112, 88, 215,
        43, 194, 59, 35, 44, 86, 2, 252, 132, 113, 24, 4, 109, 98, 68, 209, 53, 191, 213, 162, 221,
        114, 213, 66, 58, 254, 152, 79, 82, 222, 79, 76, 1, 68, 255, 3, 218, 218, 83, 98, 85, 108,
        65, 85,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_65537_mod_1024_bits_odd(b: &mut Bencher) {
    let exponent = [1, 0, 1];

    // a random base
    let base = [
        212, 177, 76, 117, 205, 210, 60, 63, 64, 203, 108, 135, 17, 114, 163, 95, 224, 106, 191,
        205, 112, 55, 118, 224, 234, 35, 97, 95, 254, 115, 29, 29, 155, 136, 100, 24, 208, 77, 141,
        136, 62, 201, 198, 76, 142, 13, 77, 111, 231, 196, 136, 198, 18, 7, 42, 155, 127, 177, 176,
        250, 30, 83, 243, 231, 197, 161, 33, 122, 254, 152, 1, 214, 70, 166, 155, 142, 77, 132,
        159, 158, 102, 62, 53, 226, 255, 64, 220, 68, 201, 221, 248, 237, 88, 135, 64, 229, 111,
        183, 2, 52, 48, 1, 45, 146, 26, 132, 64, 31, 77, 137, 131, 245, 191, 166, 68, 1, 216, 211,
        177, 196, 122, 75, 212, 7, 183, 80, 240, 66,
    ];

    // a random modulus
    let modulus = [
        237, 223, 117, 18, 201, 74, 224, 241, 183, 182, 90, 141, 137, 224, 33, 142, 133, 75, 192,
        99, 45, 75, 185, 134, 226, 65, 105, 202, 253, 125, 129, 38, 135, 120, 49, 248, 112, 4, 211,
        189, 0, 156, 21, 44, 227, 62, 38, 6, 32, 45, 254, 31, 108, 151, 172, 166, 18, 46, 3, 141,
        22, 176, 57, 160, 40, 104, 39, 68, 167, 233, 192, 157, 33, 200, 231, 220, 195, 161, 137,
        235, 28, 4, 117, 223, 173, 4, 38, 143, 50, 16, 254, 176, 146, 13, 195, 210, 247, 134, 71,
        226, 101, 7, 52, 150, 227, 221, 149, 152, 52, 84, 142, 243, 197, 230, 134, 182, 126, 183,
        122, 82, 62, 74, 173, 42, 233, 38, 13, 241,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_65537_mod_2048_bits_odd(b: &mut Bencher) {
    let exponent = [1, 0, 1];

    // a random base
    let base = [
        236, 66, 86, 50, 109, 186, 105, 32, 149, 81, 7, 31, 151, 116, 93, 59, 84, 247, 239, 100,
        174, 245, 37, 174, 204, 240, 52, 226, 252, 105, 16, 88, 180, 24, 90, 223, 60, 226, 147,
        165, 255, 74, 249, 183, 1, 80, 160, 48, 111, 90, 220, 16, 237, 217, 251, 30, 208, 24, 127,
        170, 237, 244, 15, 89, 205, 254, 128, 49, 245, 233, 139, 239, 4, 126, 248, 212, 173, 103,
        222, 225, 118, 93, 219, 91, 26, 90, 142, 91, 210, 149, 227, 222, 255, 227, 182, 233, 176,
        83, 210, 143, 174, 92, 46, 19, 242, 117, 6, 125, 163, 93, 116, 63, 71, 236, 139, 17, 192,
        239, 106, 133, 145, 158, 46, 238, 107, 80, 54, 80, 231, 138, 236, 44, 55, 13, 193, 159,
        144, 85, 138, 204, 84, 126, 66, 40, 104, 232, 113, 216, 165, 184, 198, 20, 234, 225, 170,
        174, 90, 101, 253, 231, 80, 252, 28, 148, 89, 64, 65, 26, 143, 60, 158, 116, 12, 14, 46,
        210, 99, 233, 187, 212, 44, 36, 47, 227, 123, 195, 45, 115, 12, 123, 16, 164, 92, 52, 229,
        65, 127, 114, 213, 116, 210, 2, 149, 144, 217, 131, 146, 67, 66, 91, 199, 46, 58, 5, 185,
        247, 73, 170, 6, 45, 109, 0, 191, 233, 95, 239, 241, 30, 61, 119, 54, 4, 164, 214, 202,
        251, 139, 28, 22, 219, 100, 233, 195, 151, 237, 183, 41, 153, 42, 82, 208, 222, 21, 160,
        100, 100,
    ];

    // a random modulus
    let modulus = [
        155, 66, 179, 54, 45, 180, 207, 15, 110, 66, 217, 170, 218, 229, 14, 147, 163, 227, 26, 27,
        56, 162, 176, 213, 136, 239, 229, 242, 214, 53, 97, 19, 91, 195, 133, 126, 130, 1, 54, 143,
        78, 210, 176, 236, 152, 95, 92, 140, 158, 72, 151, 225, 83, 120, 44, 192, 72, 12, 100, 19,
        76, 249, 175, 180, 3, 217, 241, 47, 99, 8, 101, 17, 7, 154, 235, 191, 239, 243, 156, 137,
        147, 6, 248, 70, 44, 52, 4, 159, 137, 14, 79, 178, 247, 112, 241, 56, 240, 45, 22, 250, 99,
        99, 79, 10, 147, 188, 219, 89, 129, 60, 204, 149, 6, 112, 52, 85, 204, 62, 164, 85, 59,
        200, 11, 239, 196, 157, 53, 128, 223, 221, 90, 234, 112, 74, 195, 52, 133, 189, 35, 110,
        66, 222, 150, 19, 121, 107, 23, 171, 46, 167, 10, 253, 119, 247, 214, 175, 239, 40, 45, 24,
        115, 2, 150, 243, 44, 187, 160, 142, 68, 56, 172, 77, 143, 142, 53, 216, 228, 216, 239,
        176, 186, 96, 11, 147, 160, 127, 107, 192, 246, 173, 95, 144, 190, 167, 93, 172, 81, 89,
        163, 86, 111, 48, 30, 172, 32, 33, 34, 224, 191, 214, 244, 161, 233, 222, 113, 112, 76,
        163, 71, 99, 138, 92, 127, 203, 253, 201, 164, 231, 61, 59, 98, 165, 238, 23, 196, 10, 177,
        253, 110, 149, 31, 57, 212, 43, 16, 24, 241, 163, 72, 81, 140, 115, 37, 155, 94, 39,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_65537_mod_4096_bits_odd(b: &mut Bencher) {
    let exponent = [1, 0, 1];

    // a random base
    let base = [
        167, 62, 26, 222, 69, 55, 136, 13, 139, 152, 251, 52, 101, 99, 181, 160, 55, 137, 73, 135,
        48, 71, 232, 168, 183, 14, 161, 145, 163, 22, 31, 14, 111, 65, 93, 52, 253, 42, 115, 167,
        211, 27, 143, 95, 20, 20, 179, 231, 241, 169, 110, 220, 42, 178, 222, 110, 97, 186, 76,
        210, 43, 205, 210, 166, 58, 173, 28, 10, 192, 102, 35, 122, 240, 244, 186, 197, 211, 39,
        230, 59, 140, 42, 221, 90, 81, 107, 88, 210, 1, 222, 215, 116, 7, 190, 251, 226, 87, 131,
        53, 137, 108, 245, 194, 62, 173, 26, 124, 253, 192, 10, 143, 10, 56, 237, 191, 216, 13, 6,
        22, 37, 130, 226, 241, 58, 157, 230, 247, 125, 170, 124, 167, 184, 148, 204, 247, 5, 181,
        73, 81, 195, 64, 72, 33, 187, 100, 173, 35, 86, 99, 228, 124, 83, 64, 1, 176, 178, 194,
        236, 110, 202, 70, 204, 59, 147, 232, 246, 152, 93, 124, 222, 192, 136, 149, 14, 36, 225,
        122, 228, 16, 215, 49, 80, 30, 230, 235, 158, 29, 86, 209, 67, 172, 163, 79, 22, 167, 113,
        154, 5, 157, 187, 237, 144, 76, 77, 45, 177, 15, 21, 19, 7, 181, 125, 232, 9, 138, 111,
        163, 207, 196, 197, 162, 67, 66, 81, 36, 3, 68, 68, 36, 213, 243, 42, 41, 12, 173, 71, 197,
        51, 217, 122, 98, 142, 79, 177, 34, 6, 160, 189, 158, 248, 115, 238, 226, 10, 71, 179, 92,
        87, 14, 100, 13, 220, 105, 238, 238, 0, 229, 100, 96, 29, 18, 155, 53, 247, 10, 146, 252,
        2, 219, 96, 206, 225, 171, 59, 135, 177, 90, 174, 245, 3, 222, 95, 35, 1, 31, 107, 53, 99,
        117, 93, 207, 148, 62, 12, 140, 211, 0, 122, 39, 66, 21, 71, 197, 137, 97, 142, 240, 61,
        69, 25, 121, 165, 53, 40, 218, 248, 112, 213, 8, 91, 217, 236, 173, 227, 103, 250, 133,
        143, 0, 227, 117, 159, 69, 8, 52, 147, 0, 175, 255, 76, 129, 47, 230, 234, 58, 233, 128, 2,
        184, 190, 104, 241, 69, 95, 74, 176, 112, 75, 236, 81, 100, 222, 111, 171, 126, 16, 55, 26,
        1, 63, 15, 24, 147, 176, 50, 20, 23, 176, 176, 182, 72, 208, 105, 138, 160, 243, 113, 123,
        106, 145, 39, 176, 0, 192, 137, 187, 182, 9, 5, 16, 72, 2, 121, 54, 224, 231, 208, 116,
        144, 55, 134, 20, 235, 123, 80, 105, 37, 196, 166, 51, 26, 151, 84, 52, 120, 162, 194, 97,
        212, 242, 153, 205, 112, 158, 151, 244, 242, 79, 219, 128, 99, 246, 32, 38, 132, 34, 131,
        28, 117, 162, 68, 107, 21, 25, 68, 66, 203, 7, 179, 46, 119, 8, 31, 36, 239, 179, 140, 51,
        221, 151, 200, 68, 183, 132, 36, 171, 247, 80, 191, 125, 22, 210, 204, 83, 200, 103, 97,
        184, 61, 225, 127, 73, 104, 159, 172, 185, 202, 116, 154, 28, 150, 6,
    ];

    // a random modulus
    let modulus = [
        186, 147, 169, 139, 184, 41, 41, 80, 108, 1, 29, 237, 140, 15, 147, 98, 8, 192, 145, 125,
        112, 68, 139, 38, 31, 131, 18, 213, 128, 28, 239, 126, 50, 54, 178, 48, 110, 200, 255, 10,
        206, 143, 168, 51, 43, 18, 24, 80, 166, 137, 197, 251, 215, 163, 248, 230, 16, 81, 164, 40,
        118, 207, 12, 149, 30, 55, 8, 135, 56, 9, 118, 228, 18, 190, 224, 181, 78, 70, 141, 109,
        114, 123, 200, 223, 241, 143, 249, 55, 171, 184, 16, 113, 63, 95, 194, 141, 196, 90, 118,
        203, 194, 73, 229, 187, 126, 241, 137, 9, 58, 68, 205, 95, 6, 254, 191, 224, 240, 85, 254,
        200, 6, 210, 24, 1, 32, 131, 185, 210, 109, 71, 116, 58, 213, 187, 115, 210, 107, 65, 116,
        172, 131, 209, 45, 65, 144, 7, 82, 199, 187, 97, 202, 5, 47, 159, 80, 235, 221, 89, 102,
        236, 123, 243, 25, 133, 28, 19, 140, 117, 77, 214, 127, 208, 94, 18, 24, 166, 38, 101, 165,
        144, 183, 109, 78, 100, 128, 67, 215, 253, 248, 244, 242, 182, 219, 26, 135, 24, 192, 149,
        247, 65, 206, 203, 28, 66, 205, 95, 65, 190, 163, 68, 24, 142, 228, 152, 231, 136, 169,
        197, 37, 124, 201, 235, 127, 51, 7, 197, 221, 161, 243, 212, 204, 128, 76, 52, 58, 3, 60,
        158, 162, 140, 51, 233, 167, 231, 190, 227, 111, 82, 210, 48, 79, 232, 80, 202, 114, 161,
        230, 250, 197, 122, 65, 90, 91, 110, 7, 106, 55, 170, 156, 118, 133, 101, 248, 234, 156,
        106, 228, 193, 33, 74, 82, 188, 205, 42, 2, 187, 75, 213, 33, 67, 206, 15, 163, 25, 72, 32,
        192, 92, 239, 40, 15, 116, 155, 120, 213, 50, 142, 47, 161, 30, 44, 213, 100, 217, 213,
        159, 190, 230, 163, 63, 152, 119, 190, 42, 154, 81, 39, 70, 179, 101, 169, 74, 37, 143, 49,
        134, 99, 130, 101, 87, 111, 240, 38, 240, 164, 134, 162, 124, 109, 161, 181, 192, 42, 57,
        240, 133, 18, 215, 175, 164, 45, 150, 161, 30, 104, 217, 68, 3, 5, 57, 170, 148, 24, 116,
        96, 172, 23, 139, 23, 230, 127, 182, 178, 167, 100, 46, 182, 190, 176, 4, 245, 103, 158,
        50, 96, 24, 27, 107, 135, 46, 122, 17, 225, 183, 0, 58, 3, 243, 168, 84, 64, 34, 29, 39,
        17, 227, 240, 30, 150, 28, 76, 11, 86, 53, 143, 25, 164, 215, 164, 70, 71, 232, 195, 95,
        221, 117, 135, 3, 241, 40, 207, 123, 122, 148, 118, 56, 65, 218, 8, 88, 124, 219, 91, 236,
        194, 245, 37, 246, 54, 202, 36, 151, 160, 40, 75, 92, 175, 241, 161, 113, 53, 220, 67, 249,
        7, 118, 77, 214, 122, 148, 44, 164, 249, 239, 57, 102, 45, 136, 93, 4, 76, 51, 186, 156,
        101, 82, 217, 116, 211, 255, 92, 183, 101, 50, 73, 38, 79, 219, 63, 140, 174, 217,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_65537_mod_0512_bits_even(b: &mut Bencher) {
    let exponent = [1, 0, 1];

    // a random base
    let base = [
        243, 125, 228, 56, 107, 91, 133, 57, 46, 184, 164, 236, 176, 173, 36, 149, 58, 238, 150,
        32, 181, 248, 42, 134, 92, 170, 70, 16, 109, 212, 16, 28, 195, 174, 187, 226, 140, 22, 3,
        31, 96, 234, 110, 254, 106, 215, 101, 164, 190, 88, 14, 112, 151, 78, 205, 151, 254, 225,
        153, 125, 109, 4, 68, 87,
    ];

    // a random modulus
    let modulus = [
        249, 242, 76, 142, 109, 239, 3, 168, 130, 45, 156, 105, 209, 72, 218, 93, 86, 112, 88, 215,
        43, 194, 59, 35, 44, 86, 2, 252, 132, 113, 24, 4, 109, 98, 68, 209, 53, 191, 213, 162, 221,
        114, 213, 66, 58, 254, 152, 79, 82, 222, 79, 76, 1, 68, 255, 3, 218, 218, 83, 98, 85, 108,
        65, 86,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_65537_mod_1024_bits_even(b: &mut Bencher) {
    let exponent = [1, 0, 1];

    // a random base
    let base = [
        212, 177, 76, 117, 205, 210, 60, 63, 64, 203, 108, 135, 17, 114, 163, 95, 224, 106, 191,
        205, 112, 55, 118, 224, 234, 35, 97, 95, 254, 115, 29, 29, 155, 136, 100, 24, 208, 77, 141,
        136, 62, 201, 198, 76, 142, 13, 77, 111, 231, 196, 136, 198, 18, 7, 42, 155, 127, 177, 176,
        250, 30, 83, 243, 231, 197, 161, 33, 122, 254, 152, 1, 214, 70, 166, 155, 142, 77, 132,
        159, 158, 102, 62, 53, 226, 255, 64, 220, 68, 201, 221, 248, 237, 88, 135, 64, 229, 111,
        183, 2, 52, 48, 1, 45, 146, 26, 132, 64, 31, 77, 137, 131, 245, 191, 166, 68, 1, 216, 211,
        177, 196, 122, 75, 212, 7, 183, 80, 240, 66,
    ];

    // a random modulus
    let modulus = [
        237, 223, 117, 18, 201, 74, 224, 241, 183, 182, 90, 141, 137, 224, 33, 142, 133, 75, 192,
        99, 45, 75, 185, 134, 226, 65, 105, 202, 253, 125, 129, 38, 135, 120, 49, 248, 112, 4, 211,
        189, 0, 156, 21, 44, 227, 62, 38, 6, 32, 45, 254, 31, 108, 151, 172, 166, 18, 46, 3, 141,
        22, 176, 57, 160, 40, 104, 39, 68, 167, 233, 192, 157, 33, 200, 231, 220, 195, 161, 137,
        235, 28, 4, 117, 223, 173, 4, 38, 143, 50, 16, 254, 176, 146, 13, 195, 210, 247, 134, 71,
        226, 101, 7, 52, 150, 227, 221, 149, 152, 52, 84, 142, 243, 197, 230, 134, 182, 126, 183,
        122, 82, 62, 74, 173, 42, 233, 38, 13, 242,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_65537_mod_2048_bits_even(b: &mut Bencher) {
    let exponent = [1, 0, 1];

    // a random base
    let base = [
        236, 66, 86, 50, 109, 186, 105, 32, 149, 81, 7, 31, 151, 116, 93, 59, 84, 247, 239, 100,
        174, 245, 37, 174, 204, 240, 52, 226, 252, 105, 16, 88, 180, 24, 90, 223, 60, 226, 147,
        165, 255, 74, 249, 183, 1, 80, 160, 48, 111, 90, 220, 16, 237, 217, 251, 30, 208, 24, 127,
        170, 237, 244, 15, 89, 205, 254, 128, 49, 245, 233, 139, 239, 4, 126, 248, 212, 173, 103,
        222, 225, 118, 93, 219, 91, 26, 90, 142, 91, 210, 149, 227, 222, 255, 227, 182, 233, 176,
        83, 210, 143, 174, 92, 46, 19, 242, 117, 6, 125, 163, 93, 116, 63, 71, 236, 139, 17, 192,
        239, 106, 133, 145, 158, 46, 238, 107, 80, 54, 80, 231, 138, 236, 44, 55, 13, 193, 159,
        144, 85, 138, 204, 84, 126, 66, 40, 104, 232, 113, 216, 165, 184, 198, 20, 234, 225, 170,
        174, 90, 101, 253, 231, 80, 252, 28, 148, 89, 64, 65, 26, 143, 60, 158, 116, 12, 14, 46,
        210, 99, 233, 187, 212, 44, 36, 47, 227, 123, 195, 45, 115, 12, 123, 16, 164, 92, 52, 229,
        65, 127, 114, 213, 116, 210, 2, 149, 144, 217, 131, 146, 67, 66, 91, 199, 46, 58, 5, 185,
        247, 73, 170, 6, 45, 109, 0, 191, 233, 95, 239, 241, 30, 61, 119, 54, 4, 164, 214, 202,
        251, 139, 28, 22, 219, 100, 233, 195, 151, 237, 183, 41, 153, 42, 82, 208, 222, 21, 160,
        100, 100,
    ];

    // a random modulus
    let modulus = [
        155, 66, 179, 54, 45, 180, 207, 15, 110, 66, 217, 170, 218, 229, 14, 147, 163, 227, 26, 27,
        56, 162, 176, 213, 136, 239, 229, 242, 214, 53, 97, 19, 91, 195, 133, 126, 130, 1, 54, 143,
        78, 210, 176, 236, 152, 95, 92, 140, 158, 72, 151, 225, 83, 120, 44, 192, 72, 12, 100, 19,
        76, 249, 175, 180, 3, 217, 241, 47, 99, 8, 101, 17, 7, 154, 235, 191, 239, 243, 156, 137,
        147, 6, 248, 70, 44, 52, 4, 159, 137, 14, 79, 178, 247, 112, 241, 56, 240, 45, 22, 250, 99,
        99, 79, 10, 147, 188, 219, 89, 129, 60, 204, 149, 6, 112, 52, 85, 204, 62, 164, 85, 59,
        200, 11, 239, 196, 157, 53, 128, 223, 221, 90, 234, 112, 74, 195, 52, 133, 189, 35, 110,
        66, 222, 150, 19, 121, 107, 23, 171, 46, 167, 10, 253, 119, 247, 214, 175, 239, 40, 45, 24,
        115, 2, 150, 243, 44, 187, 160, 142, 68, 56, 172, 77, 143, 142, 53, 216, 228, 216, 239,
        176, 186, 96, 11, 147, 160, 127, 107, 192, 246, 173, 95, 144, 190, 167, 93, 172, 81, 89,
        163, 86, 111, 48, 30, 172, 32, 33, 34, 224, 191, 214, 244, 161, 233, 222, 113, 112, 76,
        163, 71, 99, 138, 92, 127, 203, 253, 201, 164, 231, 61, 59, 98, 165, 238, 23, 196, 10, 177,
        253, 110, 149, 31, 57, 212, 43, 16, 24, 241, 163, 72, 81, 140, 115, 37, 155, 94, 40,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_65537_mod_4096_bits_even(b: &mut Bencher) {
    let exponent = [1, 0, 1];

    // a random base
    let base = [
        167, 62, 26, 222, 69, 55, 136, 13, 139, 152, 251, 52, 101, 99, 181, 160, 55, 137, 73, 135,
        48, 71, 232, 168, 183, 14, 161, 145, 163, 22, 31, 14, 111, 65, 93, 52, 253, 42, 115, 167,
        211, 27, 143, 95, 20, 20, 179, 231, 241, 169, 110, 220, 42, 178, 222, 110, 97, 186, 76,
        210, 43, 205, 210, 166, 58, 173, 28, 10, 192, 102, 35, 122, 240, 244, 186, 197, 211, 39,
        230, 59, 140, 42, 221, 90, 81, 107, 88, 210, 1, 222, 215, 116, 7, 190, 251, 226, 87, 131,
        53, 137, 108, 245, 194, 62, 173, 26, 124, 253, 192, 10, 143, 10, 56, 237, 191, 216, 13, 6,
        22, 37, 130, 226, 241, 58, 157, 230, 247, 125, 170, 124, 167, 184, 148, 204, 247, 5, 181,
        73, 81, 195, 64, 72, 33, 187, 100, 173, 35, 86, 99, 228, 124, 83, 64, 1, 176, 178, 194,
        236, 110, 202, 70, 204, 59, 147, 232, 246, 152, 93, 124, 222, 192, 136, 149, 14, 36, 225,
        122, 228, 16, 215, 49, 80, 30, 230, 235, 158, 29, 86, 209, 67, 172, 163, 79, 22, 167, 113,
        154, 5, 157, 187, 237, 144, 76, 77, 45, 177, 15, 21, 19, 7, 181, 125, 232, 9, 138, 111,
        163, 207, 196, 197, 162, 67, 66, 81, 36, 3, 68, 68, 36, 213, 243, 42, 41, 12, 173, 71, 197,
        51, 217, 122, 98, 142, 79, 177, 34, 6, 160, 189, 158, 248, 115, 238, 226, 10, 71, 179, 92,
        87, 14, 100, 13, 220, 105, 238, 238, 0, 229, 100, 96, 29, 18, 155, 53, 247, 10, 146, 252,
        2, 219, 96, 206, 225, 171, 59, 135, 177, 90, 174, 245, 3, 222, 95, 35, 1, 31, 107, 53, 99,
        117, 93, 207, 148, 62, 12, 140, 211, 0, 122, 39, 66, 21, 71, 197, 137, 97, 142, 240, 61,
        69, 25, 121, 165, 53, 40, 218, 248, 112, 213, 8, 91, 217, 236, 173, 227, 103, 250, 133,
        143, 0, 227, 117, 159, 69, 8, 52, 147, 0, 175, 255, 76, 129, 47, 230, 234, 58, 233, 128, 2,
        184, 190, 104, 241, 69, 95, 74, 176, 112, 75, 236, 81, 100, 222, 111, 171, 126, 16, 55, 26,
        1, 63, 15, 24, 147, 176, 50, 20, 23, 176, 176, 182, 72, 208, 105, 138, 160, 243, 113, 123,
        106, 145, 39, 176, 0, 192, 137, 187, 182, 9, 5, 16, 72, 2, 121, 54, 224, 231, 208, 116,
        144, 55, 134, 20, 235, 123, 80, 105, 37, 196, 166, 51, 26, 151, 84, 52, 120, 162, 194, 97,
        212, 242, 153, 205, 112, 158, 151, 244, 242, 79, 219, 128, 99, 246, 32, 38, 132, 34, 131,
        28, 117, 162, 68, 107, 21, 25, 68, 66, 203, 7, 179, 46, 119, 8, 31, 36, 239, 179, 140, 51,
        221, 151, 200, 68, 183, 132, 36, 171, 247, 80, 191, 125, 22, 210, 204, 83, 200, 103, 97,
        184, 61, 225, 127, 73, 104, 159, 172, 185, 202, 116, 154, 28, 150, 6,
    ];

    // a random modulus
    let modulus = [
        186, 147, 169, 139, 184, 41, 41, 80, 108, 1, 29, 237, 140, 15, 147, 98, 8, 192, 145, 125,
        112, 68, 139, 38, 31, 131, 18, 213, 128, 28, 239, 126, 50, 54, 178, 48, 110, 200, 255, 10,
        206, 143, 168, 51, 43, 18, 24, 80, 166, 137, 197, 251, 215, 163, 248, 230, 16, 81, 164, 40,
        118, 207, 12, 149, 30, 55, 8, 135, 56, 9, 118, 228, 18, 190, 224, 181, 78, 70, 141, 109,
        114, 123, 200, 223, 241, 143, 249, 55, 171, 184, 16, 113, 63, 95, 194, 141, 196, 90, 118,
        203, 194, 73, 229, 187, 126, 241, 137, 9, 58, 68, 205, 95, 6, 254, 191, 224, 240, 85, 254,
        200, 6, 210, 24, 1, 32, 131, 185, 210, 109, 71, 116, 58, 213, 187, 115, 210, 107, 65, 116,
        172, 131, 209, 45, 65, 144, 7, 82, 199, 187, 97, 202, 5, 47, 159, 80, 235, 221, 89, 102,
        236, 123, 243, 25, 133, 28, 19, 140, 117, 77, 214, 127, 208, 94, 18, 24, 166, 38, 101, 165,
        144, 183, 109, 78, 100, 128, 67, 215, 253, 248, 244, 242, 182, 219, 26, 135, 24, 192, 149,
        247, 65, 206, 203, 28, 66, 205, 95, 65, 190, 163, 68, 24, 142, 228, 152, 231, 136, 169,
        197, 37, 124, 201, 235, 127, 51, 7, 197, 221, 161, 243, 212, 204, 128, 76, 52, 58, 3, 60,
        158, 162, 140, 51, 233, 167, 231, 190, 227, 111, 82, 210, 48, 79, 232, 80, 202, 114, 161,
        230, 250, 197, 122, 65, 90, 91, 110, 7, 106, 55, 170, 156, 118, 133, 101, 248, 234, 156,
        106, 228, 193, 33, 74, 82, 188, 205, 42, 2, 187, 75, 213, 33, 67, 206, 15, 163, 25, 72, 32,
        192, 92, 239, 40, 15, 116, 155, 120, 213, 50, 142, 47, 161, 30, 44, 213, 100, 217, 213,
        159, 190, 230, 163, 63, 152, 119, 190, 42, 154, 81, 39, 70, 179, 101, 169, 74, 37, 143, 49,
        134, 99, 130, 101, 87, 111, 240, 38, 240, 164, 134, 162, 124, 109, 161, 181, 192, 42, 57,
        240, 133, 18, 215, 175, 164, 45, 150, 161, 30, 104, 217, 68, 3, 5, 57, 170, 148, 24, 116,
        96, 172, 23, 139, 23, 230, 127, 182, 178, 167, 100, 46, 182, 190, 176, 4, 245, 103, 158,
        50, 96, 24, 27, 107, 135, 46, 122, 17, 225, 183, 0, 58, 3, 243, 168, 84, 64, 34, 29, 39,
        17, 227, 240, 30, 150, 28, 76, 11, 86, 53, 143, 25, 164, 215, 164, 70, 71, 232, 195, 95,
        221, 117, 135, 3, 241, 40, 207, 123, 122, 148, 118, 56, 65, 218, 8, 88, 124, 219, 91, 236,
        194, 245, 37, 246, 54, 202, 36, 151, 160, 40, 75, 92, 175, 241, 161, 113, 53, 220, 67, 249,
        7, 118, 77, 214, 122, 148, 44, 164, 249, 239, 57, 102, 45, 136, 93, 4, 76, 51, 186, 156,
        101, 82, 217, 116, 211, 255, 92, 183, 101, 50, 73, 38, 79, 219, 63, 140, 174, 218,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_0512_bits_mod_0512_bits_odd(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        30, 194, 86, 18, 177, 202, 203, 133, 7, 22, 133, 138, 102, 21, 107, 127, 141, 28, 166, 162,
        46, 101, 20, 119, 208, 32, 66, 141, 122, 70, 94, 79, 61, 251, 202, 97, 35, 129, 9, 73, 215,
        220, 88, 174, 113, 124, 103, 113, 254, 44, 96, 79, 250, 252, 33, 252, 12, 70, 43, 238, 90,
        32, 248, 119,
    ];

    // a random base
    let base = [
        243, 125, 228, 56, 107, 91, 133, 57, 46, 184, 164, 236, 176, 173, 36, 149, 58, 238, 150,
        32, 181, 248, 42, 134, 92, 170, 70, 16, 109, 212, 16, 28, 195, 174, 187, 226, 140, 22, 3,
        31, 96, 234, 110, 254, 106, 215, 101, 164, 190, 88, 14, 112, 151, 78, 205, 151, 254, 225,
        153, 125, 109, 4, 68, 87,
    ];

    // a random modulus
    let modulus = [
        249, 242, 76, 142, 109, 239, 3, 168, 130, 45, 156, 105, 209, 72, 218, 93, 86, 112, 88, 215,
        43, 194, 59, 35, 44, 86, 2, 252, 132, 113, 24, 4, 109, 98, 68, 209, 53, 191, 213, 162, 221,
        114, 213, 66, 58, 254, 152, 79, 82, 222, 79, 76, 1, 68, 255, 3, 218, 218, 83, 98, 85, 108,
        65, 85,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_0512_bits_mod_1024_bits_odd(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        30, 194, 86, 18, 177, 202, 203, 133, 7, 22, 133, 138, 102, 21, 107, 127, 141, 28, 166, 162,
        46, 101, 20, 119, 208, 32, 66, 141, 122, 70, 94, 79, 61, 251, 202, 97, 35, 129, 9, 73, 215,
        220, 88, 174, 113, 124, 103, 113, 254, 44, 96, 79, 250, 252, 33, 252, 12, 70, 43, 238, 90,
        32, 248, 119,
    ];

    // a random base
    let base = [
        212, 177, 76, 117, 205, 210, 60, 63, 64, 203, 108, 135, 17, 114, 163, 95, 224, 106, 191,
        205, 112, 55, 118, 224, 234, 35, 97, 95, 254, 115, 29, 29, 155, 136, 100, 24, 208, 77, 141,
        136, 62, 201, 198, 76, 142, 13, 77, 111, 231, 196, 136, 198, 18, 7, 42, 155, 127, 177, 176,
        250, 30, 83, 243, 231, 197, 161, 33, 122, 254, 152, 1, 214, 70, 166, 155, 142, 77, 132,
        159, 158, 102, 62, 53, 226, 255, 64, 220, 68, 201, 221, 248, 237, 88, 135, 64, 229, 111,
        183, 2, 52, 48, 1, 45, 146, 26, 132, 64, 31, 77, 137, 131, 245, 191, 166, 68, 1, 216, 211,
        177, 196, 122, 75, 212, 7, 183, 80, 240, 66,
    ];

    // a random modulus
    let modulus = [
        237, 223, 117, 18, 201, 74, 224, 241, 183, 182, 90, 141, 137, 224, 33, 142, 133, 75, 192,
        99, 45, 75, 185, 134, 226, 65, 105, 202, 253, 125, 129, 38, 135, 120, 49, 248, 112, 4, 211,
        189, 0, 156, 21, 44, 227, 62, 38, 6, 32, 45, 254, 31, 108, 151, 172, 166, 18, 46, 3, 141,
        22, 176, 57, 160, 40, 104, 39, 68, 167, 233, 192, 157, 33, 200, 231, 220, 195, 161, 137,
        235, 28, 4, 117, 223, 173, 4, 38, 143, 50, 16, 254, 176, 146, 13, 195, 210, 247, 134, 71,
        226, 101, 7, 52, 150, 227, 221, 149, 152, 52, 84, 142, 243, 197, 230, 134, 182, 126, 183,
        122, 82, 62, 74, 173, 42, 233, 38, 13, 241,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_1024_bits_mod_1024_bits_odd(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        107, 92, 159, 59, 101, 117, 205, 228, 222, 58, 188, 58, 254, 101, 230, 53, 203, 200, 138,
        56, 160, 233, 81, 218, 113, 119, 10, 214, 68, 109, 113, 15, 146, 191, 225, 80, 22, 199,
        119, 236, 23, 159, 148, 40, 113, 28, 75, 45, 15, 54, 5, 64, 103, 55, 1, 220, 236, 41, 218,
        41, 93, 6, 3, 106, 235, 31, 22, 73, 243, 113, 171, 111, 20, 237, 200, 8, 99, 252, 202, 99,
        122, 242, 84, 180, 8, 58, 3, 129, 145, 62, 179, 78, 199, 35, 212, 16, 3, 55, 9, 197, 217,
        30, 42, 67, 220, 121, 193, 16, 15, 170, 116, 65, 157, 109, 34, 211, 41, 116, 161, 40, 77,
        223, 200, 240, 31, 17, 141, 189,
    ];

    // a random base
    let base = [
        212, 177, 76, 117, 205, 210, 60, 63, 64, 203, 108, 135, 17, 114, 163, 95, 224, 106, 191,
        205, 112, 55, 118, 224, 234, 35, 97, 95, 254, 115, 29, 29, 155, 136, 100, 24, 208, 77, 141,
        136, 62, 201, 198, 76, 142, 13, 77, 111, 231, 196, 136, 198, 18, 7, 42, 155, 127, 177, 176,
        250, 30, 83, 243, 231, 197, 161, 33, 122, 254, 152, 1, 214, 70, 166, 155, 142, 77, 132,
        159, 158, 102, 62, 53, 226, 255, 64, 220, 68, 201, 221, 248, 237, 88, 135, 64, 229, 111,
        183, 2, 52, 48, 1, 45, 146, 26, 132, 64, 31, 77, 137, 131, 245, 191, 166, 68, 1, 216, 211,
        177, 196, 122, 75, 212, 7, 183, 80, 240, 66,
    ];

    // a random modulus
    let modulus = [
        237, 223, 117, 18, 201, 74, 224, 241, 183, 182, 90, 141, 137, 224, 33, 142, 133, 75, 192,
        99, 45, 75, 185, 134, 226, 65, 105, 202, 253, 125, 129, 38, 135, 120, 49, 248, 112, 4, 211,
        189, 0, 156, 21, 44, 227, 62, 38, 6, 32, 45, 254, 31, 108, 151, 172, 166, 18, 46, 3, 141,
        22, 176, 57, 160, 40, 104, 39, 68, 167, 233, 192, 157, 33, 200, 231, 220, 195, 161, 137,
        235, 28, 4, 117, 223, 173, 4, 38, 143, 50, 16, 254, 176, 146, 13, 195, 210, 247, 134, 71,
        226, 101, 7, 52, 150, 227, 221, 149, 152, 52, 84, 142, 243, 197, 230, 134, 182, 126, 183,
        122, 82, 62, 74, 173, 42, 233, 38, 13, 241,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_0512_bits_mod_2048_bits_odd(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        30, 194, 86, 18, 177, 202, 203, 133, 7, 22, 133, 138, 102, 21, 107, 127, 141, 28, 166, 162,
        46, 101, 20, 119, 208, 32, 66, 141, 122, 70, 94, 79, 61, 251, 202, 97, 35, 129, 9, 73, 215,
        220, 88, 174, 113, 124, 103, 113, 254, 44, 96, 79, 250, 252, 33, 252, 12, 70, 43, 238, 90,
        32, 248, 119,
    ];

    // a random base
    let base = [
        236, 66, 86, 50, 109, 186, 105, 32, 149, 81, 7, 31, 151, 116, 93, 59, 84, 247, 239, 100,
        174, 245, 37, 174, 204, 240, 52, 226, 252, 105, 16, 88, 180, 24, 90, 223, 60, 226, 147,
        165, 255, 74, 249, 183, 1, 80, 160, 48, 111, 90, 220, 16, 237, 217, 251, 30, 208, 24, 127,
        170, 237, 244, 15, 89, 205, 254, 128, 49, 245, 233, 139, 239, 4, 126, 248, 212, 173, 103,
        222, 225, 118, 93, 219, 91, 26, 90, 142, 91, 210, 149, 227, 222, 255, 227, 182, 233, 176,
        83, 210, 143, 174, 92, 46, 19, 242, 117, 6, 125, 163, 93, 116, 63, 71, 236, 139, 17, 192,
        239, 106, 133, 145, 158, 46, 238, 107, 80, 54, 80, 231, 138, 236, 44, 55, 13, 193, 159,
        144, 85, 138, 204, 84, 126, 66, 40, 104, 232, 113, 216, 165, 184, 198, 20, 234, 225, 170,
        174, 90, 101, 253, 231, 80, 252, 28, 148, 89, 64, 65, 26, 143, 60, 158, 116, 12, 14, 46,
        210, 99, 233, 187, 212, 44, 36, 47, 227, 123, 195, 45, 115, 12, 123, 16, 164, 92, 52, 229,
        65, 127, 114, 213, 116, 210, 2, 149, 144, 217, 131, 146, 67, 66, 91, 199, 46, 58, 5, 185,
        247, 73, 170, 6, 45, 109, 0, 191, 233, 95, 239, 241, 30, 61, 119, 54, 4, 164, 214, 202,
        251, 139, 28, 22, 219, 100, 233, 195, 151, 237, 183, 41, 153, 42, 82, 208, 222, 21, 160,
        100, 100,
    ];

    // a random modulus
    let modulus = [
        155, 66, 179, 54, 45, 180, 207, 15, 110, 66, 217, 170, 218, 229, 14, 147, 163, 227, 26, 27,
        56, 162, 176, 213, 136, 239, 229, 242, 214, 53, 97, 19, 91, 195, 133, 126, 130, 1, 54, 143,
        78, 210, 176, 236, 152, 95, 92, 140, 158, 72, 151, 225, 83, 120, 44, 192, 72, 12, 100, 19,
        76, 249, 175, 180, 3, 217, 241, 47, 99, 8, 101, 17, 7, 154, 235, 191, 239, 243, 156, 137,
        147, 6, 248, 70, 44, 52, 4, 159, 137, 14, 79, 178, 247, 112, 241, 56, 240, 45, 22, 250, 99,
        99, 79, 10, 147, 188, 219, 89, 129, 60, 204, 149, 6, 112, 52, 85, 204, 62, 164, 85, 59,
        200, 11, 239, 196, 157, 53, 128, 223, 221, 90, 234, 112, 74, 195, 52, 133, 189, 35, 110,
        66, 222, 150, 19, 121, 107, 23, 171, 46, 167, 10, 253, 119, 247, 214, 175, 239, 40, 45, 24,
        115, 2, 150, 243, 44, 187, 160, 142, 68, 56, 172, 77, 143, 142, 53, 216, 228, 216, 239,
        176, 186, 96, 11, 147, 160, 127, 107, 192, 246, 173, 95, 144, 190, 167, 93, 172, 81, 89,
        163, 86, 111, 48, 30, 172, 32, 33, 34, 224, 191, 214, 244, 161, 233, 222, 113, 112, 76,
        163, 71, 99, 138, 92, 127, 203, 253, 201, 164, 231, 61, 59, 98, 165, 238, 23, 196, 10, 177,
        253, 110, 149, 31, 57, 212, 43, 16, 24, 241, 163, 72, 81, 140, 115, 37, 155, 94, 39,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_1024_bits_mod_2048_bits_odd(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        107, 92, 159, 59, 101, 117, 205, 228, 222, 58, 188, 58, 254, 101, 230, 53, 203, 200, 138,
        56, 160, 233, 81, 218, 113, 119, 10, 214, 68, 109, 113, 15, 146, 191, 225, 80, 22, 199,
        119, 236, 23, 159, 148, 40, 113, 28, 75, 45, 15, 54, 5, 64, 103, 55, 1, 220, 236, 41, 218,
        41, 93, 6, 3, 106, 235, 31, 22, 73, 243, 113, 171, 111, 20, 237, 200, 8, 99, 252, 202, 99,
        122, 242, 84, 180, 8, 58, 3, 129, 145, 62, 179, 78, 199, 35, 212, 16, 3, 55, 9, 197, 217,
        30, 42, 67, 220, 121, 193, 16, 15, 170, 116, 65, 157, 109, 34, 211, 41, 116, 161, 40, 77,
        223, 200, 240, 31, 17, 141, 189,
    ];

    // a random base
    let base = [
        236, 66, 86, 50, 109, 186, 105, 32, 149, 81, 7, 31, 151, 116, 93, 59, 84, 247, 239, 100,
        174, 245, 37, 174, 204, 240, 52, 226, 252, 105, 16, 88, 180, 24, 90, 223, 60, 226, 147,
        165, 255, 74, 249, 183, 1, 80, 160, 48, 111, 90, 220, 16, 237, 217, 251, 30, 208, 24, 127,
        170, 237, 244, 15, 89, 205, 254, 128, 49, 245, 233, 139, 239, 4, 126, 248, 212, 173, 103,
        222, 225, 118, 93, 219, 91, 26, 90, 142, 91, 210, 149, 227, 222, 255, 227, 182, 233, 176,
        83, 210, 143, 174, 92, 46, 19, 242, 117, 6, 125, 163, 93, 116, 63, 71, 236, 139, 17, 192,
        239, 106, 133, 145, 158, 46, 238, 107, 80, 54, 80, 231, 138, 236, 44, 55, 13, 193, 159,
        144, 85, 138, 204, 84, 126, 66, 40, 104, 232, 113, 216, 165, 184, 198, 20, 234, 225, 170,
        174, 90, 101, 253, 231, 80, 252, 28, 148, 89, 64, 65, 26, 143, 60, 158, 116, 12, 14, 46,
        210, 99, 233, 187, 212, 44, 36, 47, 227, 123, 195, 45, 115, 12, 123, 16, 164, 92, 52, 229,
        65, 127, 114, 213, 116, 210, 2, 149, 144, 217, 131, 146, 67, 66, 91, 199, 46, 58, 5, 185,
        247, 73, 170, 6, 45, 109, 0, 191, 233, 95, 239, 241, 30, 61, 119, 54, 4, 164, 214, 202,
        251, 139, 28, 22, 219, 100, 233, 195, 151, 237, 183, 41, 153, 42, 82, 208, 222, 21, 160,
        100, 100,
    ];

    // a random modulus
    let modulus = [
        155, 66, 179, 54, 45, 180, 207, 15, 110, 66, 217, 170, 218, 229, 14, 147, 163, 227, 26, 27,
        56, 162, 176, 213, 136, 239, 229, 242, 214, 53, 97, 19, 91, 195, 133, 126, 130, 1, 54, 143,
        78, 210, 176, 236, 152, 95, 92, 140, 158, 72, 151, 225, 83, 120, 44, 192, 72, 12, 100, 19,
        76, 249, 175, 180, 3, 217, 241, 47, 99, 8, 101, 17, 7, 154, 235, 191, 239, 243, 156, 137,
        147, 6, 248, 70, 44, 52, 4, 159, 137, 14, 79, 178, 247, 112, 241, 56, 240, 45, 22, 250, 99,
        99, 79, 10, 147, 188, 219, 89, 129, 60, 204, 149, 6, 112, 52, 85, 204, 62, 164, 85, 59,
        200, 11, 239, 196, 157, 53, 128, 223, 221, 90, 234, 112, 74, 195, 52, 133, 189, 35, 110,
        66, 222, 150, 19, 121, 107, 23, 171, 46, 167, 10, 253, 119, 247, 214, 175, 239, 40, 45, 24,
        115, 2, 150, 243, 44, 187, 160, 142, 68, 56, 172, 77, 143, 142, 53, 216, 228, 216, 239,
        176, 186, 96, 11, 147, 160, 127, 107, 192, 246, 173, 95, 144, 190, 167, 93, 172, 81, 89,
        163, 86, 111, 48, 30, 172, 32, 33, 34, 224, 191, 214, 244, 161, 233, 222, 113, 112, 76,
        163, 71, 99, 138, 92, 127, 203, 253, 201, 164, 231, 61, 59, 98, 165, 238, 23, 196, 10, 177,
        253, 110, 149, 31, 57, 212, 43, 16, 24, 241, 163, 72, 81, 140, 115, 37, 155, 94, 39,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_2048_bits_mod_2048_bits_odd(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        234, 85, 222, 102, 95, 165, 186, 221, 192, 109, 51, 204, 69, 225, 36, 118, 52, 132, 39,
        190, 10, 8, 82, 87, 149, 233, 35, 50, 36, 102, 243, 84, 50, 26, 54, 64, 38, 68, 154, 97,
        100, 221, 4, 81, 15, 47, 229, 100, 163, 68, 127, 163, 138, 24, 244, 125, 166, 116, 68, 126,
        201, 43, 192, 13, 236, 182, 213, 203, 235, 20, 2, 81, 168, 251, 87, 97, 69, 159, 138, 203,
        53, 43, 243, 14, 212, 5, 0, 229, 80, 72, 147, 130, 47, 13, 236, 180, 25, 100, 178, 148,
        171, 231, 252, 68, 57, 79, 14, 185, 155, 82, 103, 1, 98, 32, 204, 127, 242, 86, 25, 37, 19,
        240, 21, 64, 3, 160, 100, 76, 72, 220, 67, 123, 123, 139, 206, 75, 33, 177, 61, 129, 69,
        57, 186, 166, 3, 94, 162, 249, 22, 89, 245, 106, 180, 116, 222, 177, 231, 57, 73, 6, 217,
        252, 58, 212, 233, 219, 42, 144, 68, 92, 168, 147, 116, 82, 211, 224, 214, 156, 1, 52, 112,
        114, 193, 158, 137, 195, 46, 73, 179, 7, 229, 69, 151, 34, 78, 108, 138, 207, 37, 178, 41,
        142, 41, 163, 144, 206, 181, 71, 13, 195, 186, 74, 56, 93, 151, 97, 73, 57, 114, 198, 203,
        216, 182, 98, 88, 9, 68, 211, 235, 78, 105, 182, 245, 96, 5, 119, 229, 2, 50, 187, 159,
        131, 24, 4, 154, 234, 61, 95, 45, 102, 134, 106, 208, 39, 202, 165,
    ];

    // a random base
    let base = [
        236, 66, 86, 50, 109, 186, 105, 32, 149, 81, 7, 31, 151, 116, 93, 59, 84, 247, 239, 100,
        174, 245, 37, 174, 204, 240, 52, 226, 252, 105, 16, 88, 180, 24, 90, 223, 60, 226, 147,
        165, 255, 74, 249, 183, 1, 80, 160, 48, 111, 90, 220, 16, 237, 217, 251, 30, 208, 24, 127,
        170, 237, 244, 15, 89, 205, 254, 128, 49, 245, 233, 139, 239, 4, 126, 248, 212, 173, 103,
        222, 225, 118, 93, 219, 91, 26, 90, 142, 91, 210, 149, 227, 222, 255, 227, 182, 233, 176,
        83, 210, 143, 174, 92, 46, 19, 242, 117, 6, 125, 163, 93, 116, 63, 71, 236, 139, 17, 192,
        239, 106, 133, 145, 158, 46, 238, 107, 80, 54, 80, 231, 138, 236, 44, 55, 13, 193, 159,
        144, 85, 138, 204, 84, 126, 66, 40, 104, 232, 113, 216, 165, 184, 198, 20, 234, 225, 170,
        174, 90, 101, 253, 231, 80, 252, 28, 148, 89, 64, 65, 26, 143, 60, 158, 116, 12, 14, 46,
        210, 99, 233, 187, 212, 44, 36, 47, 227, 123, 195, 45, 115, 12, 123, 16, 164, 92, 52, 229,
        65, 127, 114, 213, 116, 210, 2, 149, 144, 217, 131, 146, 67, 66, 91, 199, 46, 58, 5, 185,
        247, 73, 170, 6, 45, 109, 0, 191, 233, 95, 239, 241, 30, 61, 119, 54, 4, 164, 214, 202,
        251, 139, 28, 22, 219, 100, 233, 195, 151, 237, 183, 41, 153, 42, 82, 208, 222, 21, 160,
        100, 100,
    ];

    // a random modulus
    let modulus = [
        155, 66, 179, 54, 45, 180, 207, 15, 110, 66, 217, 170, 218, 229, 14, 147, 163, 227, 26, 27,
        56, 162, 176, 213, 136, 239, 229, 242, 214, 53, 97, 19, 91, 195, 133, 126, 130, 1, 54, 143,
        78, 210, 176, 236, 152, 95, 92, 140, 158, 72, 151, 225, 83, 120, 44, 192, 72, 12, 100, 19,
        76, 249, 175, 180, 3, 217, 241, 47, 99, 8, 101, 17, 7, 154, 235, 191, 239, 243, 156, 137,
        147, 6, 248, 70, 44, 52, 4, 159, 137, 14, 79, 178, 247, 112, 241, 56, 240, 45, 22, 250, 99,
        99, 79, 10, 147, 188, 219, 89, 129, 60, 204, 149, 6, 112, 52, 85, 204, 62, 164, 85, 59,
        200, 11, 239, 196, 157, 53, 128, 223, 221, 90, 234, 112, 74, 195, 52, 133, 189, 35, 110,
        66, 222, 150, 19, 121, 107, 23, 171, 46, 167, 10, 253, 119, 247, 214, 175, 239, 40, 45, 24,
        115, 2, 150, 243, 44, 187, 160, 142, 68, 56, 172, 77, 143, 142, 53, 216, 228, 216, 239,
        176, 186, 96, 11, 147, 160, 127, 107, 192, 246, 173, 95, 144, 190, 167, 93, 172, 81, 89,
        163, 86, 111, 48, 30, 172, 32, 33, 34, 224, 191, 214, 244, 161, 233, 222, 113, 112, 76,
        163, 71, 99, 138, 92, 127, 203, 253, 201, 164, 231, 61, 59, 98, 165, 238, 23, 196, 10, 177,
        253, 110, 149, 31, 57, 212, 43, 16, 24, 241, 163, 72, 81, 140, 115, 37, 155, 94, 39,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_0512_bits_mod_4096_bits_odd(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        30, 194, 86, 18, 177, 202, 203, 133, 7, 22, 133, 138, 102, 21, 107, 127, 141, 28, 166, 162,
        46, 101, 20, 119, 208, 32, 66, 141, 122, 70, 94, 79, 61, 251, 202, 97, 35, 129, 9, 73, 215,
        220, 88, 174, 113, 124, 103, 113, 254, 44, 96, 79, 250, 252, 33, 252, 12, 70, 43, 238, 90,
        32, 248, 119,
    ];

    // a random base
    let base = [
        167, 62, 26, 222, 69, 55, 136, 13, 139, 152, 251, 52, 101, 99, 181, 160, 55, 137, 73, 135,
        48, 71, 232, 168, 183, 14, 161, 145, 163, 22, 31, 14, 111, 65, 93, 52, 253, 42, 115, 167,
        211, 27, 143, 95, 20, 20, 179, 231, 241, 169, 110, 220, 42, 178, 222, 110, 97, 186, 76,
        210, 43, 205, 210, 166, 58, 173, 28, 10, 192, 102, 35, 122, 240, 244, 186, 197, 211, 39,
        230, 59, 140, 42, 221, 90, 81, 107, 88, 210, 1, 222, 215, 116, 7, 190, 251, 226, 87, 131,
        53, 137, 108, 245, 194, 62, 173, 26, 124, 253, 192, 10, 143, 10, 56, 237, 191, 216, 13, 6,
        22, 37, 130, 226, 241, 58, 157, 230, 247, 125, 170, 124, 167, 184, 148, 204, 247, 5, 181,
        73, 81, 195, 64, 72, 33, 187, 100, 173, 35, 86, 99, 228, 124, 83, 64, 1, 176, 178, 194,
        236, 110, 202, 70, 204, 59, 147, 232, 246, 152, 93, 124, 222, 192, 136, 149, 14, 36, 225,
        122, 228, 16, 215, 49, 80, 30, 230, 235, 158, 29, 86, 209, 67, 172, 163, 79, 22, 167, 113,
        154, 5, 157, 187, 237, 144, 76, 77, 45, 177, 15, 21, 19, 7, 181, 125, 232, 9, 138, 111,
        163, 207, 196, 197, 162, 67, 66, 81, 36, 3, 68, 68, 36, 213, 243, 42, 41, 12, 173, 71, 197,
        51, 217, 122, 98, 142, 79, 177, 34, 6, 160, 189, 158, 248, 115, 238, 226, 10, 71, 179, 92,
        87, 14, 100, 13, 220, 105, 238, 238, 0, 229, 100, 96, 29, 18, 155, 53, 247, 10, 146, 252,
        2, 219, 96, 206, 225, 171, 59, 135, 177, 90, 174, 245, 3, 222, 95, 35, 1, 31, 107, 53, 99,
        117, 93, 207, 148, 62, 12, 140, 211, 0, 122, 39, 66, 21, 71, 197, 137, 97, 142, 240, 61,
        69, 25, 121, 165, 53, 40, 218, 248, 112, 213, 8, 91, 217, 236, 173, 227, 103, 250, 133,
        143, 0, 227, 117, 159, 69, 8, 52, 147, 0, 175, 255, 76, 129, 47, 230, 234, 58, 233, 128, 2,
        184, 190, 104, 241, 69, 95, 74, 176, 112, 75, 236, 81, 100, 222, 111, 171, 126, 16, 55, 26,
        1, 63, 15, 24, 147, 176, 50, 20, 23, 176, 176, 182, 72, 208, 105, 138, 160, 243, 113, 123,
        106, 145, 39, 176, 0, 192, 137, 187, 182, 9, 5, 16, 72, 2, 121, 54, 224, 231, 208, 116,
        144, 55, 134, 20, 235, 123, 80, 105, 37, 196, 166, 51, 26, 151, 84, 52, 120, 162, 194, 97,
        212, 242, 153, 205, 112, 158, 151, 244, 242, 79, 219, 128, 99, 246, 32, 38, 132, 34, 131,
        28, 117, 162, 68, 107, 21, 25, 68, 66, 203, 7, 179, 46, 119, 8, 31, 36, 239, 179, 140, 51,
        221, 151, 200, 68, 183, 132, 36, 171, 247, 80, 191, 125, 22, 210, 204, 83, 200, 103, 97,
        184, 61, 225, 127, 73, 104, 159, 172, 185, 202, 116, 154, 28, 150, 6,
    ];

    // a random modulus
    let modulus = [
        186, 147, 169, 139, 184, 41, 41, 80, 108, 1, 29, 237, 140, 15, 147, 98, 8, 192, 145, 125,
        112, 68, 139, 38, 31, 131, 18, 213, 128, 28, 239, 126, 50, 54, 178, 48, 110, 200, 255, 10,
        206, 143, 168, 51, 43, 18, 24, 80, 166, 137, 197, 251, 215, 163, 248, 230, 16, 81, 164, 40,
        118, 207, 12, 149, 30, 55, 8, 135, 56, 9, 118, 228, 18, 190, 224, 181, 78, 70, 141, 109,
        114, 123, 200, 223, 241, 143, 249, 55, 171, 184, 16, 113, 63, 95, 194, 141, 196, 90, 118,
        203, 194, 73, 229, 187, 126, 241, 137, 9, 58, 68, 205, 95, 6, 254, 191, 224, 240, 85, 254,
        200, 6, 210, 24, 1, 32, 131, 185, 210, 109, 71, 116, 58, 213, 187, 115, 210, 107, 65, 116,
        172, 131, 209, 45, 65, 144, 7, 82, 199, 187, 97, 202, 5, 47, 159, 80, 235, 221, 89, 102,
        236, 123, 243, 25, 133, 28, 19, 140, 117, 77, 214, 127, 208, 94, 18, 24, 166, 38, 101, 165,
        144, 183, 109, 78, 100, 128, 67, 215, 253, 248, 244, 242, 182, 219, 26, 135, 24, 192, 149,
        247, 65, 206, 203, 28, 66, 205, 95, 65, 190, 163, 68, 24, 142, 228, 152, 231, 136, 169,
        197, 37, 124, 201, 235, 127, 51, 7, 197, 221, 161, 243, 212, 204, 128, 76, 52, 58, 3, 60,
        158, 162, 140, 51, 233, 167, 231, 190, 227, 111, 82, 210, 48, 79, 232, 80, 202, 114, 161,
        230, 250, 197, 122, 65, 90, 91, 110, 7, 106, 55, 170, 156, 118, 133, 101, 248, 234, 156,
        106, 228, 193, 33, 74, 82, 188, 205, 42, 2, 187, 75, 213, 33, 67, 206, 15, 163, 25, 72, 32,
        192, 92, 239, 40, 15, 116, 155, 120, 213, 50, 142, 47, 161, 30, 44, 213, 100, 217, 213,
        159, 190, 230, 163, 63, 152, 119, 190, 42, 154, 81, 39, 70, 179, 101, 169, 74, 37, 143, 49,
        134, 99, 130, 101, 87, 111, 240, 38, 240, 164, 134, 162, 124, 109, 161, 181, 192, 42, 57,
        240, 133, 18, 215, 175, 164, 45, 150, 161, 30, 104, 217, 68, 3, 5, 57, 170, 148, 24, 116,
        96, 172, 23, 139, 23, 230, 127, 182, 178, 167, 100, 46, 182, 190, 176, 4, 245, 103, 158,
        50, 96, 24, 27, 107, 135, 46, 122, 17, 225, 183, 0, 58, 3, 243, 168, 84, 64, 34, 29, 39,
        17, 227, 240, 30, 150, 28, 76, 11, 86, 53, 143, 25, 164, 215, 164, 70, 71, 232, 195, 95,
        221, 117, 135, 3, 241, 40, 207, 123, 122, 148, 118, 56, 65, 218, 8, 88, 124, 219, 91, 236,
        194, 245, 37, 246, 54, 202, 36, 151, 160, 40, 75, 92, 175, 241, 161, 113, 53, 220, 67, 249,
        7, 118, 77, 214, 122, 148, 44, 164, 249, 239, 57, 102, 45, 136, 93, 4, 76, 51, 186, 156,
        101, 82, 217, 116, 211, 255, 92, 183, 101, 50, 73, 38, 79, 219, 63, 140, 174, 217,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_1024_bits_mod_4096_bits_odd(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        107, 92, 159, 59, 101, 117, 205, 228, 222, 58, 188, 58, 254, 101, 230, 53, 203, 200, 138,
        56, 160, 233, 81, 218, 113, 119, 10, 214, 68, 109, 113, 15, 146, 191, 225, 80, 22, 199,
        119, 236, 23, 159, 148, 40, 113, 28, 75, 45, 15, 54, 5, 64, 103, 55, 1, 220, 236, 41, 218,
        41, 93, 6, 3, 106, 235, 31, 22, 73, 243, 113, 171, 111, 20, 237, 200, 8, 99, 252, 202, 99,
        122, 242, 84, 180, 8, 58, 3, 129, 145, 62, 179, 78, 199, 35, 212, 16, 3, 55, 9, 197, 217,
        30, 42, 67, 220, 121, 193, 16, 15, 170, 116, 65, 157, 109, 34, 211, 41, 116, 161, 40, 77,
        223, 200, 240, 31, 17, 141, 189,
    ];

    // a random base
    let base = [
        167, 62, 26, 222, 69, 55, 136, 13, 139, 152, 251, 52, 101, 99, 181, 160, 55, 137, 73, 135,
        48, 71, 232, 168, 183, 14, 161, 145, 163, 22, 31, 14, 111, 65, 93, 52, 253, 42, 115, 167,
        211, 27, 143, 95, 20, 20, 179, 231, 241, 169, 110, 220, 42, 178, 222, 110, 97, 186, 76,
        210, 43, 205, 210, 166, 58, 173, 28, 10, 192, 102, 35, 122, 240, 244, 186, 197, 211, 39,
        230, 59, 140, 42, 221, 90, 81, 107, 88, 210, 1, 222, 215, 116, 7, 190, 251, 226, 87, 131,
        53, 137, 108, 245, 194, 62, 173, 26, 124, 253, 192, 10, 143, 10, 56, 237, 191, 216, 13, 6,
        22, 37, 130, 226, 241, 58, 157, 230, 247, 125, 170, 124, 167, 184, 148, 204, 247, 5, 181,
        73, 81, 195, 64, 72, 33, 187, 100, 173, 35, 86, 99, 228, 124, 83, 64, 1, 176, 178, 194,
        236, 110, 202, 70, 204, 59, 147, 232, 246, 152, 93, 124, 222, 192, 136, 149, 14, 36, 225,
        122, 228, 16, 215, 49, 80, 30, 230, 235, 158, 29, 86, 209, 67, 172, 163, 79, 22, 167, 113,
        154, 5, 157, 187, 237, 144, 76, 77, 45, 177, 15, 21, 19, 7, 181, 125, 232, 9, 138, 111,
        163, 207, 196, 197, 162, 67, 66, 81, 36, 3, 68, 68, 36, 213, 243, 42, 41, 12, 173, 71, 197,
        51, 217, 122, 98, 142, 79, 177, 34, 6, 160, 189, 158, 248, 115, 238, 226, 10, 71, 179, 92,
        87, 14, 100, 13, 220, 105, 238, 238, 0, 229, 100, 96, 29, 18, 155, 53, 247, 10, 146, 252,
        2, 219, 96, 206, 225, 171, 59, 135, 177, 90, 174, 245, 3, 222, 95, 35, 1, 31, 107, 53, 99,
        117, 93, 207, 148, 62, 12, 140, 211, 0, 122, 39, 66, 21, 71, 197, 137, 97, 142, 240, 61,
        69, 25, 121, 165, 53, 40, 218, 248, 112, 213, 8, 91, 217, 236, 173, 227, 103, 250, 133,
        143, 0, 227, 117, 159, 69, 8, 52, 147, 0, 175, 255, 76, 129, 47, 230, 234, 58, 233, 128, 2,
        184, 190, 104, 241, 69, 95, 74, 176, 112, 75, 236, 81, 100, 222, 111, 171, 126, 16, 55, 26,
        1, 63, 15, 24, 147, 176, 50, 20, 23, 176, 176, 182, 72, 208, 105, 138, 160, 243, 113, 123,
        106, 145, 39, 176, 0, 192, 137, 187, 182, 9, 5, 16, 72, 2, 121, 54, 224, 231, 208, 116,
        144, 55, 134, 20, 235, 123, 80, 105, 37, 196, 166, 51, 26, 151, 84, 52, 120, 162, 194, 97,
        212, 242, 153, 205, 112, 158, 151, 244, 242, 79, 219, 128, 99, 246, 32, 38, 132, 34, 131,
        28, 117, 162, 68, 107, 21, 25, 68, 66, 203, 7, 179, 46, 119, 8, 31, 36, 239, 179, 140, 51,
        221, 151, 200, 68, 183, 132, 36, 171, 247, 80, 191, 125, 22, 210, 204, 83, 200, 103, 97,
        184, 61, 225, 127, 73, 104, 159, 172, 185, 202, 116, 154, 28, 150, 6,
    ];

    // a random modulus
    let modulus = [
        186, 147, 169, 139, 184, 41, 41, 80, 108, 1, 29, 237, 140, 15, 147, 98, 8, 192, 145, 125,
        112, 68, 139, 38, 31, 131, 18, 213, 128, 28, 239, 126, 50, 54, 178, 48, 110, 200, 255, 10,
        206, 143, 168, 51, 43, 18, 24, 80, 166, 137, 197, 251, 215, 163, 248, 230, 16, 81, 164, 40,
        118, 207, 12, 149, 30, 55, 8, 135, 56, 9, 118, 228, 18, 190, 224, 181, 78, 70, 141, 109,
        114, 123, 200, 223, 241, 143, 249, 55, 171, 184, 16, 113, 63, 95, 194, 141, 196, 90, 118,
        203, 194, 73, 229, 187, 126, 241, 137, 9, 58, 68, 205, 95, 6, 254, 191, 224, 240, 85, 254,
        200, 6, 210, 24, 1, 32, 131, 185, 210, 109, 71, 116, 58, 213, 187, 115, 210, 107, 65, 116,
        172, 131, 209, 45, 65, 144, 7, 82, 199, 187, 97, 202, 5, 47, 159, 80, 235, 221, 89, 102,
        236, 123, 243, 25, 133, 28, 19, 140, 117, 77, 214, 127, 208, 94, 18, 24, 166, 38, 101, 165,
        144, 183, 109, 78, 100, 128, 67, 215, 253, 248, 244, 242, 182, 219, 26, 135, 24, 192, 149,
        247, 65, 206, 203, 28, 66, 205, 95, 65, 190, 163, 68, 24, 142, 228, 152, 231, 136, 169,
        197, 37, 124, 201, 235, 127, 51, 7, 197, 221, 161, 243, 212, 204, 128, 76, 52, 58, 3, 60,
        158, 162, 140, 51, 233, 167, 231, 190, 227, 111, 82, 210, 48, 79, 232, 80, 202, 114, 161,
        230, 250, 197, 122, 65, 90, 91, 110, 7, 106, 55, 170, 156, 118, 133, 101, 248, 234, 156,
        106, 228, 193, 33, 74, 82, 188, 205, 42, 2, 187, 75, 213, 33, 67, 206, 15, 163, 25, 72, 32,
        192, 92, 239, 40, 15, 116, 155, 120, 213, 50, 142, 47, 161, 30, 44, 213, 100, 217, 213,
        159, 190, 230, 163, 63, 152, 119, 190, 42, 154, 81, 39, 70, 179, 101, 169, 74, 37, 143, 49,
        134, 99, 130, 101, 87, 111, 240, 38, 240, 164, 134, 162, 124, 109, 161, 181, 192, 42, 57,
        240, 133, 18, 215, 175, 164, 45, 150, 161, 30, 104, 217, 68, 3, 5, 57, 170, 148, 24, 116,
        96, 172, 23, 139, 23, 230, 127, 182, 178, 167, 100, 46, 182, 190, 176, 4, 245, 103, 158,
        50, 96, 24, 27, 107, 135, 46, 122, 17, 225, 183, 0, 58, 3, 243, 168, 84, 64, 34, 29, 39,
        17, 227, 240, 30, 150, 28, 76, 11, 86, 53, 143, 25, 164, 215, 164, 70, 71, 232, 195, 95,
        221, 117, 135, 3, 241, 40, 207, 123, 122, 148, 118, 56, 65, 218, 8, 88, 124, 219, 91, 236,
        194, 245, 37, 246, 54, 202, 36, 151, 160, 40, 75, 92, 175, 241, 161, 113, 53, 220, 67, 249,
        7, 118, 77, 214, 122, 148, 44, 164, 249, 239, 57, 102, 45, 136, 93, 4, 76, 51, 186, 156,
        101, 82, 217, 116, 211, 255, 92, 183, 101, 50, 73, 38, 79, 219, 63, 140, 174, 217,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_2048_bits_mod_4096_bits_odd(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        234, 85, 222, 102, 95, 165, 186, 221, 192, 109, 51, 204, 69, 225, 36, 118, 52, 132, 39,
        190, 10, 8, 82, 87, 149, 233, 35, 50, 36, 102, 243, 84, 50, 26, 54, 64, 38, 68, 154, 97,
        100, 221, 4, 81, 15, 47, 229, 100, 163, 68, 127, 163, 138, 24, 244, 125, 166, 116, 68, 126,
        201, 43, 192, 13, 236, 182, 213, 203, 235, 20, 2, 81, 168, 251, 87, 97, 69, 159, 138, 203,
        53, 43, 243, 14, 212, 5, 0, 229, 80, 72, 147, 130, 47, 13, 236, 180, 25, 100, 178, 148,
        171, 231, 252, 68, 57, 79, 14, 185, 155, 82, 103, 1, 98, 32, 204, 127, 242, 86, 25, 37, 19,
        240, 21, 64, 3, 160, 100, 76, 72, 220, 67, 123, 123, 139, 206, 75, 33, 177, 61, 129, 69,
        57, 186, 166, 3, 94, 162, 249, 22, 89, 245, 106, 180, 116, 222, 177, 231, 57, 73, 6, 217,
        252, 58, 212, 233, 219, 42, 144, 68, 92, 168, 147, 116, 82, 211, 224, 214, 156, 1, 52, 112,
        114, 193, 158, 137, 195, 46, 73, 179, 7, 229, 69, 151, 34, 78, 108, 138, 207, 37, 178, 41,
        142, 41, 163, 144, 206, 181, 71, 13, 195, 186, 74, 56, 93, 151, 97, 73, 57, 114, 198, 203,
        216, 182, 98, 88, 9, 68, 211, 235, 78, 105, 182, 245, 96, 5, 119, 229, 2, 50, 187, 159,
        131, 24, 4, 154, 234, 61, 95, 45, 102, 134, 106, 208, 39, 202, 165,
    ];

    // a random base
    let base = [
        167, 62, 26, 222, 69, 55, 136, 13, 139, 152, 251, 52, 101, 99, 181, 160, 55, 137, 73, 135,
        48, 71, 232, 168, 183, 14, 161, 145, 163, 22, 31, 14, 111, 65, 93, 52, 253, 42, 115, 167,
        211, 27, 143, 95, 20, 20, 179, 231, 241, 169, 110, 220, 42, 178, 222, 110, 97, 186, 76,
        210, 43, 205, 210, 166, 58, 173, 28, 10, 192, 102, 35, 122, 240, 244, 186, 197, 211, 39,
        230, 59, 140, 42, 221, 90, 81, 107, 88, 210, 1, 222, 215, 116, 7, 190, 251, 226, 87, 131,
        53, 137, 108, 245, 194, 62, 173, 26, 124, 253, 192, 10, 143, 10, 56, 237, 191, 216, 13, 6,
        22, 37, 130, 226, 241, 58, 157, 230, 247, 125, 170, 124, 167, 184, 148, 204, 247, 5, 181,
        73, 81, 195, 64, 72, 33, 187, 100, 173, 35, 86, 99, 228, 124, 83, 64, 1, 176, 178, 194,
        236, 110, 202, 70, 204, 59, 147, 232, 246, 152, 93, 124, 222, 192, 136, 149, 14, 36, 225,
        122, 228, 16, 215, 49, 80, 30, 230, 235, 158, 29, 86, 209, 67, 172, 163, 79, 22, 167, 113,
        154, 5, 157, 187, 237, 144, 76, 77, 45, 177, 15, 21, 19, 7, 181, 125, 232, 9, 138, 111,
        163, 207, 196, 197, 162, 67, 66, 81, 36, 3, 68, 68, 36, 213, 243, 42, 41, 12, 173, 71, 197,
        51, 217, 122, 98, 142, 79, 177, 34, 6, 160, 189, 158, 248, 115, 238, 226, 10, 71, 179, 92,
        87, 14, 100, 13, 220, 105, 238, 238, 0, 229, 100, 96, 29, 18, 155, 53, 247, 10, 146, 252,
        2, 219, 96, 206, 225, 171, 59, 135, 177, 90, 174, 245, 3, 222, 95, 35, 1, 31, 107, 53, 99,
        117, 93, 207, 148, 62, 12, 140, 211, 0, 122, 39, 66, 21, 71, 197, 137, 97, 142, 240, 61,
        69, 25, 121, 165, 53, 40, 218, 248, 112, 213, 8, 91, 217, 236, 173, 227, 103, 250, 133,
        143, 0, 227, 117, 159, 69, 8, 52, 147, 0, 175, 255, 76, 129, 47, 230, 234, 58, 233, 128, 2,
        184, 190, 104, 241, 69, 95, 74, 176, 112, 75, 236, 81, 100, 222, 111, 171, 126, 16, 55, 26,
        1, 63, 15, 24, 147, 176, 50, 20, 23, 176, 176, 182, 72, 208, 105, 138, 160, 243, 113, 123,
        106, 145, 39, 176, 0, 192, 137, 187, 182, 9, 5, 16, 72, 2, 121, 54, 224, 231, 208, 116,
        144, 55, 134, 20, 235, 123, 80, 105, 37, 196, 166, 51, 26, 151, 84, 52, 120, 162, 194, 97,
        212, 242, 153, 205, 112, 158, 151, 244, 242, 79, 219, 128, 99, 246, 32, 38, 132, 34, 131,
        28, 117, 162, 68, 107, 21, 25, 68, 66, 203, 7, 179, 46, 119, 8, 31, 36, 239, 179, 140, 51,
        221, 151, 200, 68, 183, 132, 36, 171, 247, 80, 191, 125, 22, 210, 204, 83, 200, 103, 97,
        184, 61, 225, 127, 73, 104, 159, 172, 185, 202, 116, 154, 28, 150, 6,
    ];

    // a random modulus
    let modulus = [
        186, 147, 169, 139, 184, 41, 41, 80, 108, 1, 29, 237, 140, 15, 147, 98, 8, 192, 145, 125,
        112, 68, 139, 38, 31, 131, 18, 213, 128, 28, 239, 126, 50, 54, 178, 48, 110, 200, 255, 10,
        206, 143, 168, 51, 43, 18, 24, 80, 166, 137, 197, 251, 215, 163, 248, 230, 16, 81, 164, 40,
        118, 207, 12, 149, 30, 55, 8, 135, 56, 9, 118, 228, 18, 190, 224, 181, 78, 70, 141, 109,
        114, 123, 200, 223, 241, 143, 249, 55, 171, 184, 16, 113, 63, 95, 194, 141, 196, 90, 118,
        203, 194, 73, 229, 187, 126, 241, 137, 9, 58, 68, 205, 95, 6, 254, 191, 224, 240, 85, 254,
        200, 6, 210, 24, 1, 32, 131, 185, 210, 109, 71, 116, 58, 213, 187, 115, 210, 107, 65, 116,
        172, 131, 209, 45, 65, 144, 7, 82, 199, 187, 97, 202, 5, 47, 159, 80, 235, 221, 89, 102,
        236, 123, 243, 25, 133, 28, 19, 140, 117, 77, 214, 127, 208, 94, 18, 24, 166, 38, 101, 165,
        144, 183, 109, 78, 100, 128, 67, 215, 253, 248, 244, 242, 182, 219, 26, 135, 24, 192, 149,
        247, 65, 206, 203, 28, 66, 205, 95, 65, 190, 163, 68, 24, 142, 228, 152, 231, 136, 169,
        197, 37, 124, 201, 235, 127, 51, 7, 197, 221, 161, 243, 212, 204, 128, 76, 52, 58, 3, 60,
        158, 162, 140, 51, 233, 167, 231, 190, 227, 111, 82, 210, 48, 79, 232, 80, 202, 114, 161,
        230, 250, 197, 122, 65, 90, 91, 110, 7, 106, 55, 170, 156, 118, 133, 101, 248, 234, 156,
        106, 228, 193, 33, 74, 82, 188, 205, 42, 2, 187, 75, 213, 33, 67, 206, 15, 163, 25, 72, 32,
        192, 92, 239, 40, 15, 116, 155, 120, 213, 50, 142, 47, 161, 30, 44, 213, 100, 217, 213,
        159, 190, 230, 163, 63, 152, 119, 190, 42, 154, 81, 39, 70, 179, 101, 169, 74, 37, 143, 49,
        134, 99, 130, 101, 87, 111, 240, 38, 240, 164, 134, 162, 124, 109, 161, 181, 192, 42, 57,
        240, 133, 18, 215, 175, 164, 45, 150, 161, 30, 104, 217, 68, 3, 5, 57, 170, 148, 24, 116,
        96, 172, 23, 139, 23, 230, 127, 182, 178, 167, 100, 46, 182, 190, 176, 4, 245, 103, 158,
        50, 96, 24, 27, 107, 135, 46, 122, 17, 225, 183, 0, 58, 3, 243, 168, 84, 64, 34, 29, 39,
        17, 227, 240, 30, 150, 28, 76, 11, 86, 53, 143, 25, 164, 215, 164, 70, 71, 232, 195, 95,
        221, 117, 135, 3, 241, 40, 207, 123, 122, 148, 118, 56, 65, 218, 8, 88, 124, 219, 91, 236,
        194, 245, 37, 246, 54, 202, 36, 151, 160, 40, 75, 92, 175, 241, 161, 113, 53, 220, 67, 249,
        7, 118, 77, 214, 122, 148, 44, 164, 249, 239, 57, 102, 45, 136, 93, 4, 76, 51, 186, 156,
        101, 82, 217, 116, 211, 255, 92, 183, 101, 50, 73, 38, 79, 219, 63, 140, 174, 217,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_4096_bits_mod_4096_bits_odd(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        160, 136, 220, 27, 39, 209, 128, 184, 213, 36, 207, 49, 210, 27, 221, 106, 122, 123, 158,
        77, 226, 64, 168, 133, 129, 82, 217, 239, 209, 235, 234, 153, 175, 4, 59, 165, 18, 109, 42,
        228, 194, 227, 131, 189, 98, 154, 242, 164, 206, 128, 151, 139, 167, 129, 179, 1, 31, 20,
        62, 168, 118, 232, 139, 241, 174, 171, 180, 238, 21, 190, 206, 250, 115, 99, 66, 152, 176,
        110, 213, 251, 176, 158, 145, 38, 61, 121, 183, 157, 18, 8, 202, 154, 26, 198, 32, 252,
        213, 40, 31, 241, 234, 13, 97, 166, 12, 199, 215, 205, 64, 121, 192, 240, 168, 241, 224,
        86, 157, 194, 1, 3, 135, 99, 201, 95, 185, 193, 142, 218, 122, 250, 84, 90, 150, 146, 2,
        173, 55, 255, 166, 150, 196, 182, 97, 4, 161, 85, 162, 74, 230, 138, 154, 128, 100, 161,
        62, 19, 74, 36, 249, 111, 45, 13, 236, 140, 73, 123, 174, 114, 45, 133, 124, 150, 15, 148,
        24, 192, 29, 116, 90, 51, 215, 62, 176, 177, 23, 211, 67, 97, 19, 231, 83, 147, 140, 99,
        186, 233, 213, 100, 109, 27, 124, 193, 193, 2, 212, 2, 55, 104, 122, 170, 249, 228, 183,
        109, 141, 58, 63, 232, 54, 255, 218, 109, 149, 174, 74, 157, 63, 252, 249, 149, 173, 28,
        249, 12, 39, 64, 90, 179, 81, 210, 129, 14, 247, 248, 169, 243, 182, 74, 143, 236, 217,
        255, 201, 184, 228, 67, 254, 115, 187, 93, 171, 34, 62, 182, 218, 0, 237, 66, 224, 51, 124,
        75, 28, 149, 207, 3, 41, 2, 113, 49, 2, 147, 227, 101, 82, 13, 120, 75, 4, 67, 244, 28,
        123, 32, 81, 32, 224, 63, 51, 62, 121, 8, 62, 234, 179, 181, 91, 76, 123, 183, 247, 135,
        40, 212, 170, 236, 45, 164, 17, 239, 65, 202, 175, 31, 116, 141, 219, 209, 253, 40, 231,
        175, 115, 59, 199, 88, 149, 101, 10, 29, 25, 233, 171, 62, 132, 90, 175, 237, 237, 167,
        153, 5, 114, 103, 227, 79, 128, 89, 207, 142, 215, 104, 92, 236, 98, 61, 187, 233, 145,
        119, 209, 100, 4, 146, 11, 247, 106, 123, 73, 200, 120, 121, 161, 118, 214, 69, 17, 66, 29,
        170, 255, 100, 59, 53, 107, 10, 231, 72, 103, 11, 223, 36, 14, 243, 252, 181, 247, 80, 253,
        110, 84, 106, 132, 102, 231, 136, 96, 148, 246, 185, 249, 207, 49, 205, 171, 168, 150, 84,
        146, 144, 228, 95, 164, 160, 194, 124, 69, 58, 168, 59, 101, 32, 196, 167, 87, 108, 99,
        126, 157, 244, 227, 224, 29, 105, 73, 249, 45, 109, 62, 180, 234, 251, 53, 11, 72, 116, 46,
        18, 102, 59, 160, 138, 22, 117, 161, 104, 189, 21, 193, 31, 175, 178, 131, 47, 127, 146,
        209, 129, 96, 61, 43, 238, 88, 211, 79, 157, 85, 15, 57, 82, 54, 104, 104, 201, 171, 60,
        26, 1, 137, 90, 234, 249,
    ];

    // a random base
    let base = [
        167, 62, 26, 222, 69, 55, 136, 13, 139, 152, 251, 52, 101, 99, 181, 160, 55, 137, 73, 135,
        48, 71, 232, 168, 183, 14, 161, 145, 163, 22, 31, 14, 111, 65, 93, 52, 253, 42, 115, 167,
        211, 27, 143, 95, 20, 20, 179, 231, 241, 169, 110, 220, 42, 178, 222, 110, 97, 186, 76,
        210, 43, 205, 210, 166, 58, 173, 28, 10, 192, 102, 35, 122, 240, 244, 186, 197, 211, 39,
        230, 59, 140, 42, 221, 90, 81, 107, 88, 210, 1, 222, 215, 116, 7, 190, 251, 226, 87, 131,
        53, 137, 108, 245, 194, 62, 173, 26, 124, 253, 192, 10, 143, 10, 56, 237, 191, 216, 13, 6,
        22, 37, 130, 226, 241, 58, 157, 230, 247, 125, 170, 124, 167, 184, 148, 204, 247, 5, 181,
        73, 81, 195, 64, 72, 33, 187, 100, 173, 35, 86, 99, 228, 124, 83, 64, 1, 176, 178, 194,
        236, 110, 202, 70, 204, 59, 147, 232, 246, 152, 93, 124, 222, 192, 136, 149, 14, 36, 225,
        122, 228, 16, 215, 49, 80, 30, 230, 235, 158, 29, 86, 209, 67, 172, 163, 79, 22, 167, 113,
        154, 5, 157, 187, 237, 144, 76, 77, 45, 177, 15, 21, 19, 7, 181, 125, 232, 9, 138, 111,
        163, 207, 196, 197, 162, 67, 66, 81, 36, 3, 68, 68, 36, 213, 243, 42, 41, 12, 173, 71, 197,
        51, 217, 122, 98, 142, 79, 177, 34, 6, 160, 189, 158, 248, 115, 238, 226, 10, 71, 179, 92,
        87, 14, 100, 13, 220, 105, 238, 238, 0, 229, 100, 96, 29, 18, 155, 53, 247, 10, 146, 252,
        2, 219, 96, 206, 225, 171, 59, 135, 177, 90, 174, 245, 3, 222, 95, 35, 1, 31, 107, 53, 99,
        117, 93, 207, 148, 62, 12, 140, 211, 0, 122, 39, 66, 21, 71, 197, 137, 97, 142, 240, 61,
        69, 25, 121, 165, 53, 40, 218, 248, 112, 213, 8, 91, 217, 236, 173, 227, 103, 250, 133,
        143, 0, 227, 117, 159, 69, 8, 52, 147, 0, 175, 255, 76, 129, 47, 230, 234, 58, 233, 128, 2,
        184, 190, 104, 241, 69, 95, 74, 176, 112, 75, 236, 81, 100, 222, 111, 171, 126, 16, 55, 26,
        1, 63, 15, 24, 147, 176, 50, 20, 23, 176, 176, 182, 72, 208, 105, 138, 160, 243, 113, 123,
        106, 145, 39, 176, 0, 192, 137, 187, 182, 9, 5, 16, 72, 2, 121, 54, 224, 231, 208, 116,
        144, 55, 134, 20, 235, 123, 80, 105, 37, 196, 166, 51, 26, 151, 84, 52, 120, 162, 194, 97,
        212, 242, 153, 205, 112, 158, 151, 244, 242, 79, 219, 128, 99, 246, 32, 38, 132, 34, 131,
        28, 117, 162, 68, 107, 21, 25, 68, 66, 203, 7, 179, 46, 119, 8, 31, 36, 239, 179, 140, 51,
        221, 151, 200, 68, 183, 132, 36, 171, 247, 80, 191, 125, 22, 210, 204, 83, 200, 103, 97,
        184, 61, 225, 127, 73, 104, 159, 172, 185, 202, 116, 154, 28, 150, 6,
    ];

    // a random modulus
    let modulus = [
        186, 147, 169, 139, 184, 41, 41, 80, 108, 1, 29, 237, 140, 15, 147, 98, 8, 192, 145, 125,
        112, 68, 139, 38, 31, 131, 18, 213, 128, 28, 239, 126, 50, 54, 178, 48, 110, 200, 255, 10,
        206, 143, 168, 51, 43, 18, 24, 80, 166, 137, 197, 251, 215, 163, 248, 230, 16, 81, 164, 40,
        118, 207, 12, 149, 30, 55, 8, 135, 56, 9, 118, 228, 18, 190, 224, 181, 78, 70, 141, 109,
        114, 123, 200, 223, 241, 143, 249, 55, 171, 184, 16, 113, 63, 95, 194, 141, 196, 90, 118,
        203, 194, 73, 229, 187, 126, 241, 137, 9, 58, 68, 205, 95, 6, 254, 191, 224, 240, 85, 254,
        200, 6, 210, 24, 1, 32, 131, 185, 210, 109, 71, 116, 58, 213, 187, 115, 210, 107, 65, 116,
        172, 131, 209, 45, 65, 144, 7, 82, 199, 187, 97, 202, 5, 47, 159, 80, 235, 221, 89, 102,
        236, 123, 243, 25, 133, 28, 19, 140, 117, 77, 214, 127, 208, 94, 18, 24, 166, 38, 101, 165,
        144, 183, 109, 78, 100, 128, 67, 215, 253, 248, 244, 242, 182, 219, 26, 135, 24, 192, 149,
        247, 65, 206, 203, 28, 66, 205, 95, 65, 190, 163, 68, 24, 142, 228, 152, 231, 136, 169,
        197, 37, 124, 201, 235, 127, 51, 7, 197, 221, 161, 243, 212, 204, 128, 76, 52, 58, 3, 60,
        158, 162, 140, 51, 233, 167, 231, 190, 227, 111, 82, 210, 48, 79, 232, 80, 202, 114, 161,
        230, 250, 197, 122, 65, 90, 91, 110, 7, 106, 55, 170, 156, 118, 133, 101, 248, 234, 156,
        106, 228, 193, 33, 74, 82, 188, 205, 42, 2, 187, 75, 213, 33, 67, 206, 15, 163, 25, 72, 32,
        192, 92, 239, 40, 15, 116, 155, 120, 213, 50, 142, 47, 161, 30, 44, 213, 100, 217, 213,
        159, 190, 230, 163, 63, 152, 119, 190, 42, 154, 81, 39, 70, 179, 101, 169, 74, 37, 143, 49,
        134, 99, 130, 101, 87, 111, 240, 38, 240, 164, 134, 162, 124, 109, 161, 181, 192, 42, 57,
        240, 133, 18, 215, 175, 164, 45, 150, 161, 30, 104, 217, 68, 3, 5, 57, 170, 148, 24, 116,
        96, 172, 23, 139, 23, 230, 127, 182, 178, 167, 100, 46, 182, 190, 176, 4, 245, 103, 158,
        50, 96, 24, 27, 107, 135, 46, 122, 17, 225, 183, 0, 58, 3, 243, 168, 84, 64, 34, 29, 39,
        17, 227, 240, 30, 150, 28, 76, 11, 86, 53, 143, 25, 164, 215, 164, 70, 71, 232, 195, 95,
        221, 117, 135, 3, 241, 40, 207, 123, 122, 148, 118, 56, 65, 218, 8, 88, 124, 219, 91, 236,
        194, 245, 37, 246, 54, 202, 36, 151, 160, 40, 75, 92, 175, 241, 161, 113, 53, 220, 67, 249,
        7, 118, 77, 214, 122, 148, 44, 164, 249, 239, 57, 102, 45, 136, 93, 4, 76, 51, 186, 156,
        101, 82, 217, 116, 211, 255, 92, 183, 101, 50, 73, 38, 79, 219, 63, 140, 174, 217,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_0512_bits_mod_0512_bits_even(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        30, 194, 86, 18, 177, 202, 203, 133, 7, 22, 133, 138, 102, 21, 107, 127, 141, 28, 166, 162,
        46, 101, 20, 119, 208, 32, 66, 141, 122, 70, 94, 79, 61, 251, 202, 97, 35, 129, 9, 73, 215,
        220, 88, 174, 113, 124, 103, 113, 254, 44, 96, 79, 250, 252, 33, 252, 12, 70, 43, 238, 90,
        32, 248, 119,
    ];

    // a random base
    let base = [
        243, 125, 228, 56, 107, 91, 133, 57, 46, 184, 164, 236, 176, 173, 36, 149, 58, 238, 150,
        32, 181, 248, 42, 134, 92, 170, 70, 16, 109, 212, 16, 28, 195, 174, 187, 226, 140, 22, 3,
        31, 96, 234, 110, 254, 106, 215, 101, 164, 190, 88, 14, 112, 151, 78, 205, 151, 254, 225,
        153, 125, 109, 4, 68, 87,
    ];

    // a random modulus
    let modulus = [
        249, 242, 76, 142, 109, 239, 3, 168, 130, 45, 156, 105, 209, 72, 218, 93, 86, 112, 88, 215,
        43, 194, 59, 35, 44, 86, 2, 252, 132, 113, 24, 4, 109, 98, 68, 209, 53, 191, 213, 162, 221,
        114, 213, 66, 58, 254, 152, 79, 82, 222, 79, 76, 1, 68, 255, 3, 218, 218, 83, 98, 85, 108,
        65, 86,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_0512_bits_mod_1024_bits_even(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        30, 194, 86, 18, 177, 202, 203, 133, 7, 22, 133, 138, 102, 21, 107, 127, 141, 28, 166, 162,
        46, 101, 20, 119, 208, 32, 66, 141, 122, 70, 94, 79, 61, 251, 202, 97, 35, 129, 9, 73, 215,
        220, 88, 174, 113, 124, 103, 113, 254, 44, 96, 79, 250, 252, 33, 252, 12, 70, 43, 238, 90,
        32, 248, 119,
    ];

    // a random base
    let base = [
        212, 177, 76, 117, 205, 210, 60, 63, 64, 203, 108, 135, 17, 114, 163, 95, 224, 106, 191,
        205, 112, 55, 118, 224, 234, 35, 97, 95, 254, 115, 29, 29, 155, 136, 100, 24, 208, 77, 141,
        136, 62, 201, 198, 76, 142, 13, 77, 111, 231, 196, 136, 198, 18, 7, 42, 155, 127, 177, 176,
        250, 30, 83, 243, 231, 197, 161, 33, 122, 254, 152, 1, 214, 70, 166, 155, 142, 77, 132,
        159, 158, 102, 62, 53, 226, 255, 64, 220, 68, 201, 221, 248, 237, 88, 135, 64, 229, 111,
        183, 2, 52, 48, 1, 45, 146, 26, 132, 64, 31, 77, 137, 131, 245, 191, 166, 68, 1, 216, 211,
        177, 196, 122, 75, 212, 7, 183, 80, 240, 66,
    ];

    // a random modulus
    let modulus = [
        237, 223, 117, 18, 201, 74, 224, 241, 183, 182, 90, 141, 137, 224, 33, 142, 133, 75, 192,
        99, 45, 75, 185, 134, 226, 65, 105, 202, 253, 125, 129, 38, 135, 120, 49, 248, 112, 4, 211,
        189, 0, 156, 21, 44, 227, 62, 38, 6, 32, 45, 254, 31, 108, 151, 172, 166, 18, 46, 3, 141,
        22, 176, 57, 160, 40, 104, 39, 68, 167, 233, 192, 157, 33, 200, 231, 220, 195, 161, 137,
        235, 28, 4, 117, 223, 173, 4, 38, 143, 50, 16, 254, 176, 146, 13, 195, 210, 247, 134, 71,
        226, 101, 7, 52, 150, 227, 221, 149, 152, 52, 84, 142, 243, 197, 230, 134, 182, 126, 183,
        122, 82, 62, 74, 173, 42, 233, 38, 13, 242,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_1024_bits_mod_1024_bits_even(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        107, 92, 159, 59, 101, 117, 205, 228, 222, 58, 188, 58, 254, 101, 230, 53, 203, 200, 138,
        56, 160, 233, 81, 218, 113, 119, 10, 214, 68, 109, 113, 15, 146, 191, 225, 80, 22, 199,
        119, 236, 23, 159, 148, 40, 113, 28, 75, 45, 15, 54, 5, 64, 103, 55, 1, 220, 236, 41, 218,
        41, 93, 6, 3, 106, 235, 31, 22, 73, 243, 113, 171, 111, 20, 237, 200, 8, 99, 252, 202, 99,
        122, 242, 84, 180, 8, 58, 3, 129, 145, 62, 179, 78, 199, 35, 212, 16, 3, 55, 9, 197, 217,
        30, 42, 67, 220, 121, 193, 16, 15, 170, 116, 65, 157, 109, 34, 211, 41, 116, 161, 40, 77,
        223, 200, 240, 31, 17, 141, 189,
    ];

    // a random base
    let base = [
        212, 177, 76, 117, 205, 210, 60, 63, 64, 203, 108, 135, 17, 114, 163, 95, 224, 106, 191,
        205, 112, 55, 118, 224, 234, 35, 97, 95, 254, 115, 29, 29, 155, 136, 100, 24, 208, 77, 141,
        136, 62, 201, 198, 76, 142, 13, 77, 111, 231, 196, 136, 198, 18, 7, 42, 155, 127, 177, 176,
        250, 30, 83, 243, 231, 197, 161, 33, 122, 254, 152, 1, 214, 70, 166, 155, 142, 77, 132,
        159, 158, 102, 62, 53, 226, 255, 64, 220, 68, 201, 221, 248, 237, 88, 135, 64, 229, 111,
        183, 2, 52, 48, 1, 45, 146, 26, 132, 64, 31, 77, 137, 131, 245, 191, 166, 68, 1, 216, 211,
        177, 196, 122, 75, 212, 7, 183, 80, 240, 66,
    ];

    // a random modulus
    let modulus = [
        237, 223, 117, 18, 201, 74, 224, 241, 183, 182, 90, 141, 137, 224, 33, 142, 133, 75, 192,
        99, 45, 75, 185, 134, 226, 65, 105, 202, 253, 125, 129, 38, 135, 120, 49, 248, 112, 4, 211,
        189, 0, 156, 21, 44, 227, 62, 38, 6, 32, 45, 254, 31, 108, 151, 172, 166, 18, 46, 3, 141,
        22, 176, 57, 160, 40, 104, 39, 68, 167, 233, 192, 157, 33, 200, 231, 220, 195, 161, 137,
        235, 28, 4, 117, 223, 173, 4, 38, 143, 50, 16, 254, 176, 146, 13, 195, 210, 247, 134, 71,
        226, 101, 7, 52, 150, 227, 221, 149, 152, 52, 84, 142, 243, 197, 230, 134, 182, 126, 183,
        122, 82, 62, 74, 173, 42, 233, 38, 13, 242,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_0512_bits_mod_2048_bits_even(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        30, 194, 86, 18, 177, 202, 203, 133, 7, 22, 133, 138, 102, 21, 107, 127, 141, 28, 166, 162,
        46, 101, 20, 119, 208, 32, 66, 141, 122, 70, 94, 79, 61, 251, 202, 97, 35, 129, 9, 73, 215,
        220, 88, 174, 113, 124, 103, 113, 254, 44, 96, 79, 250, 252, 33, 252, 12, 70, 43, 238, 90,
        32, 248, 119,
    ];

    // a random base
    let base = [
        236, 66, 86, 50, 109, 186, 105, 32, 149, 81, 7, 31, 151, 116, 93, 59, 84, 247, 239, 100,
        174, 245, 37, 174, 204, 240, 52, 226, 252, 105, 16, 88, 180, 24, 90, 223, 60, 226, 147,
        165, 255, 74, 249, 183, 1, 80, 160, 48, 111, 90, 220, 16, 237, 217, 251, 30, 208, 24, 127,
        170, 237, 244, 15, 89, 205, 254, 128, 49, 245, 233, 139, 239, 4, 126, 248, 212, 173, 103,
        222, 225, 118, 93, 219, 91, 26, 90, 142, 91, 210, 149, 227, 222, 255, 227, 182, 233, 176,
        83, 210, 143, 174, 92, 46, 19, 242, 117, 6, 125, 163, 93, 116, 63, 71, 236, 139, 17, 192,
        239, 106, 133, 145, 158, 46, 238, 107, 80, 54, 80, 231, 138, 236, 44, 55, 13, 193, 159,
        144, 85, 138, 204, 84, 126, 66, 40, 104, 232, 113, 216, 165, 184, 198, 20, 234, 225, 170,
        174, 90, 101, 253, 231, 80, 252, 28, 148, 89, 64, 65, 26, 143, 60, 158, 116, 12, 14, 46,
        210, 99, 233, 187, 212, 44, 36, 47, 227, 123, 195, 45, 115, 12, 123, 16, 164, 92, 52, 229,
        65, 127, 114, 213, 116, 210, 2, 149, 144, 217, 131, 146, 67, 66, 91, 199, 46, 58, 5, 185,
        247, 73, 170, 6, 45, 109, 0, 191, 233, 95, 239, 241, 30, 61, 119, 54, 4, 164, 214, 202,
        251, 139, 28, 22, 219, 100, 233, 195, 151, 237, 183, 41, 153, 42, 82, 208, 222, 21, 160,
        100, 100,
    ];

    // a random modulus
    let modulus = [
        155, 66, 179, 54, 45, 180, 207, 15, 110, 66, 217, 170, 218, 229, 14, 147, 163, 227, 26, 27,
        56, 162, 176, 213, 136, 239, 229, 242, 214, 53, 97, 19, 91, 195, 133, 126, 130, 1, 54, 143,
        78, 210, 176, 236, 152, 95, 92, 140, 158, 72, 151, 225, 83, 120, 44, 192, 72, 12, 100, 19,
        76, 249, 175, 180, 3, 217, 241, 47, 99, 8, 101, 17, 7, 154, 235, 191, 239, 243, 156, 137,
        147, 6, 248, 70, 44, 52, 4, 159, 137, 14, 79, 178, 247, 112, 241, 56, 240, 45, 22, 250, 99,
        99, 79, 10, 147, 188, 219, 89, 129, 60, 204, 149, 6, 112, 52, 85, 204, 62, 164, 85, 59,
        200, 11, 239, 196, 157, 53, 128, 223, 221, 90, 234, 112, 74, 195, 52, 133, 189, 35, 110,
        66, 222, 150, 19, 121, 107, 23, 171, 46, 167, 10, 253, 119, 247, 214, 175, 239, 40, 45, 24,
        115, 2, 150, 243, 44, 187, 160, 142, 68, 56, 172, 77, 143, 142, 53, 216, 228, 216, 239,
        176, 186, 96, 11, 147, 160, 127, 107, 192, 246, 173, 95, 144, 190, 167, 93, 172, 81, 89,
        163, 86, 111, 48, 30, 172, 32, 33, 34, 224, 191, 214, 244, 161, 233, 222, 113, 112, 76,
        163, 71, 99, 138, 92, 127, 203, 253, 201, 164, 231, 61, 59, 98, 165, 238, 23, 196, 10, 177,
        253, 110, 149, 31, 57, 212, 43, 16, 24, 241, 163, 72, 81, 140, 115, 37, 155, 94, 40,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_1024_bits_mod_2048_bits_even(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        107, 92, 159, 59, 101, 117, 205, 228, 222, 58, 188, 58, 254, 101, 230, 53, 203, 200, 138,
        56, 160, 233, 81, 218, 113, 119, 10, 214, 68, 109, 113, 15, 146, 191, 225, 80, 22, 199,
        119, 236, 23, 159, 148, 40, 113, 28, 75, 45, 15, 54, 5, 64, 103, 55, 1, 220, 236, 41, 218,
        41, 93, 6, 3, 106, 235, 31, 22, 73, 243, 113, 171, 111, 20, 237, 200, 8, 99, 252, 202, 99,
        122, 242, 84, 180, 8, 58, 3, 129, 145, 62, 179, 78, 199, 35, 212, 16, 3, 55, 9, 197, 217,
        30, 42, 67, 220, 121, 193, 16, 15, 170, 116, 65, 157, 109, 34, 211, 41, 116, 161, 40, 77,
        223, 200, 240, 31, 17, 141, 189,
    ];

    // a random base
    let base = [
        236, 66, 86, 50, 109, 186, 105, 32, 149, 81, 7, 31, 151, 116, 93, 59, 84, 247, 239, 100,
        174, 245, 37, 174, 204, 240, 52, 226, 252, 105, 16, 88, 180, 24, 90, 223, 60, 226, 147,
        165, 255, 74, 249, 183, 1, 80, 160, 48, 111, 90, 220, 16, 237, 217, 251, 30, 208, 24, 127,
        170, 237, 244, 15, 89, 205, 254, 128, 49, 245, 233, 139, 239, 4, 126, 248, 212, 173, 103,
        222, 225, 118, 93, 219, 91, 26, 90, 142, 91, 210, 149, 227, 222, 255, 227, 182, 233, 176,
        83, 210, 143, 174, 92, 46, 19, 242, 117, 6, 125, 163, 93, 116, 63, 71, 236, 139, 17, 192,
        239, 106, 133, 145, 158, 46, 238, 107, 80, 54, 80, 231, 138, 236, 44, 55, 13, 193, 159,
        144, 85, 138, 204, 84, 126, 66, 40, 104, 232, 113, 216, 165, 184, 198, 20, 234, 225, 170,
        174, 90, 101, 253, 231, 80, 252, 28, 148, 89, 64, 65, 26, 143, 60, 158, 116, 12, 14, 46,
        210, 99, 233, 187, 212, 44, 36, 47, 227, 123, 195, 45, 115, 12, 123, 16, 164, 92, 52, 229,
        65, 127, 114, 213, 116, 210, 2, 149, 144, 217, 131, 146, 67, 66, 91, 199, 46, 58, 5, 185,
        247, 73, 170, 6, 45, 109, 0, 191, 233, 95, 239, 241, 30, 61, 119, 54, 4, 164, 214, 202,
        251, 139, 28, 22, 219, 100, 233, 195, 151, 237, 183, 41, 153, 42, 82, 208, 222, 21, 160,
        100, 100,
    ];

    // a random modulus
    let modulus = [
        155, 66, 179, 54, 45, 180, 207, 15, 110, 66, 217, 170, 218, 229, 14, 147, 163, 227, 26, 27,
        56, 162, 176, 213, 136, 239, 229, 242, 214, 53, 97, 19, 91, 195, 133, 126, 130, 1, 54, 143,
        78, 210, 176, 236, 152, 95, 92, 140, 158, 72, 151, 225, 83, 120, 44, 192, 72, 12, 100, 19,
        76, 249, 175, 180, 3, 217, 241, 47, 99, 8, 101, 17, 7, 154, 235, 191, 239, 243, 156, 137,
        147, 6, 248, 70, 44, 52, 4, 159, 137, 14, 79, 178, 247, 112, 241, 56, 240, 45, 22, 250, 99,
        99, 79, 10, 147, 188, 219, 89, 129, 60, 204, 149, 6, 112, 52, 85, 204, 62, 164, 85, 59,
        200, 11, 239, 196, 157, 53, 128, 223, 221, 90, 234, 112, 74, 195, 52, 133, 189, 35, 110,
        66, 222, 150, 19, 121, 107, 23, 171, 46, 167, 10, 253, 119, 247, 214, 175, 239, 40, 45, 24,
        115, 2, 150, 243, 44, 187, 160, 142, 68, 56, 172, 77, 143, 142, 53, 216, 228, 216, 239,
        176, 186, 96, 11, 147, 160, 127, 107, 192, 246, 173, 95, 144, 190, 167, 93, 172, 81, 89,
        163, 86, 111, 48, 30, 172, 32, 33, 34, 224, 191, 214, 244, 161, 233, 222, 113, 112, 76,
        163, 71, 99, 138, 92, 127, 203, 253, 201, 164, 231, 61, 59, 98, 165, 238, 23, 196, 10, 177,
        253, 110, 149, 31, 57, 212, 43, 16, 24, 241, 163, 72, 81, 140, 115, 37, 155, 94, 40,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_2048_bits_mod_2048_bits_even(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        234, 85, 222, 102, 95, 165, 186, 221, 192, 109, 51, 204, 69, 225, 36, 118, 52, 132, 39,
        190, 10, 8, 82, 87, 149, 233, 35, 50, 36, 102, 243, 84, 50, 26, 54, 64, 38, 68, 154, 97,
        100, 221, 4, 81, 15, 47, 229, 100, 163, 68, 127, 163, 138, 24, 244, 125, 166, 116, 68, 126,
        201, 43, 192, 13, 236, 182, 213, 203, 235, 20, 2, 81, 168, 251, 87, 97, 69, 159, 138, 203,
        53, 43, 243, 14, 212, 5, 0, 229, 80, 72, 147, 130, 47, 13, 236, 180, 25, 100, 178, 148,
        171, 231, 252, 68, 57, 79, 14, 185, 155, 82, 103, 1, 98, 32, 204, 127, 242, 86, 25, 37, 19,
        240, 21, 64, 3, 160, 100, 76, 72, 220, 67, 123, 123, 139, 206, 75, 33, 177, 61, 129, 69,
        57, 186, 166, 3, 94, 162, 249, 22, 89, 245, 106, 180, 116, 222, 177, 231, 57, 73, 6, 217,
        252, 58, 212, 233, 219, 42, 144, 68, 92, 168, 147, 116, 82, 211, 224, 214, 156, 1, 52, 112,
        114, 193, 158, 137, 195, 46, 73, 179, 7, 229, 69, 151, 34, 78, 108, 138, 207, 37, 178, 41,
        142, 41, 163, 144, 206, 181, 71, 13, 195, 186, 74, 56, 93, 151, 97, 73, 57, 114, 198, 203,
        216, 182, 98, 88, 9, 68, 211, 235, 78, 105, 182, 245, 96, 5, 119, 229, 2, 50, 187, 159,
        131, 24, 4, 154, 234, 61, 95, 45, 102, 134, 106, 208, 39, 202, 165,
    ];

    // a random base
    let base = [
        236, 66, 86, 50, 109, 186, 105, 32, 149, 81, 7, 31, 151, 116, 93, 59, 84, 247, 239, 100,
        174, 245, 37, 174, 204, 240, 52, 226, 252, 105, 16, 88, 180, 24, 90, 223, 60, 226, 147,
        165, 255, 74, 249, 183, 1, 80, 160, 48, 111, 90, 220, 16, 237, 217, 251, 30, 208, 24, 127,
        170, 237, 244, 15, 89, 205, 254, 128, 49, 245, 233, 139, 239, 4, 126, 248, 212, 173, 103,
        222, 225, 118, 93, 219, 91, 26, 90, 142, 91, 210, 149, 227, 222, 255, 227, 182, 233, 176,
        83, 210, 143, 174, 92, 46, 19, 242, 117, 6, 125, 163, 93, 116, 63, 71, 236, 139, 17, 192,
        239, 106, 133, 145, 158, 46, 238, 107, 80, 54, 80, 231, 138, 236, 44, 55, 13, 193, 159,
        144, 85, 138, 204, 84, 126, 66, 40, 104, 232, 113, 216, 165, 184, 198, 20, 234, 225, 170,
        174, 90, 101, 253, 231, 80, 252, 28, 148, 89, 64, 65, 26, 143, 60, 158, 116, 12, 14, 46,
        210, 99, 233, 187, 212, 44, 36, 47, 227, 123, 195, 45, 115, 12, 123, 16, 164, 92, 52, 229,
        65, 127, 114, 213, 116, 210, 2, 149, 144, 217, 131, 146, 67, 66, 91, 199, 46, 58, 5, 185,
        247, 73, 170, 6, 45, 109, 0, 191, 233, 95, 239, 241, 30, 61, 119, 54, 4, 164, 214, 202,
        251, 139, 28, 22, 219, 100, 233, 195, 151, 237, 183, 41, 153, 42, 82, 208, 222, 21, 160,
        100, 100,
    ];

    // a random modulus
    let modulus = [
        155, 66, 179, 54, 45, 180, 207, 15, 110, 66, 217, 170, 218, 229, 14, 147, 163, 227, 26, 27,
        56, 162, 176, 213, 136, 239, 229, 242, 214, 53, 97, 19, 91, 195, 133, 126, 130, 1, 54, 143,
        78, 210, 176, 236, 152, 95, 92, 140, 158, 72, 151, 225, 83, 120, 44, 192, 72, 12, 100, 19,
        76, 249, 175, 180, 3, 217, 241, 47, 99, 8, 101, 17, 7, 154, 235, 191, 239, 243, 156, 137,
        147, 6, 248, 70, 44, 52, 4, 159, 137, 14, 79, 178, 247, 112, 241, 56, 240, 45, 22, 250, 99,
        99, 79, 10, 147, 188, 219, 89, 129, 60, 204, 149, 6, 112, 52, 85, 204, 62, 164, 85, 59,
        200, 11, 239, 196, 157, 53, 128, 223, 221, 90, 234, 112, 74, 195, 52, 133, 189, 35, 110,
        66, 222, 150, 19, 121, 107, 23, 171, 46, 167, 10, 253, 119, 247, 214, 175, 239, 40, 45, 24,
        115, 2, 150, 243, 44, 187, 160, 142, 68, 56, 172, 77, 143, 142, 53, 216, 228, 216, 239,
        176, 186, 96, 11, 147, 160, 127, 107, 192, 246, 173, 95, 144, 190, 167, 93, 172, 81, 89,
        163, 86, 111, 48, 30, 172, 32, 33, 34, 224, 191, 214, 244, 161, 233, 222, 113, 112, 76,
        163, 71, 99, 138, 92, 127, 203, 253, 201, 164, 231, 61, 59, 98, 165, 238, 23, 196, 10, 177,
        253, 110, 149, 31, 57, 212, 43, 16, 24, 241, 163, 72, 81, 140, 115, 37, 155, 94, 40,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_0512_bits_mod_4096_bits_even(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        30, 194, 86, 18, 177, 202, 203, 133, 7, 22, 133, 138, 102, 21, 107, 127, 141, 28, 166, 162,
        46, 101, 20, 119, 208, 32, 66, 141, 122, 70, 94, 79, 61, 251, 202, 97, 35, 129, 9, 73, 215,
        220, 88, 174, 113, 124, 103, 113, 254, 44, 96, 79, 250, 252, 33, 252, 12, 70, 43, 238, 90,
        32, 248, 119,
    ];

    // a random base
    let base = [
        167, 62, 26, 222, 69, 55, 136, 13, 139, 152, 251, 52, 101, 99, 181, 160, 55, 137, 73, 135,
        48, 71, 232, 168, 183, 14, 161, 145, 163, 22, 31, 14, 111, 65, 93, 52, 253, 42, 115, 167,
        211, 27, 143, 95, 20, 20, 179, 231, 241, 169, 110, 220, 42, 178, 222, 110, 97, 186, 76,
        210, 43, 205, 210, 166, 58, 173, 28, 10, 192, 102, 35, 122, 240, 244, 186, 197, 211, 39,
        230, 59, 140, 42, 221, 90, 81, 107, 88, 210, 1, 222, 215, 116, 7, 190, 251, 226, 87, 131,
        53, 137, 108, 245, 194, 62, 173, 26, 124, 253, 192, 10, 143, 10, 56, 237, 191, 216, 13, 6,
        22, 37, 130, 226, 241, 58, 157, 230, 247, 125, 170, 124, 167, 184, 148, 204, 247, 5, 181,
        73, 81, 195, 64, 72, 33, 187, 100, 173, 35, 86, 99, 228, 124, 83, 64, 1, 176, 178, 194,
        236, 110, 202, 70, 204, 59, 147, 232, 246, 152, 93, 124, 222, 192, 136, 149, 14, 36, 225,
        122, 228, 16, 215, 49, 80, 30, 230, 235, 158, 29, 86, 209, 67, 172, 163, 79, 22, 167, 113,
        154, 5, 157, 187, 237, 144, 76, 77, 45, 177, 15, 21, 19, 7, 181, 125, 232, 9, 138, 111,
        163, 207, 196, 197, 162, 67, 66, 81, 36, 3, 68, 68, 36, 213, 243, 42, 41, 12, 173, 71, 197,
        51, 217, 122, 98, 142, 79, 177, 34, 6, 160, 189, 158, 248, 115, 238, 226, 10, 71, 179, 92,
        87, 14, 100, 13, 220, 105, 238, 238, 0, 229, 100, 96, 29, 18, 155, 53, 247, 10, 146, 252,
        2, 219, 96, 206, 225, 171, 59, 135, 177, 90, 174, 245, 3, 222, 95, 35, 1, 31, 107, 53, 99,
        117, 93, 207, 148, 62, 12, 140, 211, 0, 122, 39, 66, 21, 71, 197, 137, 97, 142, 240, 61,
        69, 25, 121, 165, 53, 40, 218, 248, 112, 213, 8, 91, 217, 236, 173, 227, 103, 250, 133,
        143, 0, 227, 117, 159, 69, 8, 52, 147, 0, 175, 255, 76, 129, 47, 230, 234, 58, 233, 128, 2,
        184, 190, 104, 241, 69, 95, 74, 176, 112, 75, 236, 81, 100, 222, 111, 171, 126, 16, 55, 26,
        1, 63, 15, 24, 147, 176, 50, 20, 23, 176, 176, 182, 72, 208, 105, 138, 160, 243, 113, 123,
        106, 145, 39, 176, 0, 192, 137, 187, 182, 9, 5, 16, 72, 2, 121, 54, 224, 231, 208, 116,
        144, 55, 134, 20, 235, 123, 80, 105, 37, 196, 166, 51, 26, 151, 84, 52, 120, 162, 194, 97,
        212, 242, 153, 205, 112, 158, 151, 244, 242, 79, 219, 128, 99, 246, 32, 38, 132, 34, 131,
        28, 117, 162, 68, 107, 21, 25, 68, 66, 203, 7, 179, 46, 119, 8, 31, 36, 239, 179, 140, 51,
        221, 151, 200, 68, 183, 132, 36, 171, 247, 80, 191, 125, 22, 210, 204, 83, 200, 103, 97,
        184, 61, 225, 127, 73, 104, 159, 172, 185, 202, 116, 154, 28, 150, 6,
    ];

    // a random modulus
    let modulus = [
        186, 147, 169, 139, 184, 41, 41, 80, 108, 1, 29, 237, 140, 15, 147, 98, 8, 192, 145, 125,
        112, 68, 139, 38, 31, 131, 18, 213, 128, 28, 239, 126, 50, 54, 178, 48, 110, 200, 255, 10,
        206, 143, 168, 51, 43, 18, 24, 80, 166, 137, 197, 251, 215, 163, 248, 230, 16, 81, 164, 40,
        118, 207, 12, 149, 30, 55, 8, 135, 56, 9, 118, 228, 18, 190, 224, 181, 78, 70, 141, 109,
        114, 123, 200, 223, 241, 143, 249, 55, 171, 184, 16, 113, 63, 95, 194, 141, 196, 90, 118,
        203, 194, 73, 229, 187, 126, 241, 137, 9, 58, 68, 205, 95, 6, 254, 191, 224, 240, 85, 254,
        200, 6, 210, 24, 1, 32, 131, 185, 210, 109, 71, 116, 58, 213, 187, 115, 210, 107, 65, 116,
        172, 131, 209, 45, 65, 144, 7, 82, 199, 187, 97, 202, 5, 47, 159, 80, 235, 221, 89, 102,
        236, 123, 243, 25, 133, 28, 19, 140, 117, 77, 214, 127, 208, 94, 18, 24, 166, 38, 101, 165,
        144, 183, 109, 78, 100, 128, 67, 215, 253, 248, 244, 242, 182, 219, 26, 135, 24, 192, 149,
        247, 65, 206, 203, 28, 66, 205, 95, 65, 190, 163, 68, 24, 142, 228, 152, 231, 136, 169,
        197, 37, 124, 201, 235, 127, 51, 7, 197, 221, 161, 243, 212, 204, 128, 76, 52, 58, 3, 60,
        158, 162, 140, 51, 233, 167, 231, 190, 227, 111, 82, 210, 48, 79, 232, 80, 202, 114, 161,
        230, 250, 197, 122, 65, 90, 91, 110, 7, 106, 55, 170, 156, 118, 133, 101, 248, 234, 156,
        106, 228, 193, 33, 74, 82, 188, 205, 42, 2, 187, 75, 213, 33, 67, 206, 15, 163, 25, 72, 32,
        192, 92, 239, 40, 15, 116, 155, 120, 213, 50, 142, 47, 161, 30, 44, 213, 100, 217, 213,
        159, 190, 230, 163, 63, 152, 119, 190, 42, 154, 81, 39, 70, 179, 101, 169, 74, 37, 143, 49,
        134, 99, 130, 101, 87, 111, 240, 38, 240, 164, 134, 162, 124, 109, 161, 181, 192, 42, 57,
        240, 133, 18, 215, 175, 164, 45, 150, 161, 30, 104, 217, 68, 3, 5, 57, 170, 148, 24, 116,
        96, 172, 23, 139, 23, 230, 127, 182, 178, 167, 100, 46, 182, 190, 176, 4, 245, 103, 158,
        50, 96, 24, 27, 107, 135, 46, 122, 17, 225, 183, 0, 58, 3, 243, 168, 84, 64, 34, 29, 39,
        17, 227, 240, 30, 150, 28, 76, 11, 86, 53, 143, 25, 164, 215, 164, 70, 71, 232, 195, 95,
        221, 117, 135, 3, 241, 40, 207, 123, 122, 148, 118, 56, 65, 218, 8, 88, 124, 219, 91, 236,
        194, 245, 37, 246, 54, 202, 36, 151, 160, 40, 75, 92, 175, 241, 161, 113, 53, 220, 67, 249,
        7, 118, 77, 214, 122, 148, 44, 164, 249, 239, 57, 102, 45, 136, 93, 4, 76, 51, 186, 156,
        101, 82, 217, 116, 211, 255, 92, 183, 101, 50, 73, 38, 79, 219, 63, 140, 174, 218,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_1024_bits_mod_4096_bits_even(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        107, 92, 159, 59, 101, 117, 205, 228, 222, 58, 188, 58, 254, 101, 230, 53, 203, 200, 138,
        56, 160, 233, 81, 218, 113, 119, 10, 214, 68, 109, 113, 15, 146, 191, 225, 80, 22, 199,
        119, 236, 23, 159, 148, 40, 113, 28, 75, 45, 15, 54, 5, 64, 103, 55, 1, 220, 236, 41, 218,
        41, 93, 6, 3, 106, 235, 31, 22, 73, 243, 113, 171, 111, 20, 237, 200, 8, 99, 252, 202, 99,
        122, 242, 84, 180, 8, 58, 3, 129, 145, 62, 179, 78, 199, 35, 212, 16, 3, 55, 9, 197, 217,
        30, 42, 67, 220, 121, 193, 16, 15, 170, 116, 65, 157, 109, 34, 211, 41, 116, 161, 40, 77,
        223, 200, 240, 31, 17, 141, 189,
    ];

    // a random base
    let base = [
        167, 62, 26, 222, 69, 55, 136, 13, 139, 152, 251, 52, 101, 99, 181, 160, 55, 137, 73, 135,
        48, 71, 232, 168, 183, 14, 161, 145, 163, 22, 31, 14, 111, 65, 93, 52, 253, 42, 115, 167,
        211, 27, 143, 95, 20, 20, 179, 231, 241, 169, 110, 220, 42, 178, 222, 110, 97, 186, 76,
        210, 43, 205, 210, 166, 58, 173, 28, 10, 192, 102, 35, 122, 240, 244, 186, 197, 211, 39,
        230, 59, 140, 42, 221, 90, 81, 107, 88, 210, 1, 222, 215, 116, 7, 190, 251, 226, 87, 131,
        53, 137, 108, 245, 194, 62, 173, 26, 124, 253, 192, 10, 143, 10, 56, 237, 191, 216, 13, 6,
        22, 37, 130, 226, 241, 58, 157, 230, 247, 125, 170, 124, 167, 184, 148, 204, 247, 5, 181,
        73, 81, 195, 64, 72, 33, 187, 100, 173, 35, 86, 99, 228, 124, 83, 64, 1, 176, 178, 194,
        236, 110, 202, 70, 204, 59, 147, 232, 246, 152, 93, 124, 222, 192, 136, 149, 14, 36, 225,
        122, 228, 16, 215, 49, 80, 30, 230, 235, 158, 29, 86, 209, 67, 172, 163, 79, 22, 167, 113,
        154, 5, 157, 187, 237, 144, 76, 77, 45, 177, 15, 21, 19, 7, 181, 125, 232, 9, 138, 111,
        163, 207, 196, 197, 162, 67, 66, 81, 36, 3, 68, 68, 36, 213, 243, 42, 41, 12, 173, 71, 197,
        51, 217, 122, 98, 142, 79, 177, 34, 6, 160, 189, 158, 248, 115, 238, 226, 10, 71, 179, 92,
        87, 14, 100, 13, 220, 105, 238, 238, 0, 229, 100, 96, 29, 18, 155, 53, 247, 10, 146, 252,
        2, 219, 96, 206, 225, 171, 59, 135, 177, 90, 174, 245, 3, 222, 95, 35, 1, 31, 107, 53, 99,
        117, 93, 207, 148, 62, 12, 140, 211, 0, 122, 39, 66, 21, 71, 197, 137, 97, 142, 240, 61,
        69, 25, 121, 165, 53, 40, 218, 248, 112, 213, 8, 91, 217, 236, 173, 227, 103, 250, 133,
        143, 0, 227, 117, 159, 69, 8, 52, 147, 0, 175, 255, 76, 129, 47, 230, 234, 58, 233, 128, 2,
        184, 190, 104, 241, 69, 95, 74, 176, 112, 75, 236, 81, 100, 222, 111, 171, 126, 16, 55, 26,
        1, 63, 15, 24, 147, 176, 50, 20, 23, 176, 176, 182, 72, 208, 105, 138, 160, 243, 113, 123,
        106, 145, 39, 176, 0, 192, 137, 187, 182, 9, 5, 16, 72, 2, 121, 54, 224, 231, 208, 116,
        144, 55, 134, 20, 235, 123, 80, 105, 37, 196, 166, 51, 26, 151, 84, 52, 120, 162, 194, 97,
        212, 242, 153, 205, 112, 158, 151, 244, 242, 79, 219, 128, 99, 246, 32, 38, 132, 34, 131,
        28, 117, 162, 68, 107, 21, 25, 68, 66, 203, 7, 179, 46, 119, 8, 31, 36, 239, 179, 140, 51,
        221, 151, 200, 68, 183, 132, 36, 171, 247, 80, 191, 125, 22, 210, 204, 83, 200, 103, 97,
        184, 61, 225, 127, 73, 104, 159, 172, 185, 202, 116, 154, 28, 150, 6,
    ];

    // a random modulus
    let modulus = [
        186, 147, 169, 139, 184, 41, 41, 80, 108, 1, 29, 237, 140, 15, 147, 98, 8, 192, 145, 125,
        112, 68, 139, 38, 31, 131, 18, 213, 128, 28, 239, 126, 50, 54, 178, 48, 110, 200, 255, 10,
        206, 143, 168, 51, 43, 18, 24, 80, 166, 137, 197, 251, 215, 163, 248, 230, 16, 81, 164, 40,
        118, 207, 12, 149, 30, 55, 8, 135, 56, 9, 118, 228, 18, 190, 224, 181, 78, 70, 141, 109,
        114, 123, 200, 223, 241, 143, 249, 55, 171, 184, 16, 113, 63, 95, 194, 141, 196, 90, 118,
        203, 194, 73, 229, 187, 126, 241, 137, 9, 58, 68, 205, 95, 6, 254, 191, 224, 240, 85, 254,
        200, 6, 210, 24, 1, 32, 131, 185, 210, 109, 71, 116, 58, 213, 187, 115, 210, 107, 65, 116,
        172, 131, 209, 45, 65, 144, 7, 82, 199, 187, 97, 202, 5, 47, 159, 80, 235, 221, 89, 102,
        236, 123, 243, 25, 133, 28, 19, 140, 117, 77, 214, 127, 208, 94, 18, 24, 166, 38, 101, 165,
        144, 183, 109, 78, 100, 128, 67, 215, 253, 248, 244, 242, 182, 219, 26, 135, 24, 192, 149,
        247, 65, 206, 203, 28, 66, 205, 95, 65, 190, 163, 68, 24, 142, 228, 152, 231, 136, 169,
        197, 37, 124, 201, 235, 127, 51, 7, 197, 221, 161, 243, 212, 204, 128, 76, 52, 58, 3, 60,
        158, 162, 140, 51, 233, 167, 231, 190, 227, 111, 82, 210, 48, 79, 232, 80, 202, 114, 161,
        230, 250, 197, 122, 65, 90, 91, 110, 7, 106, 55, 170, 156, 118, 133, 101, 248, 234, 156,
        106, 228, 193, 33, 74, 82, 188, 205, 42, 2, 187, 75, 213, 33, 67, 206, 15, 163, 25, 72, 32,
        192, 92, 239, 40, 15, 116, 155, 120, 213, 50, 142, 47, 161, 30, 44, 213, 100, 217, 213,
        159, 190, 230, 163, 63, 152, 119, 190, 42, 154, 81, 39, 70, 179, 101, 169, 74, 37, 143, 49,
        134, 99, 130, 101, 87, 111, 240, 38, 240, 164, 134, 162, 124, 109, 161, 181, 192, 42, 57,
        240, 133, 18, 215, 175, 164, 45, 150, 161, 30, 104, 217, 68, 3, 5, 57, 170, 148, 24, 116,
        96, 172, 23, 139, 23, 230, 127, 182, 178, 167, 100, 46, 182, 190, 176, 4, 245, 103, 158,
        50, 96, 24, 27, 107, 135, 46, 122, 17, 225, 183, 0, 58, 3, 243, 168, 84, 64, 34, 29, 39,
        17, 227, 240, 30, 150, 28, 76, 11, 86, 53, 143, 25, 164, 215, 164, 70, 71, 232, 195, 95,
        221, 117, 135, 3, 241, 40, 207, 123, 122, 148, 118, 56, 65, 218, 8, 88, 124, 219, 91, 236,
        194, 245, 37, 246, 54, 202, 36, 151, 160, 40, 75, 92, 175, 241, 161, 113, 53, 220, 67, 249,
        7, 118, 77, 214, 122, 148, 44, 164, 249, 239, 57, 102, 45, 136, 93, 4, 76, 51, 186, 156,
        101, 82, 217, 116, 211, 255, 92, 183, 101, 50, 73, 38, 79, 219, 63, 140, 174, 218,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_2048_bits_mod_4096_bits_even(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        234, 85, 222, 102, 95, 165, 186, 221, 192, 109, 51, 204, 69, 225, 36, 118, 52, 132, 39,
        190, 10, 8, 82, 87, 149, 233, 35, 50, 36, 102, 243, 84, 50, 26, 54, 64, 38, 68, 154, 97,
        100, 221, 4, 81, 15, 47, 229, 100, 163, 68, 127, 163, 138, 24, 244, 125, 166, 116, 68, 126,
        201, 43, 192, 13, 236, 182, 213, 203, 235, 20, 2, 81, 168, 251, 87, 97, 69, 159, 138, 203,
        53, 43, 243, 14, 212, 5, 0, 229, 80, 72, 147, 130, 47, 13, 236, 180, 25, 100, 178, 148,
        171, 231, 252, 68, 57, 79, 14, 185, 155, 82, 103, 1, 98, 32, 204, 127, 242, 86, 25, 37, 19,
        240, 21, 64, 3, 160, 100, 76, 72, 220, 67, 123, 123, 139, 206, 75, 33, 177, 61, 129, 69,
        57, 186, 166, 3, 94, 162, 249, 22, 89, 245, 106, 180, 116, 222, 177, 231, 57, 73, 6, 217,
        252, 58, 212, 233, 219, 42, 144, 68, 92, 168, 147, 116, 82, 211, 224, 214, 156, 1, 52, 112,
        114, 193, 158, 137, 195, 46, 73, 179, 7, 229, 69, 151, 34, 78, 108, 138, 207, 37, 178, 41,
        142, 41, 163, 144, 206, 181, 71, 13, 195, 186, 74, 56, 93, 151, 97, 73, 57, 114, 198, 203,
        216, 182, 98, 88, 9, 68, 211, 235, 78, 105, 182, 245, 96, 5, 119, 229, 2, 50, 187, 159,
        131, 24, 4, 154, 234, 61, 95, 45, 102, 134, 106, 208, 39, 202, 165,
    ];

    // a random base
    let base = [
        167, 62, 26, 222, 69, 55, 136, 13, 139, 152, 251, 52, 101, 99, 181, 160, 55, 137, 73, 135,
        48, 71, 232, 168, 183, 14, 161, 145, 163, 22, 31, 14, 111, 65, 93, 52, 253, 42, 115, 167,
        211, 27, 143, 95, 20, 20, 179, 231, 241, 169, 110, 220, 42, 178, 222, 110, 97, 186, 76,
        210, 43, 205, 210, 166, 58, 173, 28, 10, 192, 102, 35, 122, 240, 244, 186, 197, 211, 39,
        230, 59, 140, 42, 221, 90, 81, 107, 88, 210, 1, 222, 215, 116, 7, 190, 251, 226, 87, 131,
        53, 137, 108, 245, 194, 62, 173, 26, 124, 253, 192, 10, 143, 10, 56, 237, 191, 216, 13, 6,
        22, 37, 130, 226, 241, 58, 157, 230, 247, 125, 170, 124, 167, 184, 148, 204, 247, 5, 181,
        73, 81, 195, 64, 72, 33, 187, 100, 173, 35, 86, 99, 228, 124, 83, 64, 1, 176, 178, 194,
        236, 110, 202, 70, 204, 59, 147, 232, 246, 152, 93, 124, 222, 192, 136, 149, 14, 36, 225,
        122, 228, 16, 215, 49, 80, 30, 230, 235, 158, 29, 86, 209, 67, 172, 163, 79, 22, 167, 113,
        154, 5, 157, 187, 237, 144, 76, 77, 45, 177, 15, 21, 19, 7, 181, 125, 232, 9, 138, 111,
        163, 207, 196, 197, 162, 67, 66, 81, 36, 3, 68, 68, 36, 213, 243, 42, 41, 12, 173, 71, 197,
        51, 217, 122, 98, 142, 79, 177, 34, 6, 160, 189, 158, 248, 115, 238, 226, 10, 71, 179, 92,
        87, 14, 100, 13, 220, 105, 238, 238, 0, 229, 100, 96, 29, 18, 155, 53, 247, 10, 146, 252,
        2, 219, 96, 206, 225, 171, 59, 135, 177, 90, 174, 245, 3, 222, 95, 35, 1, 31, 107, 53, 99,
        117, 93, 207, 148, 62, 12, 140, 211, 0, 122, 39, 66, 21, 71, 197, 137, 97, 142, 240, 61,
        69, 25, 121, 165, 53, 40, 218, 248, 112, 213, 8, 91, 217, 236, 173, 227, 103, 250, 133,
        143, 0, 227, 117, 159, 69, 8, 52, 147, 0, 175, 255, 76, 129, 47, 230, 234, 58, 233, 128, 2,
        184, 190, 104, 241, 69, 95, 74, 176, 112, 75, 236, 81, 100, 222, 111, 171, 126, 16, 55, 26,
        1, 63, 15, 24, 147, 176, 50, 20, 23, 176, 176, 182, 72, 208, 105, 138, 160, 243, 113, 123,
        106, 145, 39, 176, 0, 192, 137, 187, 182, 9, 5, 16, 72, 2, 121, 54, 224, 231, 208, 116,
        144, 55, 134, 20, 235, 123, 80, 105, 37, 196, 166, 51, 26, 151, 84, 52, 120, 162, 194, 97,
        212, 242, 153, 205, 112, 158, 151, 244, 242, 79, 219, 128, 99, 246, 32, 38, 132, 34, 131,
        28, 117, 162, 68, 107, 21, 25, 68, 66, 203, 7, 179, 46, 119, 8, 31, 36, 239, 179, 140, 51,
        221, 151, 200, 68, 183, 132, 36, 171, 247, 80, 191, 125, 22, 210, 204, 83, 200, 103, 97,
        184, 61, 225, 127, 73, 104, 159, 172, 185, 202, 116, 154, 28, 150, 6,
    ];

    // a random modulus
    let modulus = [
        186, 147, 169, 139, 184, 41, 41, 80, 108, 1, 29, 237, 140, 15, 147, 98, 8, 192, 145, 125,
        112, 68, 139, 38, 31, 131, 18, 213, 128, 28, 239, 126, 50, 54, 178, 48, 110, 200, 255, 10,
        206, 143, 168, 51, 43, 18, 24, 80, 166, 137, 197, 251, 215, 163, 248, 230, 16, 81, 164, 40,
        118, 207, 12, 149, 30, 55, 8, 135, 56, 9, 118, 228, 18, 190, 224, 181, 78, 70, 141, 109,
        114, 123, 200, 223, 241, 143, 249, 55, 171, 184, 16, 113, 63, 95, 194, 141, 196, 90, 118,
        203, 194, 73, 229, 187, 126, 241, 137, 9, 58, 68, 205, 95, 6, 254, 191, 224, 240, 85, 254,
        200, 6, 210, 24, 1, 32, 131, 185, 210, 109, 71, 116, 58, 213, 187, 115, 210, 107, 65, 116,
        172, 131, 209, 45, 65, 144, 7, 82, 199, 187, 97, 202, 5, 47, 159, 80, 235, 221, 89, 102,
        236, 123, 243, 25, 133, 28, 19, 140, 117, 77, 214, 127, 208, 94, 18, 24, 166, 38, 101, 165,
        144, 183, 109, 78, 100, 128, 67, 215, 253, 248, 244, 242, 182, 219, 26, 135, 24, 192, 149,
        247, 65, 206, 203, 28, 66, 205, 95, 65, 190, 163, 68, 24, 142, 228, 152, 231, 136, 169,
        197, 37, 124, 201, 235, 127, 51, 7, 197, 221, 161, 243, 212, 204, 128, 76, 52, 58, 3, 60,
        158, 162, 140, 51, 233, 167, 231, 190, 227, 111, 82, 210, 48, 79, 232, 80, 202, 114, 161,
        230, 250, 197, 122, 65, 90, 91, 110, 7, 106, 55, 170, 156, 118, 133, 101, 248, 234, 156,
        106, 228, 193, 33, 74, 82, 188, 205, 42, 2, 187, 75, 213, 33, 67, 206, 15, 163, 25, 72, 32,
        192, 92, 239, 40, 15, 116, 155, 120, 213, 50, 142, 47, 161, 30, 44, 213, 100, 217, 213,
        159, 190, 230, 163, 63, 152, 119, 190, 42, 154, 81, 39, 70, 179, 101, 169, 74, 37, 143, 49,
        134, 99, 130, 101, 87, 111, 240, 38, 240, 164, 134, 162, 124, 109, 161, 181, 192, 42, 57,
        240, 133, 18, 215, 175, 164, 45, 150, 161, 30, 104, 217, 68, 3, 5, 57, 170, 148, 24, 116,
        96, 172, 23, 139, 23, 230, 127, 182, 178, 167, 100, 46, 182, 190, 176, 4, 245, 103, 158,
        50, 96, 24, 27, 107, 135, 46, 122, 17, 225, 183, 0, 58, 3, 243, 168, 84, 64, 34, 29, 39,
        17, 227, 240, 30, 150, 28, 76, 11, 86, 53, 143, 25, 164, 215, 164, 70, 71, 232, 195, 95,
        221, 117, 135, 3, 241, 40, 207, 123, 122, 148, 118, 56, 65, 218, 8, 88, 124, 219, 91, 236,
        194, 245, 37, 246, 54, 202, 36, 151, 160, 40, 75, 92, 175, 241, 161, 113, 53, 220, 67, 249,
        7, 118, 77, 214, 122, 148, 44, 164, 249, 239, 57, 102, 45, 136, 93, 4, 76, 51, 186, 156,
        101, 82, 217, 116, 211, 255, 92, 183, 101, 50, 73, 38, 79, 219, 63, 140, 174, 218,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}

#[bench]
fn bench_big_num_exponentiation_exp_4096_bits_mod_4096_bits_even(b: &mut Bencher) {
    // a random exponent
    let exponent = [
        160, 136, 220, 27, 39, 209, 128, 184, 213, 36, 207, 49, 210, 27, 221, 106, 122, 123, 158,
        77, 226, 64, 168, 133, 129, 82, 217, 239, 209, 235, 234, 153, 175, 4, 59, 165, 18, 109, 42,
        228, 194, 227, 131, 189, 98, 154, 242, 164, 206, 128, 151, 139, 167, 129, 179, 1, 31, 20,
        62, 168, 118, 232, 139, 241, 174, 171, 180, 238, 21, 190, 206, 250, 115, 99, 66, 152, 176,
        110, 213, 251, 176, 158, 145, 38, 61, 121, 183, 157, 18, 8, 202, 154, 26, 198, 32, 252,
        213, 40, 31, 241, 234, 13, 97, 166, 12, 199, 215, 205, 64, 121, 192, 240, 168, 241, 224,
        86, 157, 194, 1, 3, 135, 99, 201, 95, 185, 193, 142, 218, 122, 250, 84, 90, 150, 146, 2,
        173, 55, 255, 166, 150, 196, 182, 97, 4, 161, 85, 162, 74, 230, 138, 154, 128, 100, 161,
        62, 19, 74, 36, 249, 111, 45, 13, 236, 140, 73, 123, 174, 114, 45, 133, 124, 150, 15, 148,
        24, 192, 29, 116, 90, 51, 215, 62, 176, 177, 23, 211, 67, 97, 19, 231, 83, 147, 140, 99,
        186, 233, 213, 100, 109, 27, 124, 193, 193, 2, 212, 2, 55, 104, 122, 170, 249, 228, 183,
        109, 141, 58, 63, 232, 54, 255, 218, 109, 149, 174, 74, 157, 63, 252, 249, 149, 173, 28,
        249, 12, 39, 64, 90, 179, 81, 210, 129, 14, 247, 248, 169, 243, 182, 74, 143, 236, 217,
        255, 201, 184, 228, 67, 254, 115, 187, 93, 171, 34, 62, 182, 218, 0, 237, 66, 224, 51, 124,
        75, 28, 149, 207, 3, 41, 2, 113, 49, 2, 147, 227, 101, 82, 13, 120, 75, 4, 67, 244, 28,
        123, 32, 81, 32, 224, 63, 51, 62, 121, 8, 62, 234, 179, 181, 91, 76, 123, 183, 247, 135,
        40, 212, 170, 236, 45, 164, 17, 239, 65, 202, 175, 31, 116, 141, 219, 209, 253, 40, 231,
        175, 115, 59, 199, 88, 149, 101, 10, 29, 25, 233, 171, 62, 132, 90, 175, 237, 237, 167,
        153, 5, 114, 103, 227, 79, 128, 89, 207, 142, 215, 104, 92, 236, 98, 61, 187, 233, 145,
        119, 209, 100, 4, 146, 11, 247, 106, 123, 73, 200, 120, 121, 161, 118, 214, 69, 17, 66, 29,
        170, 255, 100, 59, 53, 107, 10, 231, 72, 103, 11, 223, 36, 14, 243, 252, 181, 247, 80, 253,
        110, 84, 106, 132, 102, 231, 136, 96, 148, 246, 185, 249, 207, 49, 205, 171, 168, 150, 84,
        146, 144, 228, 95, 164, 160, 194, 124, 69, 58, 168, 59, 101, 32, 196, 167, 87, 108, 99,
        126, 157, 244, 227, 224, 29, 105, 73, 249, 45, 109, 62, 180, 234, 251, 53, 11, 72, 116, 46,
        18, 102, 59, 160, 138, 22, 117, 161, 104, 189, 21, 193, 31, 175, 178, 131, 47, 127, 146,
        209, 129, 96, 61, 43, 238, 88, 211, 79, 157, 85, 15, 57, 82, 54, 104, 104, 201, 171, 60,
        26, 1, 137, 90, 234, 249,
    ];

    // a random base
    let base = [
        167, 62, 26, 222, 69, 55, 136, 13, 139, 152, 251, 52, 101, 99, 181, 160, 55, 137, 73, 135,
        48, 71, 232, 168, 183, 14, 161, 145, 163, 22, 31, 14, 111, 65, 93, 52, 253, 42, 115, 167,
        211, 27, 143, 95, 20, 20, 179, 231, 241, 169, 110, 220, 42, 178, 222, 110, 97, 186, 76,
        210, 43, 205, 210, 166, 58, 173, 28, 10, 192, 102, 35, 122, 240, 244, 186, 197, 211, 39,
        230, 59, 140, 42, 221, 90, 81, 107, 88, 210, 1, 222, 215, 116, 7, 190, 251, 226, 87, 131,
        53, 137, 108, 245, 194, 62, 173, 26, 124, 253, 192, 10, 143, 10, 56, 237, 191, 216, 13, 6,
        22, 37, 130, 226, 241, 58, 157, 230, 247, 125, 170, 124, 167, 184, 148, 204, 247, 5, 181,
        73, 81, 195, 64, 72, 33, 187, 100, 173, 35, 86, 99, 228, 124, 83, 64, 1, 176, 178, 194,
        236, 110, 202, 70, 204, 59, 147, 232, 246, 152, 93, 124, 222, 192, 136, 149, 14, 36, 225,
        122, 228, 16, 215, 49, 80, 30, 230, 235, 158, 29, 86, 209, 67, 172, 163, 79, 22, 167, 113,
        154, 5, 157, 187, 237, 144, 76, 77, 45, 177, 15, 21, 19, 7, 181, 125, 232, 9, 138, 111,
        163, 207, 196, 197, 162, 67, 66, 81, 36, 3, 68, 68, 36, 213, 243, 42, 41, 12, 173, 71, 197,
        51, 217, 122, 98, 142, 79, 177, 34, 6, 160, 189, 158, 248, 115, 238, 226, 10, 71, 179, 92,
        87, 14, 100, 13, 220, 105, 238, 238, 0, 229, 100, 96, 29, 18, 155, 53, 247, 10, 146, 252,
        2, 219, 96, 206, 225, 171, 59, 135, 177, 90, 174, 245, 3, 222, 95, 35, 1, 31, 107, 53, 99,
        117, 93, 207, 148, 62, 12, 140, 211, 0, 122, 39, 66, 21, 71, 197, 137, 97, 142, 240, 61,
        69, 25, 121, 165, 53, 40, 218, 248, 112, 213, 8, 91, 217, 236, 173, 227, 103, 250, 133,
        143, 0, 227, 117, 159, 69, 8, 52, 147, 0, 175, 255, 76, 129, 47, 230, 234, 58, 233, 128, 2,
        184, 190, 104, 241, 69, 95, 74, 176, 112, 75, 236, 81, 100, 222, 111, 171, 126, 16, 55, 26,
        1, 63, 15, 24, 147, 176, 50, 20, 23, 176, 176, 182, 72, 208, 105, 138, 160, 243, 113, 123,
        106, 145, 39, 176, 0, 192, 137, 187, 182, 9, 5, 16, 72, 2, 121, 54, 224, 231, 208, 116,
        144, 55, 134, 20, 235, 123, 80, 105, 37, 196, 166, 51, 26, 151, 84, 52, 120, 162, 194, 97,
        212, 242, 153, 205, 112, 158, 151, 244, 242, 79, 219, 128, 99, 246, 32, 38, 132, 34, 131,
        28, 117, 162, 68, 107, 21, 25, 68, 66, 203, 7, 179, 46, 119, 8, 31, 36, 239, 179, 140, 51,
        221, 151, 200, 68, 183, 132, 36, 171, 247, 80, 191, 125, 22, 210, 204, 83, 200, 103, 97,
        184, 61, 225, 127, 73, 104, 159, 172, 185, 202, 116, 154, 28, 150, 6,
    ];

    // a random modulus
    let modulus = [
        186, 147, 169, 139, 184, 41, 41, 80, 108, 1, 29, 237, 140, 15, 147, 98, 8, 192, 145, 125,
        112, 68, 139, 38, 31, 131, 18, 213, 128, 28, 239, 126, 50, 54, 178, 48, 110, 200, 255, 10,
        206, 143, 168, 51, 43, 18, 24, 80, 166, 137, 197, 251, 215, 163, 248, 230, 16, 81, 164, 40,
        118, 207, 12, 149, 30, 55, 8, 135, 56, 9, 118, 228, 18, 190, 224, 181, 78, 70, 141, 109,
        114, 123, 200, 223, 241, 143, 249, 55, 171, 184, 16, 113, 63, 95, 194, 141, 196, 90, 118,
        203, 194, 73, 229, 187, 126, 241, 137, 9, 58, 68, 205, 95, 6, 254, 191, 224, 240, 85, 254,
        200, 6, 210, 24, 1, 32, 131, 185, 210, 109, 71, 116, 58, 213, 187, 115, 210, 107, 65, 116,
        172, 131, 209, 45, 65, 144, 7, 82, 199, 187, 97, 202, 5, 47, 159, 80, 235, 221, 89, 102,
        236, 123, 243, 25, 133, 28, 19, 140, 117, 77, 214, 127, 208, 94, 18, 24, 166, 38, 101, 165,
        144, 183, 109, 78, 100, 128, 67, 215, 253, 248, 244, 242, 182, 219, 26, 135, 24, 192, 149,
        247, 65, 206, 203, 28, 66, 205, 95, 65, 190, 163, 68, 24, 142, 228, 152, 231, 136, 169,
        197, 37, 124, 201, 235, 127, 51, 7, 197, 221, 161, 243, 212, 204, 128, 76, 52, 58, 3, 60,
        158, 162, 140, 51, 233, 167, 231, 190, 227, 111, 82, 210, 48, 79, 232, 80, 202, 114, 161,
        230, 250, 197, 122, 65, 90, 91, 110, 7, 106, 55, 170, 156, 118, 133, 101, 248, 234, 156,
        106, 228, 193, 33, 74, 82, 188, 205, 42, 2, 187, 75, 213, 33, 67, 206, 15, 163, 25, 72, 32,
        192, 92, 239, 40, 15, 116, 155, 120, 213, 50, 142, 47, 161, 30, 44, 213, 100, 217, 213,
        159, 190, 230, 163, 63, 152, 119, 190, 42, 154, 81, 39, 70, 179, 101, 169, 74, 37, 143, 49,
        134, 99, 130, 101, 87, 111, 240, 38, 240, 164, 134, 162, 124, 109, 161, 181, 192, 42, 57,
        240, 133, 18, 215, 175, 164, 45, 150, 161, 30, 104, 217, 68, 3, 5, 57, 170, 148, 24, 116,
        96, 172, 23, 139, 23, 230, 127, 182, 178, 167, 100, 46, 182, 190, 176, 4, 245, 103, 158,
        50, 96, 24, 27, 107, 135, 46, 122, 17, 225, 183, 0, 58, 3, 243, 168, 84, 64, 34, 29, 39,
        17, 227, 240, 30, 150, 28, 76, 11, 86, 53, 143, 25, 164, 215, 164, 70, 71, 232, 195, 95,
        221, 117, 135, 3, 241, 40, 207, 123, 122, 148, 118, 56, 65, 218, 8, 88, 124, 219, 91, 236,
        194, 245, 37, 246, 54, 202, 36, 151, 160, 40, 75, 92, 175, 241, 161, 113, 53, 220, 67, 249,
        7, 118, 77, 214, 122, 148, 44, 164, 249, 239, 57, 102, 45, 136, 93, 4, 76, 51, 186, 156,
        101, 82, 217, 116, 211, 255, 92, 183, 101, 50, 73, 38, 79, 219, 63, 140, 174, 218,
    ];

    b.iter(|| {
        big_mod_exp(&base, &exponent, &modulus);
    });
}
