use {serde::Serialize, solana_pubkey::Pubkey, std::convert::TryFrom};

#[derive(Debug)]
pub struct Zones {
    pub version: i32,
    pub zones_count: u16,
    pub algo_id: u8,
    pub min_stake_threshold: u8,
    pub start_from_midnight_seconds: u32,
    pub reserved: u32,
    pub entries: Vec<ZoneEntry>,
}

impl Zones {
    pub const HEADER_SIZE: usize = 16;
}

#[derive(Debug)]
pub struct ZoneEntry {
    pub zone_name: [char; 4],
    pub duration_ms: i32,
}

impl TryFrom<&[u8]> for Zones {
    type Error = &'static str;

    fn try_from(data: &[u8]) -> Result<Self, Self::Error> {
        if data.len() < Zones::HEADER_SIZE {
            return Err("Data buffer too small for Zone header");
        }

        let version = i32::from_le_bytes(data[0..4].try_into().unwrap());
        let zones_count = u16::from_le_bytes(data[4..6].try_into().unwrap());
        let algo_id = data[6];
        let min_stake_threshold = data[7];
        let start_from_midnight_seconds = u32::from_le_bytes(data[8..12].try_into().unwrap());
        let reserved = u32::from_le_bytes(data[12..16].try_into().unwrap());

        let mut entries = Vec::new();
        let mut offset = Zones::HEADER_SIZE;

        for _ in 0..zones_count {
            if offset + 8 > data.len() {
                return Err("Data buffer too small for ZoneEntry");
            }

            let zone_name = [
                data[offset] as char,
                data[offset + 1] as char,
                data[offset + 2] as char,
                data[offset + 3] as char,
            ];

            let duration_ms = i32::from_le_bytes(data[offset + 4..offset + 8].try_into().unwrap());

            entries.push(ZoneEntry {
                zone_name,
                duration_ms,
            });

            offset += 8;
        }

        Ok(Zones {
            version,
            zones_count,
            algo_id,
            min_stake_threshold,
            start_from_midnight_seconds,
            reserved,
            entries,
        })
    }
}

#[derive(Debug)]
pub struct ZoneValidators {
    pub version: i32,
    pub zone_count: u32,
    pub reserved: i32,
    pub entries: Vec<ZoneValidatorEntry>,
}

impl ZoneValidators {
    pub const HEADER_SIZE: usize = 12;
}

#[derive(Debug)]
pub struct ZoneValidatorEntry {
    pub zone_name: [char; 4],
    pub validator_count: u32,
    pub validator_pubkeys: Vec<Pubkey>,
}

impl TryFrom<&[u8]> for ZoneValidators {
    type Error = &'static str;

    fn try_from(data: &[u8]) -> Result<Self, Self::Error> {
        if data.len() < ZoneValidators::HEADER_SIZE {
            return Err("Data buffer too small for ZoneValidators header");
        }

        let version = i32::from_le_bytes(data[0..4].try_into().unwrap());
        let zone_count = u32::from_le_bytes(data[4..8].try_into().unwrap());
        let reserved = i32::from_le_bytes(data[8..12].try_into().unwrap());

        let mut entries = Vec::new();
        let mut offset = ZoneValidators::HEADER_SIZE;

        for _ in 0..zone_count {
            if offset + 8 > data.len() {
                return Err("Data buffer too small for ZoneValidatorEntry header");
            }

            let zone_name = [
                data[offset] as char,
                data[offset + 1] as char,
                data[offset + 2] as char,
                data[offset + 3] as char,
            ];

            let validator_count =
                u32::from_le_bytes(data[offset + 4..offset + 8].try_into().unwrap());
            offset += 8;

            if offset + (validator_count as usize * 32) > data.len() {
                return Err("Data buffer too small for validator pubkeys");
            }

            let mut validator_pubkeys = Vec::with_capacity(validator_count as usize);
            for _ in 0..validator_count {
                let pubkey = Pubkey::new_from_array(data[offset..offset + 32].try_into().unwrap());
                validator_pubkeys.push(pubkey);
                offset += 32;
            }

            entries.push(ZoneValidatorEntry {
                zone_name,
                validator_count,
                validator_pubkeys,
            });
        }

        Ok(ZoneValidators {
            version,
            zone_count,
            reserved,
            entries,
        })
    }
}

#[derive(Debug, Serialize)]
pub struct ZoneConfig {
    pub name: String,
    pub duration_ms: i32,
    pub validators: Vec<Pubkey>,
}

#[derive(Debug, Serialize, Deserialize, Default, Clone)]
#[cfg_attr(feature = "dev-context-only-utils", derive(PartialEq))]
pub struct ZoneState {
    pub name: String,
    pub duration_ms: i32,
    pub validators: Vec<Pubkey>,
    pub epoch: u64,
    pub algo_id: u8,
}

impl std::fmt::Display for ZoneState {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(
            f,
            "ZoneState {{ name: \"{}\", duration: {} s, validators: [{}] epoch: {} algo_id: {} }}",
            self.name,
            self.duration_ms / 1000,
            self.validators
                .iter()
                .map(|pk| pk.to_string())
                .collect::<Vec<_>>()
                .join(", "),
            self.epoch,
            self.algo_id,
        )
    }
}
