use fogo_sessions_sdk::session::Session;
use solana_instruction::error::InstructionError;
use solana_log_collector::ic_msg;
use solana_program_runtime::invoke_context::InvokeContext;
use solana_pubkey::{Pubkey, PubkeyError};
use solana_transaction_context::BorrowedAccount;

const TOKEN_PROGRAM: Pubkey = Pubkey::from_str_const("TokenkegQfeZyiNwAJbNbGKPFXCWuBvf9Ss623VQ5DA");
const ATA_PROGRAM: Pubkey = Pubkey::from_str_const("ATokenGPvbdGVxr1b2hvZbsiqW5xWH25efTNsLJA8knL");
const NATIVE_MINT: Pubkey = Pubkey::from_str_const("So11111111111111111111111111111111111111112");

const CREATE_PDA_COST: u64 = 1500;

/// find_program_address but with progressive cost tracking, required to ensure worst case scenario attacks
/// can't create transactions with the fix budget that are 254x more expensive
fn cost_tracked_find_program_address(
    invoke_context: &InvokeContext,
    seeds: &[&[u8]],
    program_id: &Pubkey,
) -> Result<(Pubkey, u8), InstructionError> {
    let mut bump_seed = [u8::MAX];
    for _ in 0..u8::MAX {
        {
            invoke_context
                .consume_checked(CREATE_PDA_COST)
                .map_err(|_| InstructionError::ComputationalBudgetExceeded)?;
            let mut seeds_with_bump = seeds.to_vec();
            seeds_with_bump.push(&bump_seed);
            match Pubkey::create_program_address(&seeds_with_bump, program_id) {
                Ok(address) => return Ok((address, bump_seed[0])),
                Err(PubkeyError::InvalidSeeds) => (),
                _ => break,
            }
        }
        bump_seed[0] -= 1;
    }

    Err(InstructionError::ProgramFailedToComplete)
}

/// Validates and checks the given ATA account is the correct derived address for a given wallet
fn validate_ata_address(
    invoke_context: &InvokeContext,
    from_account: &BorrowedAccount,
    to_account: &BorrowedAccount,
) -> Result<(), InstructionError> {
    let (ata_address, _) = cost_tracked_find_program_address(
        invoke_context,
        &[
            &from_account.get_key().to_bytes(),
            &TOKEN_PROGRAM.to_bytes(),
            &NATIVE_MINT.to_bytes(),
        ],
        &ATA_PROGRAM,
    )?;

    // Destination account address matches native token ATA account
    if to_account.get_key() != &ata_address {
        return Err(InstructionError::IncorrectAuthority);
    }

    Ok(())
}

/// Validates the wrap transfer between from and to accounts, authenticated via the given session account
pub fn validate_session_wrap(
    session_account: BorrowedAccount,
    from_account: BorrowedAccount,
    to_account: BorrowedAccount,
    invoke_context: &InvokeContext,
) -> Result<(), InstructionError> {
    // check to see if session account is owned by the session manager
    if session_account.get_owner() != &fogo_sessions_sdk::session::SESSION_MANAGER_ID {
        ic_msg!(
            invoke_context,
            "Session account owner {} does not match expected session manager address {}",
            session_account.get_owner(),
            fogo_sessions_sdk::session::SESSION_MANAGER_ID
        );
        return Err(InstructionError::InvalidAccountOwner);
    }

    let buf = &mut session_account.get_data();
    // parse session from account
    let session_data = Session::try_deserialize(buf).map_err(|_| {
        ic_msg!(
            invoke_context,
            "Session account {} does not deserialize to a Session",
            session_account.get_key(),
        );
        InstructionError::InvalidAccountData
    })?;

    let clock = invoke_context.get_sysvar_cache().get_clock()?;

    // checks: session liveness and revocation, returning the user
    let checked_user = session_data
        .get_user_checked_system_program(&clock)
        .map_err(|err| {
            ic_msg!(invoke_context, "Session validation error occurred: {}", err,);
            InstructionError::Custom(err.into())
        })?;

    // the user matches the funding account
    if &checked_user != from_account.get_key() {
        ic_msg!(
            invoke_context,
            "Session account user {} does not match source account {}",
            checked_user,
            from_account.get_key()
        );
        return Err(InstructionError::IncorrectAuthority);
    }

    // check to account which must be the Token ATA account for the given user and the native mint
    validate_ata_address(invoke_context, &from_account, &to_account).inspect_err(|_| {
        ic_msg!(
            invoke_context,
            "Session account user {} does not own destination token account {}",
            checked_user,
            to_account.get_key(),
        );
    })?;

    // all checks passed, session authorizes the funding account
    Ok(())
}
