//! Alt_bn128 Syscalls tests

use {
    solana_bn254::prelude::{alt_bn128_addition, alt_bn128_multiplication, alt_bn128_pairing},
    solana_msg::msg,
    solana_program_entrypoint::{custom_heap_default, custom_panic_default},
};

fn alt_bn128_addition_test() {
    // chfast1
    let result = alt_bn128_addition(&[
        0x18, 0xb1, 0x8a, 0xcf, 0xb4, 0xc2, 0xc3, 0x2, 0x76, 0xdb, 0x54, 0x11, 0x36, 0x8e, 0x71,
        0x85, 0xb3, 0x11, 0xdd, 0x12, 0x46, 0x91, 0x61, 0xc, 0x5d, 0x3b, 0x74, 0x3, 0x4e, 0x9,
        0x3d, 0xc9, 0x6, 0x3c, 0x90, 0x9c, 0x47, 0x20, 0x84, 0xc, 0xb5, 0x13, 0x4c, 0xb9, 0xf5,
        0x9f, 0xa7, 0x49, 0x75, 0x57, 0x96, 0x81, 0x96, 0x58, 0xd3, 0x2e, 0xfc, 0xd, 0x28, 0x81,
        0x98, 0xf3, 0x72, 0x66, 0x7, 0xc2, 0xb7, 0xf5, 0x8a, 0x84, 0xbd, 0x61, 0x45, 0xf0, 0xc,
        0x9c, 0x2b, 0xc0, 0xbb, 0x1a, 0x18, 0x7f, 0x20, 0xff, 0x2c, 0x92, 0x96, 0x3a, 0x88, 0x1,
        0x9e, 0x7c, 0x6a, 0x1, 0x4e, 0xed, 0x6, 0x61, 0x4e, 0x20, 0xc1, 0x47, 0xe9, 0x40, 0xf2,
        0xd7, 0xd, 0xa3, 0xf7, 0x4c, 0x9a, 0x17, 0xdf, 0x36, 0x17, 0x6, 0xa4, 0x48, 0x5c, 0x74,
        0x2b, 0xd6, 0x78, 0x84, 0x78, 0xfa, 0x17, 0xd7,
    ]); // 18b18acfb4c2c30276db5411368e7185b311dd124691610c5d3b74034e093dc9063c909c4720840cb5134cb9f59fa749755796819658d32efc0d288198f3726607c2b7f58a84bd6145f00c9c2bc0bb1a187f20ff2c92963a88019e7c6a014eed06614e20c147e940f2d70da3f74c9a17df361706a4485c742bd6788478fa17d7
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x22, 0x43, 0x52, 0x5c, 0x5e, 0xfd, 0x4b, 0x9c, 0x3d, 0x3c, 0x45, 0xac, 0xc, 0xa3,
            0xfe, 0x4d, 0xd8, 0x5e, 0x83, 0xa, 0x4c, 0xe6, 0xb6, 0x5f, 0xa1, 0xee, 0xae, 0xe2, 0x2,
            0x83, 0x97, 0x3, 0x30, 0x1d, 0x1d, 0x33, 0xbe, 0x6d, 0xa8, 0xe5, 0x9, 0xdf, 0x21, 0xcc,
            0x35, 0x96, 0x47, 0x23, 0x18, 0xe, 0xed, 0x75, 0x32, 0x53, 0x7d, 0xb9, 0xae, 0x5e,
            0x7d, 0x48, 0xf1, 0x95, 0xc9, 0x15
        ]
    ); // 2243525c5efd4b9c3d3c45ac0ca3fe4dd85e830a4ce6b65fa1eeaee202839703301d1d33be6da8e509df21cc35964723180eed7532537db9ae5e7d48f195c915

    // chfast2
    let result = alt_bn128_addition(&[
        0x22, 0x43, 0x52, 0x5c, 0x5e, 0xfd, 0x4b, 0x9c, 0x3d, 0x3c, 0x45, 0xac, 0xc, 0xa3, 0xfe,
        0x4d, 0xd8, 0x5e, 0x83, 0xa, 0x4c, 0xe6, 0xb6, 0x5f, 0xa1, 0xee, 0xae, 0xe2, 0x2, 0x83,
        0x97, 0x3, 0x30, 0x1d, 0x1d, 0x33, 0xbe, 0x6d, 0xa8, 0xe5, 0x9, 0xdf, 0x21, 0xcc, 0x35,
        0x96, 0x47, 0x23, 0x18, 0xe, 0xed, 0x75, 0x32, 0x53, 0x7d, 0xb9, 0xae, 0x5e, 0x7d, 0x48,
        0xf1, 0x95, 0xc9, 0x15, 0x18, 0xb1, 0x8a, 0xcf, 0xb4, 0xc2, 0xc3, 0x2, 0x76, 0xdb, 0x54,
        0x11, 0x36, 0x8e, 0x71, 0x85, 0xb3, 0x11, 0xdd, 0x12, 0x46, 0x91, 0x61, 0xc, 0x5d, 0x3b,
        0x74, 0x3, 0x4e, 0x9, 0x3d, 0xc9, 0x6, 0x3c, 0x90, 0x9c, 0x47, 0x20, 0x84, 0xc, 0xb5, 0x13,
        0x4c, 0xb9, 0xf5, 0x9f, 0xa7, 0x49, 0x75, 0x57, 0x96, 0x81, 0x96, 0x58, 0xd3, 0x2e, 0xfc,
        0xd, 0x28, 0x81, 0x98, 0xf3, 0x72, 0x66,
    ]); // 2243525c5efd4b9c3d3c45ac0ca3fe4dd85e830a4ce6b65fa1eeaee202839703301d1d33be6da8e509df21cc35964723180eed7532537db9ae5e7d48f195c91518b18acfb4c2c30276db5411368e7185b311dd124691610c5d3b74034e093dc9063c909c4720840cb5134cb9f59fa749755796819658d32efc0d288198f37266
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x2b, 0xd3, 0xe6, 0xd0, 0xf3, 0xb1, 0x42, 0x92, 0x4f, 0x5c, 0xa7, 0xb4, 0x9c, 0xe5,
            0xb9, 0xd5, 0x4c, 0x47, 0x3, 0xd7, 0xae, 0x56, 0x48, 0xe6, 0x1d, 0x2, 0x26, 0x8b, 0x1a,
            0xa, 0x9f, 0xb7, 0x21, 0x61, 0x1c, 0xe0, 0xa6, 0xaf, 0x85, 0x91, 0x5e, 0x2f, 0x1d,
            0x70, 0x30, 0x9, 0x9, 0xce, 0x2e, 0x49, 0xdf, 0xad, 0x4a, 0x46, 0x19, 0xc8, 0x39, 0xc,
            0xae, 0x66, 0xce, 0xfd, 0xb2, 0x4
        ]
    ); // 2bd3e6d0f3b142924f5ca7b49ce5b9d54c4703d7ae5648e61d02268b1a0a9fb721611ce0a6af85915e2f1d70300909ce2e49dfad4a4619c8390cae66cefdb204

    // cdetrio1
    let result = alt_bn128_addition(&[
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0,
    ]); // 0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0
        ]
    ); // 00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000

    // cdetrio2
    let result = alt_bn128_addition(&[
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
    ]); // 00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0
        ]
    ); // 00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000

    // cdetrio3
    let result = alt_bn128_addition(&[
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
    ]); // 0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0
        ]
    ); // 00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000

    // cdetrio4
    let result = alt_bn128_addition(&[]); // empty
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0
        ]
    ); // 00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000

    // cdetrio5
    let result = alt_bn128_addition(&[
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x2,
    ]); // 0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000000000002
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2
        ]
    ); // 00000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000000000002

    // cdetrio6
    let result = alt_bn128_addition(&[
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2,
    ]); // 00000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000000000002
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2
        ]
    ); // 00000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000000000002

    // cdetrio7
    let result = alt_bn128_addition(&[
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0,
    ]); // 0000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2
        ]
    ); // 00000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000000000002

    // cdetrio8
    let result = alt_bn128_addition(&[
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x2,
    ]); // 0000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000000000002
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x3, 0x6, 0x44, 0xe7, 0x2e, 0x13, 0x1a, 0x2, 0x9b, 0x85, 0x4, 0x5b, 0x68, 0x18, 0x15,
            0x85, 0xd9, 0x78, 0x16, 0xa9, 0x16, 0x87, 0x1c, 0xa8, 0xd3, 0xc2, 0x8, 0xc1, 0x6d,
            0x87, 0xcf, 0xd3, 0x15, 0xed, 0x73, 0x8c, 0xe, 0xa, 0x7c, 0x92, 0xe7, 0x84, 0x5f, 0x96,
            0xb2, 0xae, 0x9c, 0xa, 0x68, 0xa6, 0xa4, 0x49, 0xe3, 0x53, 0x8f, 0xc7, 0xff, 0x3e,
            0xbf, 0x7a, 0x5a, 0x18, 0xa2, 0xc4
        ]
    ); // 030644e72e131a029b85045b68181585d97816a916871ca8d3c208c16d87cfd315ed738c0e0a7c92e7845f96b2ae9c0a68a6a449e3538fc7ff3ebf7a5a18a2c4

    // cdetrio9
    let result = alt_bn128_addition(&[
        0x17, 0xc1, 0x39, 0xdf, 0xe, 0xfe, 0xe0, 0xf7, 0x66, 0xbc, 0x2, 0x4, 0x76, 0x2b, 0x77,
        0x43, 0x62, 0xe4, 0xde, 0xd8, 0x89, 0x53, 0xa3, 0x9c, 0xe8, 0x49, 0xa8, 0xa7, 0xfa, 0x16,
        0x3f, 0xa9, 0x1, 0xe0, 0x55, 0x9b, 0xac, 0xb1, 0x60, 0x66, 0x47, 0x64, 0xa3, 0x57, 0xaf,
        0x8a, 0x9f, 0xe7, 0xb, 0xaa, 0x92, 0x58, 0xe0, 0xb9, 0x59, 0x27, 0x3f, 0xfc, 0x57, 0x18,
        0xc6, 0xd4, 0xcc, 0x7c, 0x3, 0x97, 0x30, 0xea, 0x8d, 0xff, 0x12, 0x54, 0xc0, 0xfe, 0xe9,
        0xc0, 0xea, 0x77, 0x7d, 0x29, 0xa9, 0xc7, 0x10, 0xb7, 0xe6, 0x16, 0x68, 0x3f, 0x19, 0x4f,
        0x18, 0xc4, 0x3b, 0x43, 0xb8, 0x69, 0x7, 0x3a, 0x5f, 0xfc, 0xc6, 0xfc, 0x7a, 0x28, 0xc3,
        0x7, 0x23, 0xd6, 0xe5, 0x8c, 0xe5, 0x77, 0x35, 0x69, 0x82, 0xd6, 0x5b, 0x83, 0x3a, 0x5a,
        0x5c, 0x15, 0xbf, 0x90, 0x24, 0xb4, 0x3d, 0x98,
    ]); // 17c139df0efee0f766bc0204762b774362e4ded88953a39ce849a8a7fa163fa901e0559bacb160664764a357af8a9fe70baa9258e0b959273ffc5718c6d4cc7c039730ea8dff1254c0fee9c0ea777d29a9c710b7e616683f194f18c43b43b869073a5ffcc6fc7a28c30723d6e58ce577356982d65b833a5a5c15bf9024b43d98
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x15, 0xbf, 0x2b, 0xb1, 0x78, 0x80, 0x14, 0x4b, 0x5d, 0x1c, 0xd2, 0xb1, 0xf4, 0x6e,
            0xff, 0x9d, 0x61, 0x7b, 0xff, 0xd1, 0xca, 0x57, 0xc3, 0x7f, 0xb5, 0xa4, 0x9b, 0xd8,
            0x4e, 0x53, 0xcf, 0x66, 0x4, 0x9c, 0x79, 0x7f, 0x9c, 0xe0, 0xd1, 0x70, 0x83, 0xde,
            0xb3, 0x2b, 0x5e, 0x36, 0xf2, 0xea, 0x2a, 0x21, 0x2e, 0xe0, 0x36, 0x59, 0x8d, 0xd7,
            0x62, 0x4c, 0x16, 0x89, 0x93, 0xd1, 0x35, 0x5f
        ]
    ); // 15bf2bb17880144b5d1cd2b1f46eff9d617bffd1ca57c37fb5a49bd84e53cf66049c797f9ce0d17083deb32b5e36f2ea2a212ee036598dd7624c168993d1355f
}

fn alt_bn128_multiplication_test() {
    // chfast1
    let result = alt_bn128_multiplication(&[
        0x2b, 0xd3, 0xe6, 0xd0, 0xf3, 0xb1, 0x42, 0x92, 0x4f, 0x5c, 0xa7, 0xb4, 0x9c, 0xe5, 0xb9,
        0xd5, 0x4c, 0x47, 0x3, 0xd7, 0xae, 0x56, 0x48, 0xe6, 0x1d, 0x2, 0x26, 0x8b, 0x1a, 0xa,
        0x9f, 0xb7, 0x21, 0x61, 0x1c, 0xe0, 0xa6, 0xaf, 0x85, 0x91, 0x5e, 0x2f, 0x1d, 0x70, 0x30,
        0x9, 0x9, 0xce, 0x2e, 0x49, 0xdf, 0xad, 0x4a, 0x46, 0x19, 0xc8, 0x39, 0xc, 0xae, 0x66,
        0xce, 0xfd, 0xb2, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x11, 0x13, 0x8c, 0xe7, 0x50, 0xfa,
        0x15, 0xc2,
    ]); // 2bd3e6d0f3b142924f5ca7b49ce5b9d54c4703d7ae5648e61d02268b1a0a9fb721611ce0a6af85915e2f1d70300909ce2e49dfad4a4619c8390cae66cefdb20400000000000000000000000000000000000000000000000011138ce750fa15c2
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x7, 0xa, 0x8d, 0x6a, 0x98, 0x21, 0x53, 0xca, 0xe4, 0xbe, 0x29, 0xd4, 0x34, 0xe8, 0xfa,
            0xef, 0x8a, 0x47, 0xb2, 0x74, 0xa0, 0x53, 0xf5, 0xa4, 0xee, 0x2a, 0x6c, 0x9c, 0x13,
            0xc3, 0x1e, 0x5c, 0x3, 0x1b, 0x8c, 0xe9, 0x14, 0xeb, 0xa3, 0xa9, 0xff, 0xb9, 0x89,
            0xf9, 0xcd, 0xd5, 0xb0, 0xf0, 0x19, 0x43, 0x7, 0x4b, 0xf4, 0xf0, 0xf3, 0x15, 0x69, 0xe,
            0xc3, 0xce, 0xc6, 0x98, 0x1a, 0xfc
        ]
    ); // 070a8d6a982153cae4be29d434e8faef8a47b274a053f5a4ee2a6c9c13c31e5c031b8ce914eba3a9ffb989f9cdd5b0f01943074bf4f0f315690ec3cec6981afc

    // chfast2
    let result = alt_bn128_multiplication(&[
        0x7, 0xa, 0x8d, 0x6a, 0x98, 0x21, 0x53, 0xca, 0xe4, 0xbe, 0x29, 0xd4, 0x34, 0xe8, 0xfa,
        0xef, 0x8a, 0x47, 0xb2, 0x74, 0xa0, 0x53, 0xf5, 0xa4, 0xee, 0x2a, 0x6c, 0x9c, 0x13, 0xc3,
        0x1e, 0x5c, 0x3, 0x1b, 0x8c, 0xe9, 0x14, 0xeb, 0xa3, 0xa9, 0xff, 0xb9, 0x89, 0xf9, 0xcd,
        0xd5, 0xb0, 0xf0, 0x19, 0x43, 0x7, 0x4b, 0xf4, 0xf0, 0xf3, 0x15, 0x69, 0xe, 0xc3, 0xce,
        0xc6, 0x98, 0x1a, 0xfc, 0x30, 0x64, 0x4e, 0x72, 0xe1, 0x31, 0xa0, 0x29, 0xb8, 0x50, 0x45,
        0xb6, 0x81, 0x81, 0x58, 0x5d, 0x97, 0x81, 0x6a, 0x91, 0x68, 0x71, 0xca, 0x8d, 0x3c, 0x20,
        0x8c, 0x16, 0xd8, 0x7c, 0xfd, 0x46,
    ]); // 070a8d6a982153cae4be29d434e8faef8a47b274a053f5a4ee2a6c9c13c31e5c031b8ce914eba3a9ffb989f9cdd5b0f01943074bf4f0f315690ec3cec6981afc30644e72e131a029b85045b68181585d97816a916871ca8d3c208c16d87cfd46
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x2, 0x5a, 0x6f, 0x41, 0x81, 0xd2, 0xb4, 0xea, 0x8b, 0x72, 0x42, 0x90, 0xff, 0xb4, 0x1,
            0x56, 0xeb, 0xa, 0xdb, 0x51, 0x4c, 0x68, 0x85, 0x56, 0xeb, 0x79, 0xcd, 0xea, 0x7, 0x52,
            0xc2, 0xbb, 0x2e, 0xff, 0x3f, 0x31, 0xde, 0xa2, 0x15, 0xf1, 0xeb, 0x86, 0x2, 0x3a,
            0x13, 0x3a, 0x99, 0x6e, 0xb6, 0x30, 0xb, 0x44, 0xda, 0x66, 0x4d, 0x64, 0x25, 0x1d, 0x5,
            0x38, 0x1b, 0xb8, 0xa0, 0x2e
        ]
    ); // 025a6f4181d2b4ea8b724290ffb40156eb0adb514c688556eb79cdea0752c2bb2eff3f31dea215f1eb86023a133a996eb6300b44da664d64251d05381bb8a02e

    // chfast3
    let result = alt_bn128_multiplication(&[
        0x2, 0x5a, 0x6f, 0x41, 0x81, 0xd2, 0xb4, 0xea, 0x8b, 0x72, 0x42, 0x90, 0xff, 0xb4, 0x1,
        0x56, 0xeb, 0xa, 0xdb, 0x51, 0x4c, 0x68, 0x85, 0x56, 0xeb, 0x79, 0xcd, 0xea, 0x7, 0x52,
        0xc2, 0xbb, 0x2e, 0xff, 0x3f, 0x31, 0xde, 0xa2, 0x15, 0xf1, 0xeb, 0x86, 0x2, 0x3a, 0x13,
        0x3a, 0x99, 0x6e, 0xb6, 0x30, 0xb, 0x44, 0xda, 0x66, 0x4d, 0x64, 0x25, 0x1d, 0x5, 0x38,
        0x1b, 0xb8, 0xa0, 0x2e, 0x18, 0x32, 0x27, 0x39, 0x70, 0x98, 0xd0, 0x14, 0xdc, 0x28, 0x22,
        0xdb, 0x40, 0xc0, 0xac, 0x2e, 0xcb, 0xc0, 0xb5, 0x48, 0xb4, 0x38, 0xe5, 0x46, 0x9e, 0x10,
        0x46, 0xb, 0x6c, 0x3e, 0x7e, 0xa3,
    ]); // 025a6f4181d2b4ea8b724290ffb40156eb0adb514c688556eb79cdea0752c2bb2eff3f31dea215f1eb86023a133a996eb6300b44da664d64251d05381bb8a02e183227397098d014dc2822db40c0ac2ecbc0b548b438e5469e10460b6c3e7ea3
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x14, 0x78, 0x9d, 0xd, 0x4a, 0x73, 0xb, 0x35, 0x44, 0x3, 0xb5, 0xfa, 0xc9, 0x48, 0x11,
            0x37, 0x39, 0xe2, 0x76, 0xc2, 0x3e, 0x2, 0x58, 0xd8, 0x59, 0x6e, 0xe7, 0x2f, 0x9c,
            0xd9, 0xd3, 0x23, 0xa, 0xf1, 0x8a, 0x63, 0x15, 0x3e, 0xe, 0xc2, 0x5f, 0xf9, 0xf2, 0x95,
            0x1d, 0xd3, 0xfa, 0x90, 0xed, 0x1, 0x97, 0xbf, 0xef, 0x6e, 0x2a, 0x1a, 0x62, 0xb5, 0x9,
            0x5b, 0x9d, 0x2b, 0x4a, 0x27
        ]
    ); // 14789d0d4a730b354403b5fac948113739e276c23e0258d8596ee72f9cd9d3230af18a63153e0ec25ff9f2951dd3fa90ed0197bfef6e2a1a62b5095b9d2b4a27

    // cdetrio1
    let result = alt_bn128_multiplication(&[
        0x1a, 0x87, 0xb0, 0x58, 0x4c, 0xe9, 0x2f, 0x45, 0x93, 0xd1, 0x61, 0x48, 0x6, 0x14, 0xf2,
        0x98, 0x90, 0x35, 0x22, 0x56, 0x9, 0xf0, 0x80, 0x58, 0xcc, 0xfa, 0x3d, 0xf, 0x94, 0xf,
        0xeb, 0xe3, 0x1a, 0x2f, 0x3c, 0x95, 0x1f, 0x6d, 0xad, 0xcc, 0x7e, 0xe9, 0x0, 0x7d, 0xff,
        0x81, 0x50, 0x4b, 0xf, 0xcd, 0x6d, 0x7c, 0xf5, 0x99, 0x96, 0xef, 0xdc, 0x33, 0xd9, 0x2b,
        0xf7, 0xf9, 0xf8, 0xf6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    ]); // 1a87b0584ce92f4593d161480614f2989035225609f08058ccfa3d0f940febe31a2f3c951f6dadcc7ee9007dff81504b0fcd6d7cf59996efdc33d92bf7f9f8f6ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x2c, 0xde, 0x58, 0x79, 0xba, 0x6f, 0x13, 0xc0, 0xb5, 0xaa, 0x4e, 0xf6, 0x27, 0xf1,
            0x59, 0xa3, 0x34, 0x7d, 0xf9, 0x72, 0x2e, 0xfc, 0xe8, 0x8a, 0x9a, 0xfb, 0xb2, 0xb,
            0x76, 0x3b, 0x4c, 0x41, 0x1a, 0xa7, 0xe4, 0x30, 0x76, 0xf6, 0xae, 0xe2, 0x72, 0x75,
            0x5a, 0x7f, 0x9b, 0x84, 0x83, 0x2e, 0x71, 0x55, 0x9b, 0xa0, 0xd2, 0xe0, 0xb1, 0x7d,
            0x5f, 0x9f, 0x1, 0x75, 0x5e, 0x5b, 0xd, 0x11
        ]
    ); // 2cde5879ba6f13c0b5aa4ef627f159a3347df9722efce88a9afbb20b763b4c411aa7e43076f6aee272755a7f9b84832e71559ba0d2e0b17d5f9f01755e5b0d11

    // cdetrio2
    let result = alt_bn128_multiplication(&[
        0x1a, 0x87, 0xb0, 0x58, 0x4c, 0xe9, 0x2f, 0x45, 0x93, 0xd1, 0x61, 0x48, 0x6, 0x14, 0xf2,
        0x98, 0x90, 0x35, 0x22, 0x56, 0x9, 0xf0, 0x80, 0x58, 0xcc, 0xfa, 0x3d, 0xf, 0x94, 0xf,
        0xeb, 0xe3, 0x1a, 0x2f, 0x3c, 0x95, 0x1f, 0x6d, 0xad, 0xcc, 0x7e, 0xe9, 0x0, 0x7d, 0xff,
        0x81, 0x50, 0x4b, 0xf, 0xcd, 0x6d, 0x7c, 0xf5, 0x99, 0x96, 0xef, 0xdc, 0x33, 0xd9, 0x2b,
        0xf7, 0xf9, 0xf8, 0xf6, 0x30, 0x64, 0x4e, 0x72, 0xe1, 0x31, 0xa0, 0x29, 0xb8, 0x50, 0x45,
        0xb6, 0x81, 0x81, 0x58, 0x5d, 0x28, 0x33, 0xe8, 0x48, 0x79, 0xb9, 0x70, 0x91, 0x43, 0xe1,
        0xf5, 0x93, 0xf0, 0x0, 0x0, 0x0,
    ]); // 1a87b0584ce92f4593d161480614f2989035225609f08058ccfa3d0f940febe31a2f3c951f6dadcc7ee9007dff81504b0fcd6d7cf59996efdc33d92bf7f9f8f630644e72e131a029b85045b68181585d2833e84879b9709143e1f593f0000000
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x1a, 0x87, 0xb0, 0x58, 0x4c, 0xe9, 0x2f, 0x45, 0x93, 0xd1, 0x61, 0x48, 0x6, 0x14,
            0xf2, 0x98, 0x90, 0x35, 0x22, 0x56, 0x9, 0xf0, 0x80, 0x58, 0xcc, 0xfa, 0x3d, 0xf, 0x94,
            0xf, 0xeb, 0xe3, 0x16, 0x35, 0x11, 0xdd, 0xc1, 0xc3, 0xf2, 0x5d, 0x39, 0x67, 0x45,
            0x38, 0x82, 0x0, 0x8, 0x12, 0x87, 0xb3, 0xfd, 0x14, 0x72, 0xd8, 0x33, 0x9d, 0x5f, 0xec,
            0xb2, 0xea, 0xe0, 0x83, 0x4, 0x51
        ]
    ); // 1a87b0584ce92f4593d161480614f2989035225609f08058ccfa3d0f940febe3163511ddc1c3f25d396745388200081287b3fd1472d8339d5fecb2eae0830451

    // cdetrio3
    let result = alt_bn128_multiplication(&[
        0x1a, 0x87, 0xb0, 0x58, 0x4c, 0xe9, 0x2f, 0x45, 0x93, 0xd1, 0x61, 0x48, 0x6, 0x14, 0xf2,
        0x98, 0x90, 0x35, 0x22, 0x56, 0x9, 0xf0, 0x80, 0x58, 0xcc, 0xfa, 0x3d, 0xf, 0x94, 0xf,
        0xeb, 0xe3, 0x1a, 0x2f, 0x3c, 0x95, 0x1f, 0x6d, 0xad, 0xcc, 0x7e, 0xe9, 0x0, 0x7d, 0xff,
        0x81, 0x50, 0x4b, 0xf, 0xcd, 0x6d, 0x7c, 0xf5, 0x99, 0x96, 0xef, 0xdc, 0x33, 0xd9, 0x2b,
        0xf7, 0xf9, 0xf8, 0xf6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0,
    ]); // 1a87b0584ce92f4593d161480614f2989035225609f08058ccfa3d0f940febe31a2f3c951f6dadcc7ee9007dff81504b0fcd6d7cf59996efdc33d92bf7f9f8f60000000000000000000000000000000100000000000000000000000000000000
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x10, 0x51, 0xac, 0xb0, 0x70, 0xe, 0xc6, 0xd4, 0x2a, 0x88, 0x21, 0x58, 0x52, 0xd5,
            0x82, 0xef, 0xba, 0xef, 0x31, 0x52, 0x9b, 0x6f, 0xcb, 0xc3, 0x27, 0x7b, 0x5c, 0x1b,
            0x30, 0xf, 0x5c, 0xf0, 0x13, 0x5b, 0x23, 0x94, 0xbb, 0x45, 0xab, 0x4, 0xb8, 0xbd, 0x76,
            0x11, 0xbd, 0x2d, 0xfe, 0x1d, 0xe6, 0xa4, 0xe6, 0xe2, 0xcc, 0xea, 0x1e, 0xa1, 0x95,
            0x5f, 0x57, 0x7c, 0xd6, 0x6a, 0xf8, 0x5b
        ]
    ); // 1051acb0700ec6d42a88215852d582efbaef31529b6fcbc3277b5c1b300f5cf0135b2394bb45ab04b8bd7611bd2dfe1de6a4e6e2ccea1ea1955f577cd66af85b

    // cdetrio4
    let result = alt_bn128_multiplication(&[
        0x1a, 0x87, 0xb0, 0x58, 0x4c, 0xe9, 0x2f, 0x45, 0x93, 0xd1, 0x61, 0x48, 0x6, 0x14, 0xf2,
        0x98, 0x90, 0x35, 0x22, 0x56, 0x9, 0xf0, 0x80, 0x58, 0xcc, 0xfa, 0x3d, 0xf, 0x94, 0xf,
        0xeb, 0xe3, 0x1a, 0x2f, 0x3c, 0x95, 0x1f, 0x6d, 0xad, 0xcc, 0x7e, 0xe9, 0x0, 0x7d, 0xff,
        0x81, 0x50, 0x4b, 0xf, 0xcd, 0x6d, 0x7c, 0xf5, 0x99, 0x96, 0xef, 0xdc, 0x33, 0xd9, 0x2b,
        0xf7, 0xf9, 0xf8, 0xf6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x9,
    ]); // 1a87b0584ce92f4593d161480614f2989035225609f08058ccfa3d0f940febe31a2f3c951f6dadcc7ee9007dff81504b0fcd6d7cf59996efdc33d92bf7f9f8f60000000000000000000000000000000000000000000000000000000000000009
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x1d, 0xba, 0xd7, 0xd3, 0x9d, 0xbc, 0x56, 0x37, 0x9f, 0x78, 0xfa, 0xc1, 0xbc, 0xa1,
            0x47, 0xdc, 0x8e, 0x66, 0xde, 0x1b, 0x9d, 0x18, 0x3c, 0x7b, 0x16, 0x73, 0x51, 0xbf,
            0xe0, 0xae, 0xab, 0x74, 0x2c, 0xd7, 0x57, 0xd5, 0x12, 0x89, 0xcd, 0x8d, 0xbd, 0xa,
            0xcf, 0x9e, 0x67, 0x3a, 0xd6, 0x7d, 0xf, 0xa, 0x89, 0xf9, 0x12, 0xaf, 0x47, 0xed, 0x1b,
            0xe5, 0x36, 0x64, 0xf5, 0x69, 0x25, 0x75
        ]
    ); // 1dbad7d39dbc56379f78fac1bca147dc8e66de1b9d183c7b167351bfe0aeab742cd757d51289cd8dbd0acf9e673ad67d0f0a89f912af47ed1be53664f5692575

    // cdetrio5
    let result = alt_bn128_multiplication(&[
        0x1a, 0x87, 0xb0, 0x58, 0x4c, 0xe9, 0x2f, 0x45, 0x93, 0xd1, 0x61, 0x48, 0x6, 0x14, 0xf2,
        0x98, 0x90, 0x35, 0x22, 0x56, 0x9, 0xf0, 0x80, 0x58, 0xcc, 0xfa, 0x3d, 0xf, 0x94, 0xf,
        0xeb, 0xe3, 0x1a, 0x2f, 0x3c, 0x95, 0x1f, 0x6d, 0xad, 0xcc, 0x7e, 0xe9, 0x0, 0x7d, 0xff,
        0x81, 0x50, 0x4b, 0xf, 0xcd, 0x6d, 0x7c, 0xf5, 0x99, 0x96, 0xef, 0xdc, 0x33, 0xd9, 0x2b,
        0xf7, 0xf9, 0xf8, 0xf6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x1,
    ]); // 1a87b0584ce92f4593d161480614f2989035225609f08058ccfa3d0f940febe31a2f3c951f6dadcc7ee9007dff81504b0fcd6d7cf59996efdc33d92bf7f9f8f60000000000000000000000000000000000000000000000000000000000000001
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x1a, 0x87, 0xb0, 0x58, 0x4c, 0xe9, 0x2f, 0x45, 0x93, 0xd1, 0x61, 0x48, 0x6, 0x14,
            0xf2, 0x98, 0x90, 0x35, 0x22, 0x56, 0x9, 0xf0, 0x80, 0x58, 0xcc, 0xfa, 0x3d, 0xf, 0x94,
            0xf, 0xeb, 0xe3, 0x1a, 0x2f, 0x3c, 0x95, 0x1f, 0x6d, 0xad, 0xcc, 0x7e, 0xe9, 0x0, 0x7d,
            0xff, 0x81, 0x50, 0x4b, 0xf, 0xcd, 0x6d, 0x7c, 0xf5, 0x99, 0x96, 0xef, 0xdc, 0x33,
            0xd9, 0x2b, 0xf7, 0xf9, 0xf8, 0xf6
        ]
    ); // 1a87b0584ce92f4593d161480614f2989035225609f08058ccfa3d0f940febe31a2f3c951f6dadcc7ee9007dff81504b0fcd6d7cf59996efdc33d92bf7f9f8f6

    // cdetrio6
    let result = alt_bn128_multiplication(&[
        0x17, 0xc1, 0x39, 0xdf, 0xe, 0xfe, 0xe0, 0xf7, 0x66, 0xbc, 0x2, 0x4, 0x76, 0x2b, 0x77,
        0x43, 0x62, 0xe4, 0xde, 0xd8, 0x89, 0x53, 0xa3, 0x9c, 0xe8, 0x49, 0xa8, 0xa7, 0xfa, 0x16,
        0x3f, 0xa9, 0x1, 0xe0, 0x55, 0x9b, 0xac, 0xb1, 0x60, 0x66, 0x47, 0x64, 0xa3, 0x57, 0xaf,
        0x8a, 0x9f, 0xe7, 0xb, 0xaa, 0x92, 0x58, 0xe0, 0xb9, 0x59, 0x27, 0x3f, 0xfc, 0x57, 0x18,
        0xc6, 0xd4, 0xcc, 0x7c, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    ]); // 17c139df0efee0f766bc0204762b774362e4ded88953a39ce849a8a7fa163fa901e0559bacb160664764a357af8a9fe70baa9258e0b959273ffc5718c6d4cc7cffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x29, 0xe5, 0x87, 0xaa, 0xdd, 0x7c, 0x6, 0x72, 0x2a, 0xab, 0xba, 0x75, 0x30, 0x17,
            0xc0, 0x93, 0xf7, 0xb, 0xa7, 0xeb, 0x1f, 0x1c, 0x1, 0x4, 0xec, 0x5, 0x64, 0xe7, 0xe3,
            0xe2, 0x1f, 0x60, 0x22, 0xb1, 0x14, 0x3f, 0x6a, 0x41, 0x0, 0x8e, 0x77, 0x55, 0xc7,
            0x1c, 0x3d, 0x0, 0xb6, 0xb9, 0x15, 0xd3, 0x86, 0xde, 0x21, 0x78, 0x3e, 0xf5, 0x90,
            0x48, 0x6d, 0x8a, 0xfa, 0x84, 0x53, 0xb1
        ]
    ); // 29e587aadd7c06722aabba753017c093f70ba7eb1f1c0104ec0564e7e3e21f6022b1143f6a41008e7755c71c3d00b6b915d386de21783ef590486d8afa8453b1

    // cdetrio7
    let result = alt_bn128_multiplication(&[
        0x17, 0xc1, 0x39, 0xdf, 0xe, 0xfe, 0xe0, 0xf7, 0x66, 0xbc, 0x2, 0x4, 0x76, 0x2b, 0x77,
        0x43, 0x62, 0xe4, 0xde, 0xd8, 0x89, 0x53, 0xa3, 0x9c, 0xe8, 0x49, 0xa8, 0xa7, 0xfa, 0x16,
        0x3f, 0xa9, 0x1, 0xe0, 0x55, 0x9b, 0xac, 0xb1, 0x60, 0x66, 0x47, 0x64, 0xa3, 0x57, 0xaf,
        0x8a, 0x9f, 0xe7, 0xb, 0xaa, 0x92, 0x58, 0xe0, 0xb9, 0x59, 0x27, 0x3f, 0xfc, 0x57, 0x18,
        0xc6, 0xd4, 0xcc, 0x7c, 0x30, 0x64, 0x4e, 0x72, 0xe1, 0x31, 0xa0, 0x29, 0xb8, 0x50, 0x45,
        0xb6, 0x81, 0x81, 0x58, 0x5d, 0x28, 0x33, 0xe8, 0x48, 0x79, 0xb9, 0x70, 0x91, 0x43, 0xe1,
        0xf5, 0x93, 0xf0, 0x0, 0x0, 0x0,
    ]); // 17c139df0efee0f766bc0204762b774362e4ded88953a39ce849a8a7fa163fa901e0559bacb160664764a357af8a9fe70baa9258e0b959273ffc5718c6d4cc7c30644e72e131a029b85045b68181585d2833e84879b9709143e1f593f0000000
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x17, 0xc1, 0x39, 0xdf, 0xe, 0xfe, 0xe0, 0xf7, 0x66, 0xbc, 0x2, 0x4, 0x76, 0x2b, 0x77,
            0x43, 0x62, 0xe4, 0xde, 0xd8, 0x89, 0x53, 0xa3, 0x9c, 0xe8, 0x49, 0xa8, 0xa7, 0xfa,
            0x16, 0x3f, 0xa9, 0x2e, 0x83, 0xf8, 0xd7, 0x34, 0x80, 0x3f, 0xc3, 0x70, 0xeb, 0xa2,
            0x5e, 0xd1, 0xf6, 0xb8, 0x76, 0x8b, 0xd6, 0xd8, 0x38, 0x87, 0xb8, 0x71, 0x65, 0xfc,
            0x24, 0x34, 0xfe, 0x11, 0xa8, 0x30, 0xcb
        ]
    ); // 17c139df0efee0f766bc0204762b774362e4ded88953a39ce849a8a7fa163fa92e83f8d734803fc370eba25ed1f6b8768bd6d83887b87165fc2434fe11a830cb

    // cdetrio8
    let result = alt_bn128_multiplication(&[
        0x17, 0xc1, 0x39, 0xdf, 0xe, 0xfe, 0xe0, 0xf7, 0x66, 0xbc, 0x2, 0x4, 0x76, 0x2b, 0x77,
        0x43, 0x62, 0xe4, 0xde, 0xd8, 0x89, 0x53, 0xa3, 0x9c, 0xe8, 0x49, 0xa8, 0xa7, 0xfa, 0x16,
        0x3f, 0xa9, 0x1, 0xe0, 0x55, 0x9b, 0xac, 0xb1, 0x60, 0x66, 0x47, 0x64, 0xa3, 0x57, 0xaf,
        0x8a, 0x9f, 0xe7, 0xb, 0xaa, 0x92, 0x58, 0xe0, 0xb9, 0x59, 0x27, 0x3f, 0xfc, 0x57, 0x18,
        0xc6, 0xd4, 0xcc, 0x7c, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0,
    ]); // 17c139df0efee0f766bc0204762b774362e4ded88953a39ce849a8a7fa163fa901e0559bacb160664764a357af8a9fe70baa9258e0b959273ffc5718c6d4cc7c0000000000000000000000000000000100000000000000000000000000000000
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x22, 0x1a, 0x35, 0x77, 0x76, 0x38, 0x77, 0x92, 0xd, 0xd, 0x14, 0xa9, 0x1c, 0xd5, 0x9b,
            0x94, 0x79, 0xf8, 0x3b, 0x87, 0xa6, 0x53, 0xbb, 0x41, 0xf8, 0x2a, 0x3f, 0x6f, 0x12,
            0xc, 0xea, 0x7c, 0x27, 0x52, 0xc7, 0xf6, 0x4c, 0xdd, 0x7f, 0xe, 0x49, 0x4b, 0xff, 0x7b,
            0x60, 0x41, 0x9f, 0x24, 0x22, 0x10, 0xf2, 0x2, 0x6e, 0xd2, 0xec, 0x70, 0xf8, 0x9f,
            0x78, 0xa4, 0xc5, 0x6a, 0x1f, 0x15
        ]
    ); // 221a3577763877920d0d14a91cd59b9479f83b87a653bb41f82a3f6f120cea7c2752c7f64cdd7f0e494bff7b60419f242210f2026ed2ec70f89f78a4c56a1f15

    // cdetrio9
    let result = alt_bn128_multiplication(&[
        0x17, 0xc1, 0x39, 0xdf, 0xe, 0xfe, 0xe0, 0xf7, 0x66, 0xbc, 0x2, 0x4, 0x76, 0x2b, 0x77,
        0x43, 0x62, 0xe4, 0xde, 0xd8, 0x89, 0x53, 0xa3, 0x9c, 0xe8, 0x49, 0xa8, 0xa7, 0xfa, 0x16,
        0x3f, 0xa9, 0x1, 0xe0, 0x55, 0x9b, 0xac, 0xb1, 0x60, 0x66, 0x47, 0x64, 0xa3, 0x57, 0xaf,
        0x8a, 0x9f, 0xe7, 0xb, 0xaa, 0x92, 0x58, 0xe0, 0xb9, 0x59, 0x27, 0x3f, 0xfc, 0x57, 0x18,
        0xc6, 0xd4, 0xcc, 0x7c, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x9,
    ]); // 17c139df0efee0f766bc0204762b774362e4ded88953a39ce849a8a7fa163fa901e0559bacb160664764a357af8a9fe70baa9258e0b959273ffc5718c6d4cc7c0000000000000000000000000000000000000000000000000000000000000009
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x22, 0x8e, 0x68, 0x7a, 0x37, 0x9b, 0xa1, 0x54, 0x55, 0x40, 0x40, 0xf8, 0x82, 0x1f,
            0x4e, 0x41, 0xee, 0x2b, 0xe2, 0x87, 0xc2, 0x1, 0xaa, 0x9c, 0x3b, 0xc0, 0x2c, 0x9d,
            0xd1, 0x2f, 0x1e, 0x69, 0x1e, 0xf, 0xd6, 0xee, 0x67, 0x2d, 0x4, 0xcf, 0xd9, 0x24, 0xed,
            0x8f, 0xdc, 0x7b, 0xa5, 0xf2, 0xd0, 0x6c, 0x53, 0xc1, 0xed, 0xc3, 0xf, 0x65, 0xf2,
            0xaf, 0x5a, 0x5b, 0x97, 0xf0, 0xa7, 0x6a
        ]
    ); // 228e687a379ba154554040f8821f4e41ee2be287c201aa9c3bc02c9dd12f1e691e0fd6ee672d04cfd924ed8fdc7ba5f2d06c53c1edc30f65f2af5a5b97f0a76a

    // cdetrio10
    let result = alt_bn128_multiplication(&[
        0x17, 0xc1, 0x39, 0xdf, 0xe, 0xfe, 0xe0, 0xf7, 0x66, 0xbc, 0x2, 0x4, 0x76, 0x2b, 0x77,
        0x43, 0x62, 0xe4, 0xde, 0xd8, 0x89, 0x53, 0xa3, 0x9c, 0xe8, 0x49, 0xa8, 0xa7, 0xfa, 0x16,
        0x3f, 0xa9, 0x1, 0xe0, 0x55, 0x9b, 0xac, 0xb1, 0x60, 0x66, 0x47, 0x64, 0xa3, 0x57, 0xaf,
        0x8a, 0x9f, 0xe7, 0xb, 0xaa, 0x92, 0x58, 0xe0, 0xb9, 0x59, 0x27, 0x3f, 0xfc, 0x57, 0x18,
        0xc6, 0xd4, 0xcc, 0x7c, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x1,
    ]); // 17c139df0efee0f766bc0204762b774362e4ded88953a39ce849a8a7fa163fa901e0559bacb160664764a357af8a9fe70baa9258e0b959273ffc5718c6d4cc7c0000000000000000000000000000000000000000000000000000000000000001
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x17, 0xc1, 0x39, 0xdf, 0xe, 0xfe, 0xe0, 0xf7, 0x66, 0xbc, 0x2, 0x4, 0x76, 0x2b, 0x77,
            0x43, 0x62, 0xe4, 0xde, 0xd8, 0x89, 0x53, 0xa3, 0x9c, 0xe8, 0x49, 0xa8, 0xa7, 0xfa,
            0x16, 0x3f, 0xa9, 0x1, 0xe0, 0x55, 0x9b, 0xac, 0xb1, 0x60, 0x66, 0x47, 0x64, 0xa3,
            0x57, 0xaf, 0x8a, 0x9f, 0xe7, 0xb, 0xaa, 0x92, 0x58, 0xe0, 0xb9, 0x59, 0x27, 0x3f,
            0xfc, 0x57, 0x18, 0xc6, 0xd4, 0xcc, 0x7c
        ]
    ); // 17c139df0efee0f766bc0204762b774362e4ded88953a39ce849a8a7fa163fa901e0559bacb160664764a357af8a9fe70baa9258e0b959273ffc5718c6d4cc7c

    // cdetrio11
    let result = alt_bn128_multiplication(&[
        0x3, 0x97, 0x30, 0xea, 0x8d, 0xff, 0x12, 0x54, 0xc0, 0xfe, 0xe9, 0xc0, 0xea, 0x77, 0x7d,
        0x29, 0xa9, 0xc7, 0x10, 0xb7, 0xe6, 0x16, 0x68, 0x3f, 0x19, 0x4f, 0x18, 0xc4, 0x3b, 0x43,
        0xb8, 0x69, 0x7, 0x3a, 0x5f, 0xfc, 0xc6, 0xfc, 0x7a, 0x28, 0xc3, 0x7, 0x23, 0xd6, 0xe5,
        0x8c, 0xe5, 0x77, 0x35, 0x69, 0x82, 0xd6, 0x5b, 0x83, 0x3a, 0x5a, 0x5c, 0x15, 0xbf, 0x90,
        0x24, 0xb4, 0x3d, 0x98, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    ]); // 039730ea8dff1254c0fee9c0ea777d29a9c710b7e616683f194f18c43b43b869073a5ffcc6fc7a28c30723d6e58ce577356982d65b833a5a5c15bf9024b43d98ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x0, 0xa1, 0xa2, 0x34, 0xd0, 0x8e, 0xfa, 0xa2, 0x61, 0x66, 0x7, 0xe3, 0x1e, 0xca, 0x19,
            0x80, 0x12, 0x8b, 0x0, 0xb4, 0x15, 0xc8, 0x45, 0xff, 0x25, 0xbb, 0xa3, 0xaf, 0xcb,
            0x81, 0xdc, 0x0, 0x24, 0x20, 0x77, 0x29, 0xe, 0xd3, 0x39, 0x6, 0xae, 0xb8, 0xe4, 0x2f,
            0xd9, 0x8c, 0x41, 0xbc, 0xb9, 0x5, 0x7b, 0xa0, 0x34, 0x21, 0xaf, 0x3f, 0x2d, 0x8, 0xcf,
            0xc4, 0x41, 0x18, 0x60, 0x24
        ]
    ); // 00a1a234d08efaa2616607e31eca1980128b00b415c845ff25bba3afcb81dc00242077290ed33906aeb8e42fd98c41bcb9057ba03421af3f2d08cfc441186024

    // cdetrio12
    let result = alt_bn128_multiplication(&[
        0x3, 0x97, 0x30, 0xea, 0x8d, 0xff, 0x12, 0x54, 0xc0, 0xfe, 0xe9, 0xc0, 0xea, 0x77, 0x7d,
        0x29, 0xa9, 0xc7, 0x10, 0xb7, 0xe6, 0x16, 0x68, 0x3f, 0x19, 0x4f, 0x18, 0xc4, 0x3b, 0x43,
        0xb8, 0x69, 0x7, 0x3a, 0x5f, 0xfc, 0xc6, 0xfc, 0x7a, 0x28, 0xc3, 0x7, 0x23, 0xd6, 0xe5,
        0x8c, 0xe5, 0x77, 0x35, 0x69, 0x82, 0xd6, 0x5b, 0x83, 0x3a, 0x5a, 0x5c, 0x15, 0xbf, 0x90,
        0x24, 0xb4, 0x3d, 0x98, 0x30, 0x64, 0x4e, 0x72, 0xe1, 0x31, 0xa0, 0x29, 0xb8, 0x50, 0x45,
        0xb6, 0x81, 0x81, 0x58, 0x5d, 0x28, 0x33, 0xe8, 0x48, 0x79, 0xb9, 0x70, 0x91, 0x43, 0xe1,
        0xf5, 0x93, 0xf0, 0x0, 0x0, 0x0,
    ]); // 039730ea8dff1254c0fee9c0ea777d29a9c710b7e616683f194f18c43b43b869073a5ffcc6fc7a28c30723d6e58ce577356982d65b833a5a5c15bf9024b43d9830644e72e131a029b85045b68181585d2833e84879b9709143e1f593f0000000
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x3, 0x97, 0x30, 0xea, 0x8d, 0xff, 0x12, 0x54, 0xc0, 0xfe, 0xe9, 0xc0, 0xea, 0x77,
            0x7d, 0x29, 0xa9, 0xc7, 0x10, 0xb7, 0xe6, 0x16, 0x68, 0x3f, 0x19, 0x4f, 0x18, 0xc4,
            0x3b, 0x43, 0xb8, 0x69, 0x29, 0x29, 0xee, 0x76, 0x1a, 0x35, 0x26, 0x0, 0xf5, 0x49,
            0x21, 0xdf, 0x9b, 0xf4, 0x72, 0xe6, 0x62, 0x17, 0xe7, 0xbb, 0xc, 0xee, 0x90, 0x32,
            0xe0, 0xa, 0xcc, 0x86, 0xb3, 0xc8, 0xbf, 0xaf
        ]
    ); // 039730ea8dff1254c0fee9c0ea777d29a9c710b7e616683f194f18c43b43b8692929ee761a352600f54921df9bf472e66217e7bb0cee9032e00acc86b3c8bfaf

    // cdetrio13
    let result = alt_bn128_multiplication(&[
        0x3, 0x97, 0x30, 0xea, 0x8d, 0xff, 0x12, 0x54, 0xc0, 0xfe, 0xe9, 0xc0, 0xea, 0x77, 0x7d,
        0x29, 0xa9, 0xc7, 0x10, 0xb7, 0xe6, 0x16, 0x68, 0x3f, 0x19, 0x4f, 0x18, 0xc4, 0x3b, 0x43,
        0xb8, 0x69, 0x7, 0x3a, 0x5f, 0xfc, 0xc6, 0xfc, 0x7a, 0x28, 0xc3, 0x7, 0x23, 0xd6, 0xe5,
        0x8c, 0xe5, 0x77, 0x35, 0x69, 0x82, 0xd6, 0x5b, 0x83, 0x3a, 0x5a, 0x5c, 0x15, 0xbf, 0x90,
        0x24, 0xb4, 0x3d, 0x98, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0,
    ]); // 039730ea8dff1254c0fee9c0ea777d29a9c710b7e616683f194f18c43b43b869073a5ffcc6fc7a28c30723d6e58ce577356982d65b833a5a5c15bf9024b43d980000000000000000000000000000000100000000000000000000000000000000
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x10, 0x71, 0xb6, 0x30, 0x11, 0xe8, 0xc2, 0x22, 0xc5, 0xa7, 0x71, 0xdf, 0xa0, 0x3c,
            0x2e, 0x11, 0xaa, 0xc9, 0x66, 0x6d, 0xd0, 0x97, 0xf2, 0xc6, 0x20, 0x85, 0x2c, 0x39,
            0x51, 0xa4, 0x37, 0x6a, 0x2f, 0x46, 0xfe, 0x2f, 0x73, 0xe1, 0xcf, 0x31, 0xa, 0x16,
            0x8d, 0x56, 0xba, 0xa5, 0x57, 0x5a, 0x83, 0x19, 0x38, 0x9d, 0x7b, 0xfa, 0x6b, 0x29,
            0xee, 0x2d, 0x90, 0x83, 0x5, 0x79, 0x14, 0x34
        ]
    ); // 1071b63011e8c222c5a771dfa03c2e11aac9666dd097f2c620852c3951a4376a2f46fe2f73e1cf310a168d56baa5575a8319389d7bfa6b29ee2d908305791434

    // cdetrio14
    let result = alt_bn128_multiplication(&[
        0x3, 0x97, 0x30, 0xea, 0x8d, 0xff, 0x12, 0x54, 0xc0, 0xfe, 0xe9, 0xc0, 0xea, 0x77, 0x7d,
        0x29, 0xa9, 0xc7, 0x10, 0xb7, 0xe6, 0x16, 0x68, 0x3f, 0x19, 0x4f, 0x18, 0xc4, 0x3b, 0x43,
        0xb8, 0x69, 0x7, 0x3a, 0x5f, 0xfc, 0xc6, 0xfc, 0x7a, 0x28, 0xc3, 0x7, 0x23, 0xd6, 0xe5,
        0x8c, 0xe5, 0x77, 0x35, 0x69, 0x82, 0xd6, 0x5b, 0x83, 0x3a, 0x5a, 0x5c, 0x15, 0xbf, 0x90,
        0x24, 0xb4, 0x3d, 0x98, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x9,
    ]); // 039730ea8dff1254c0fee9c0ea777d29a9c710b7e616683f194f18c43b43b869073a5ffcc6fc7a28c30723d6e58ce577356982d65b833a5a5c15bf9024b43d980000000000000000000000000000000000000000000000000000000000000009
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x19, 0xf7, 0x5b, 0x9d, 0xd6, 0x8c, 0x8, 0xa, 0x68, 0x87, 0x74, 0xa6, 0x21, 0x3f, 0x13,
            0x1e, 0x30, 0x52, 0xbd, 0x35, 0x3a, 0x30, 0x4a, 0x18, 0x9d, 0x7a, 0x2e, 0xe3, 0x67,
            0xe3, 0xc2, 0x58, 0x26, 0x12, 0xf5, 0x45, 0xfb, 0x9f, 0xc8, 0x9f, 0xde, 0x80, 0xfd,
            0x81, 0xc6, 0x8f, 0xc7, 0xdc, 0xb2, 0x7f, 0xea, 0x5f, 0xc1, 0x24, 0xee, 0xda, 0x69,
            0x43, 0x3c, 0xf5, 0xc4, 0x6d, 0x2d, 0x7f
        ]
    ); // 19f75b9dd68c080a688774a6213f131e3052bd353a304a189d7a2ee367e3c2582612f545fb9fc89fde80fd81c68fc7dcb27fea5fc124eeda69433cf5c46d2d7f

    // cdetrio15
    let result = alt_bn128_multiplication(&[
        0x3, 0x97, 0x30, 0xea, 0x8d, 0xff, 0x12, 0x54, 0xc0, 0xfe, 0xe9, 0xc0, 0xea, 0x77, 0x7d,
        0x29, 0xa9, 0xc7, 0x10, 0xb7, 0xe6, 0x16, 0x68, 0x3f, 0x19, 0x4f, 0x18, 0xc4, 0x3b, 0x43,
        0xb8, 0x69, 0x7, 0x3a, 0x5f, 0xfc, 0xc6, 0xfc, 0x7a, 0x28, 0xc3, 0x7, 0x23, 0xd6, 0xe5,
        0x8c, 0xe5, 0x77, 0x35, 0x69, 0x82, 0xd6, 0x5b, 0x83, 0x3a, 0x5a, 0x5c, 0x15, 0xbf, 0x90,
        0x24, 0xb4, 0x3d, 0x98, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x1,
    ]); // 039730ea8dff1254c0fee9c0ea777d29a9c710b7e616683f194f18c43b43b869073a5ffcc6fc7a28c30723d6e58ce577356982d65b833a5a5c15bf9024b43d980000000000000000000000000000000000000000000000000000000000000001
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x3, 0x97, 0x30, 0xea, 0x8d, 0xff, 0x12, 0x54, 0xc0, 0xfe, 0xe9, 0xc0, 0xea, 0x77,
            0x7d, 0x29, 0xa9, 0xc7, 0x10, 0xb7, 0xe6, 0x16, 0x68, 0x3f, 0x19, 0x4f, 0x18, 0xc4,
            0x3b, 0x43, 0xb8, 0x69, 0x7, 0x3a, 0x5f, 0xfc, 0xc6, 0xfc, 0x7a, 0x28, 0xc3, 0x7, 0x23,
            0xd6, 0xe5, 0x8c, 0xe5, 0x77, 0x35, 0x69, 0x82, 0xd6, 0x5b, 0x83, 0x3a, 0x5a, 0x5c,
            0x15, 0xbf, 0x90, 0x24, 0xb4, 0x3d, 0x98
        ]
    ); // 039730ea8dff1254c0fee9c0ea777d29a9c710b7e616683f194f18c43b43b869073a5ffcc6fc7a28c30723d6e58ce577356982d65b833a5a5c15bf9024b43d98
}

fn alt_bn128_pairing_test() {
    // jeff1
    let result = alt_bn128_pairing(&[
        0x1c, 0x76, 0x47, 0x6f, 0x4d, 0xef, 0x4b, 0xb9, 0x45, 0x41, 0xd5, 0x7e, 0xbb, 0xa1, 0x19,
        0x33, 0x81, 0xff, 0xa7, 0xaa, 0x76, 0xad, 0xa6, 0x64, 0xdd, 0x31, 0xc1, 0x60, 0x24, 0xc4,
        0x3f, 0x59, 0x30, 0x34, 0xdd, 0x29, 0x20, 0xf6, 0x73, 0xe2, 0x4, 0xfe, 0xe2, 0x81, 0x1c,
        0x67, 0x87, 0x45, 0xfc, 0x81, 0x9b, 0x55, 0xd3, 0xe9, 0xd2, 0x94, 0xe4, 0x5c, 0x9b, 0x3,
        0xa7, 0x6a, 0xef, 0x41, 0x20, 0x9d, 0xd1, 0x5e, 0xbf, 0xf5, 0xd4, 0x6c, 0x4b, 0xd8, 0x88,
        0xe5, 0x1a, 0x93, 0xcf, 0x99, 0xa7, 0x32, 0x96, 0x36, 0xc6, 0x35, 0x14, 0x39, 0x6b, 0x4a,
        0x45, 0x20, 0x3, 0xa3, 0x5b, 0xf7, 0x4, 0xbf, 0x11, 0xca, 0x1, 0x48, 0x3b, 0xfa, 0x8b,
        0x34, 0xb4, 0x35, 0x61, 0x84, 0x8d, 0x28, 0x90, 0x59, 0x60, 0x11, 0x4c, 0x8a, 0xc0, 0x40,
        0x49, 0xaf, 0x4b, 0x63, 0x15, 0xa4, 0x16, 0x78, 0x2b, 0xb8, 0x32, 0x4a, 0xf6, 0xcf, 0xc9,
        0x35, 0x37, 0xa2, 0xad, 0x1a, 0x44, 0x5c, 0xfd, 0xc, 0xa2, 0xa7, 0x1a, 0xcd, 0x7a, 0xc4,
        0x1f, 0xad, 0xbf, 0x93, 0x3c, 0x2a, 0x51, 0xbe, 0x34, 0x4d, 0x12, 0xa, 0x2a, 0x4c, 0xf3,
        0xc, 0x1b, 0xf9, 0x84, 0x5f, 0x20, 0xc6, 0xfe, 0x39, 0xe0, 0x7e, 0xa2, 0xcc, 0xe6, 0x1f,
        0xc, 0x9b, 0xb0, 0x48, 0x16, 0x5f, 0xe5, 0xe4, 0xde, 0x87, 0x75, 0x50, 0x11, 0x1e, 0x12,
        0x9f, 0x1c, 0xf1, 0x9, 0x77, 0x10, 0xd4, 0x1c, 0x4a, 0xc7, 0xf, 0xcd, 0xfa, 0x5b, 0xa2,
        0x2, 0x3c, 0x6f, 0xf1, 0xcb, 0xea, 0xc3, 0x22, 0xde, 0x49, 0xd1, 0xb6, 0xdf, 0x7c, 0x20,
        0x32, 0xc6, 0x1a, 0x83, 0xe, 0x3c, 0x17, 0x28, 0x6d, 0xe9, 0x46, 0x2b, 0xf2, 0x42, 0xfc,
        0xa2, 0x88, 0x35, 0x85, 0xb9, 0x38, 0x70, 0xa7, 0x38, 0x53, 0xfa, 0xce, 0x6a, 0x6b, 0xf4,
        0x11, 0x19, 0x8e, 0x93, 0x93, 0x92, 0xd, 0x48, 0x3a, 0x72, 0x60, 0xbf, 0xb7, 0x31, 0xfb,
        0x5d, 0x25, 0xf1, 0xaa, 0x49, 0x33, 0x35, 0xa9, 0xe7, 0x12, 0x97, 0xe4, 0x85, 0xb7, 0xae,
        0xf3, 0x12, 0xc2, 0x18, 0x0, 0xde, 0xef, 0x12, 0x1f, 0x1e, 0x76, 0x42, 0x6a, 0x0, 0x66,
        0x5e, 0x5c, 0x44, 0x79, 0x67, 0x43, 0x22, 0xd4, 0xf7, 0x5e, 0xda, 0xdd, 0x46, 0xde, 0xbd,
        0x5c, 0xd9, 0x92, 0xf6, 0xed, 0x9, 0x6, 0x89, 0xd0, 0x58, 0x5f, 0xf0, 0x75, 0xec, 0x9e,
        0x99, 0xad, 0x69, 0xc, 0x33, 0x95, 0xbc, 0x4b, 0x31, 0x33, 0x70, 0xb3, 0x8e, 0xf3, 0x55,
        0xac, 0xda, 0xdc, 0xd1, 0x22, 0x97, 0x5b, 0x12, 0xc8, 0x5e, 0xa5, 0xdb, 0x8c, 0x6d, 0xeb,
        0x4a, 0xab, 0x71, 0x80, 0x8d, 0xcb, 0x40, 0x8f, 0xe3, 0xd1, 0xe7, 0x69, 0xc, 0x43, 0xd3,
        0x7b, 0x4c, 0xe6, 0xcc, 0x1, 0x66, 0xfa, 0x7d, 0xaa,
    ]); // 1c76476f4def4bb94541d57ebba1193381ffa7aa76ada664dd31c16024c43f593034dd2920f673e204fee2811c678745fc819b55d3e9d294e45c9b03a76aef41209dd15ebff5d46c4bd888e51a93cf99a7329636c63514396b4a452003a35bf704bf11ca01483bfa8b34b43561848d28905960114c8ac04049af4b6315a416782bb8324af6cfc93537a2ad1a445cfd0ca2a71acd7ac41fadbf933c2a51be344d120a2a4cf30c1bf9845f20c6fe39e07ea2cce61f0c9bb048165fe5e4de877550111e129f1cf1097710d41c4ac70fcdfa5ba2023c6ff1cbeac322de49d1b6df7c2032c61a830e3c17286de9462bf242fca2883585b93870a73853face6a6bf411198e9393920d483a7260bfb731fb5d25f1aa493335a9e71297e485b7aef312c21800deef121f1e76426a00665e5c4479674322d4f75edadd46debd5cd992f6ed090689d0585ff075ec9e99ad690c3395bc4b313370b38ef355acdadcd122975b12c85ea5db8c6deb4aab71808dcb408fe3d1e7690c43d37b4ce6cc0166fa7daa
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1
        ]
    ); // 0000000000000000000000000000000000000000000000000000000000000001

    // jeff2
    let result = alt_bn128_pairing(&[
        0x2e, 0xca, 0xc, 0x72, 0x38, 0xbf, 0x16, 0xe8, 0x3e, 0x7a, 0x1e, 0x6c, 0x5d, 0x49, 0x54,
        0x6, 0x85, 0xff, 0x51, 0x38, 0xf, 0x30, 0x98, 0x42, 0xa9, 0x85, 0x61, 0x55, 0x80, 0x19,
        0xfc, 0x2, 0x3, 0xd3, 0x26, 0x3, 0x61, 0xbb, 0x84, 0x51, 0xde, 0x5f, 0xf5, 0xec, 0xd1,
        0x7f, 0x1, 0xf, 0xf2, 0x2f, 0x5c, 0x31, 0xcd, 0xf1, 0x84, 0xe9, 0x2, 0xb, 0x6, 0xfa, 0x59,
        0x97, 0xdb, 0x84, 0x12, 0x13, 0xd2, 0x14, 0x9b, 0x0, 0x61, 0x37, 0xfc, 0xfb, 0x23, 0x3,
        0x66, 0x6, 0xf8, 0x48, 0xd6, 0x38, 0xd5, 0x76, 0xa1, 0x20, 0xca, 0x98, 0x1b, 0x5b, 0x1a,
        0x5f, 0x93, 0x0, 0xb3, 0xee, 0x22, 0x76, 0xcf, 0x73, 0xc, 0xf4, 0x93, 0xcd, 0x95, 0xd6,
        0x46, 0x77, 0xbb, 0xb7, 0x5f, 0xc4, 0x2d, 0xb7, 0x25, 0x13, 0xa4, 0xc1, 0xe3, 0x87, 0xb4,
        0x76, 0xd0, 0x56, 0xf8, 0xa, 0xa7, 0x5f, 0x21, 0xee, 0x62, 0x26, 0xd3, 0x14, 0x26, 0x32,
        0x2a, 0xfc, 0xda, 0x62, 0x14, 0x64, 0xd0, 0x61, 0x1d, 0x22, 0x67, 0x83, 0x26, 0x2e, 0x21,
        0xbb, 0x3b, 0xc8, 0x6b, 0x53, 0x7e, 0x98, 0x62, 0x37, 0x9, 0x6d, 0xf1, 0xf8, 0x2d, 0xff,
        0x33, 0x7d, 0xd5, 0x97, 0x2e, 0x32, 0xa8, 0xad, 0x43, 0xe2, 0x8a, 0x78, 0xa9, 0x6a, 0x82,
        0x3e, 0xf1, 0xcd, 0x4d, 0xeb, 0xe1, 0x2b, 0x65, 0x52, 0xea, 0x5f, 0x6, 0x96, 0x7a, 0x12,
        0x37, 0xeb, 0xfe, 0xca, 0x9a, 0xaa, 0xe0, 0xd6, 0xd0, 0xba, 0xb8, 0xe2, 0x8c, 0x19, 0x8c,
        0x5a, 0x33, 0x9e, 0xf8, 0xa2, 0x40, 0x7e, 0x31, 0xcd, 0xac, 0x51, 0x6d, 0xb9, 0x22, 0x16,
        0xf, 0xa2, 0x57, 0xa5, 0xfd, 0x5b, 0x28, 0x6, 0x42, 0xff, 0x47, 0xb6, 0x5e, 0xca, 0x77,
        0xe6, 0x26, 0xcb, 0x68, 0x5c, 0x84, 0xfa, 0x6d, 0x3b, 0x68, 0x82, 0xa2, 0x83, 0xdd, 0xd1,
        0x19, 0x8e, 0x93, 0x93, 0x92, 0xd, 0x48, 0x3a, 0x72, 0x60, 0xbf, 0xb7, 0x31, 0xfb, 0x5d,
        0x25, 0xf1, 0xaa, 0x49, 0x33, 0x35, 0xa9, 0xe7, 0x12, 0x97, 0xe4, 0x85, 0xb7, 0xae, 0xf3,
        0x12, 0xc2, 0x18, 0x0, 0xde, 0xef, 0x12, 0x1f, 0x1e, 0x76, 0x42, 0x6a, 0x0, 0x66, 0x5e,
        0x5c, 0x44, 0x79, 0x67, 0x43, 0x22, 0xd4, 0xf7, 0x5e, 0xda, 0xdd, 0x46, 0xde, 0xbd, 0x5c,
        0xd9, 0x92, 0xf6, 0xed, 0x9, 0x6, 0x89, 0xd0, 0x58, 0x5f, 0xf0, 0x75, 0xec, 0x9e, 0x99,
        0xad, 0x69, 0xc, 0x33, 0x95, 0xbc, 0x4b, 0x31, 0x33, 0x70, 0xb3, 0x8e, 0xf3, 0x55, 0xac,
        0xda, 0xdc, 0xd1, 0x22, 0x97, 0x5b, 0x12, 0xc8, 0x5e, 0xa5, 0xdb, 0x8c, 0x6d, 0xeb, 0x4a,
        0xab, 0x71, 0x80, 0x8d, 0xcb, 0x40, 0x8f, 0xe3, 0xd1, 0xe7, 0x69, 0xc, 0x43, 0xd3, 0x7b,
        0x4c, 0xe6, 0xcc, 0x1, 0x66, 0xfa, 0x7d, 0xaa,
    ]); // 2eca0c7238bf16e83e7a1e6c5d49540685ff51380f309842a98561558019fc0203d3260361bb8451de5ff5ecd17f010ff22f5c31cdf184e9020b06fa5997db841213d2149b006137fcfb23036606f848d638d576a120ca981b5b1a5f9300b3ee2276cf730cf493cd95d64677bbb75fc42db72513a4c1e387b476d056f80aa75f21ee6226d31426322afcda621464d0611d226783262e21bb3bc86b537e986237096df1f82dff337dd5972e32a8ad43e28a78a96a823ef1cd4debe12b6552ea5f06967a1237ebfeca9aaae0d6d0bab8e28c198c5a339ef8a2407e31cdac516db922160fa257a5fd5b280642ff47b65eca77e626cb685c84fa6d3b6882a283ddd1198e9393920d483a7260bfb731fb5d25f1aa493335a9e71297e485b7aef312c21800deef121f1e76426a00665e5c4479674322d4f75edadd46debd5cd992f6ed090689d0585ff075ec9e99ad690c3395bc4b313370b38ef355acdadcd122975b12c85ea5db8c6deb4aab71808dcb408fe3d1e7690c43d37b4ce6cc0166fa7daa
    assert!(result.is_ok());
    assert_eq!(
        result.unwrap(),
        &[
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1
        ]
    ); // 0000000000000000000000000000000000000000000000000000000000000001
}

#[no_mangle]
pub extern "C" fn entrypoint(_input: *mut u8) -> u64 {
    msg!("alt_bn128");

    alt_bn128_addition_test();
    alt_bn128_multiplication_test();
    alt_bn128_pairing_test();

    0
}

custom_heap_default!();
custom_panic_default!();
