/// FIREDANCER: Repalce PohRecorder completely with one that goes out to
///             our implementation.
use crate::leader_bank_notifier::LeaderBankNotifier;
use crossbeam_channel::{Receiver, Sender, TrySendError};
use solana_clock::Slot;
use solana_hash::Hash;
use solana_ledger::blockstore::Blockstore;
use solana_pubkey::Pubkey;
use solana_runtime::{bank::Bank, installed_scheduler_pool::BankWithScheduler};

use solana_ledger::leader_schedule_cache::LeaderScheduleCache;
use solana_poh_config::PohConfig;

use std::ffi::c_void;
use std::sync::atomic::AtomicBool;
use std::sync::Arc;

use crate::poh_recorder::{BankStart, PohLeaderStatus, WorkingBankEntry};
use crate::transaction_recorder::TransactionRecorder;

extern "C" {
    fn fd_ext_poh_initialize(
        tick_duration_nanos: u64,
        hashcnt_per_tick: u64,
        ticks_per_slot: u64,
        tick_height: u64,
        last_entry_hash: *const u8,
        signal_leader_change: *mut c_void,
    );
    fn fd_ext_poh_acquire_leader_bank() -> *const c_void;
    fn fd_ext_poh_reset_slot() -> u64;
    fn fd_ext_poh_reached_leader_slot(out_leader_slot: *mut u64, out_reset_slot: *mut u64) -> i32;
    fn fd_ext_poh_begin_leader(
        bank: *const c_void,
        slot: u64,
        epoch: u64,
        hashcnt_per_tick: u64,
        cus_block_limit: u64,
        cus_vote_cost_limit: u64,
        cus_account_cost_limit: u64,
        txn_limit: u64,
        txn_byte_limit: u64,
    );
    fn fd_ext_poh_reset(
        reset_bank_slot: u64,
        reset_blockhash: *const u8,
        hashcnt_per_tick: u64,
        block_id: *const u8,
        features_activation_slot: *const u64,
    );
    fn fd_ext_poh_get_leader_after_n_slots(n: u64, out_pubkey: *mut u8) -> i32;
    fn fd_ext_poh_get_leader_after_n_slots_range(
        start_slot: u64,
        end_slot: u64,
        out_pubkeys: *mut u8,
    ) -> i32;
    fn fd_ext_poh_update_active_descendant(max_active_descendant: u64);
}

#[no_mangle]
pub extern "C" fn fd_ext_poh_signal_leader_change(sender: *mut c_void) {
    if sender.is_null() {
        return;
    }

    let sender: &Sender<bool> = unsafe { &*(sender as *mut Sender<bool>) };
    match sender.try_send(true) {
        Ok(()) | Err(TrySendError::Full(_)) => (),
        err => err.unwrap(),
    }
}

#[no_mangle]
pub extern "C" fn fd_ext_poh_register_tick(bank: *const c_void, hash: *const u8) {
    let hash = unsafe { std::slice::from_raw_parts(hash, 32) };
    let hash = Hash::new_from_array(hash.try_into().unwrap());
    unsafe {
        (*(bank as *const Bank)).register_tick(&hash, &BankWithScheduler::no_scheduler_available())
    };
}

const FD_POH_RECORDER_FEATURES_OF_INTEREST_CNT: usize = 4usize;
static FD_POH_RECORDER_FEATURES_OF_INTEREST: [Pubkey; FD_POH_RECORDER_FEATURES_OF_INTEREST_CNT] = [
    agave_feature_set::disable_turbine_fanout_experiments::id(),
    agave_feature_set::enable_turbine_extended_fanout_experiments::id(),
    agave_feature_set::enable_vote_address_leader_schedule::id(),
    agave_feature_set::drop_unchained_merkle_shreds::id(),
];

pub struct PohRecorder {
    pub is_exited: Arc<AtomicBool>,
    pub leader_slots: u64,
}

impl PohRecorder {
    #[allow(clippy::too_many_arguments)]
    pub fn new_with_clear_signal(
        tick_height: u64,
        last_entry_hash: Hash,
        _start_bank: Arc<Bank>,
        _next_leader_slot: Option<(Slot, Slot)>,
        ticks_per_slot: u64,
        _delay_leader_block_for_pending_fork: bool,
        _blockstore: Arc<Blockstore>,
        clear_bank_signal: Option<Sender<bool>>,
        _leader_schedule_cache: &Arc<LeaderScheduleCache>,
        poh_config: &PohConfig,
        is_exited: Arc<AtomicBool>,
    ) -> (Self, Receiver<WorkingBankEntry>) {
        /* Just silence the unused warning for old_poh_recorder, without needing to modify the file. */
        let _silence_warnings = super::old_poh_recorder::create_test_recorder;

        let clear_bank_sender: *mut Sender<bool> = match clear_bank_signal {
            Some(sender) => Box::into_raw(Box::new(sender)),
            None => std::ptr::null_mut(),
        };

        unsafe {
            fd_ext_poh_initialize(
                poh_config
                    .target_tick_duration
                    .as_nanos()
                    .try_into()
                    .unwrap(),
                poh_config.hashes_per_tick.unwrap_or(1),
                ticks_per_slot,
                tick_height,
                last_entry_hash.as_ref().as_ptr(),
                clear_bank_sender as *mut c_void,
            )
        };

        let dummy1 = crossbeam_channel::unbounded();
        /* Forget so the receiver doesn't see the channel is disconnected. */
        std::mem::forget(dummy1.0);
        (
            Self {
                is_exited,
                leader_slots: 0,
            },
            dummy1.1,
        )
    }

    pub fn get_consecutive_leader_slots(&self) -> u64 {
        self.leader_slots
    }

    pub fn leader_after_n_slots(&self, slots: u64) -> Option<Pubkey> {
        /* Must be implemented. Used to determine where to send our votes. */
        let mut pubkey = [0u8; 32];
        unsafe {
            if 1 == fd_ext_poh_get_leader_after_n_slots(slots, pubkey.as_mut_ptr()) {
                Some(Pubkey::new_from_array(pubkey))
            } else {
                None
            }
        }
    }

    pub fn leader_after_n_slots_range(&self, start_slot: u64, end_slot: u64) -> Vec<Pubkey> {
        let mut pubkeys: [u8; 64] = [0u8; 64];
        unsafe {
            let count = fd_ext_poh_get_leader_after_n_slots_range(
                start_slot,
                end_slot,
                pubkeys.as_mut_ptr(),
            );
            let mut result = Vec::new();
            if count >= 1 {
                result.push(Pubkey::new_from_array(pubkeys[0..32].try_into().unwrap()));
            }
            if count >= 2 {
                result.push(Pubkey::new_from_array(pubkeys[32..64].try_into().unwrap()));
            }
            result
        }
    }

    pub fn leader_and_slot_after_n_slots(
        &self,
        _slots_in_the_future: u64,
    ) -> Option<(Pubkey, Slot)> {
        /* Not needed for any important functionality, only the RPC send
        transaction service. */
        None
    }

    pub fn bank_start(&self) -> Option<BankStart> {
        /* Unused, only called by old TPU. */
        None
    }

    pub fn new_recorder(&self) -> TransactionRecorder {
        /* Just needs a dummy value, the only caller will never use it. */
        let (sender, _) = crossbeam_channel::unbounded();
        TransactionRecorder::new(sender, Arc::new(AtomicBool::new(false)))
    }

    pub fn would_be_leader(&self, _within_next_n_ticks: u64) -> bool {
        /* The only caller asks if it's within the next ten minutes, so
        that it can forward gossiped votes to ourselves.  We can just
        always forward them. */
        true
    }

    pub fn has_bank(&self) -> bool {
        /* Must be implemented, used by replay stage. */
        self.bank().is_some()
    }

    pub fn bank(&self) -> Option<Arc<Bank>> {
        /* Must be implemented, used by replay stage. */
        let bank: *const Bank = unsafe { fd_ext_poh_acquire_leader_bank() } as *const Bank;

        if bank.is_null() {
            None
        } else {
            Some(unsafe { Arc::from_raw(bank) })
        }
    }

    pub fn new_leader_bank_notifier(&self) -> Arc<LeaderBankNotifier> {
        /* Unsued, only called by old TPU. */
        Arc::default()
    }

    pub fn update_start_bank_active_descendants(&mut self, active_descendants: &[Slot]) {
        unsafe {
            fd_ext_poh_update_active_descendant(*active_descendants.iter().max().unwrap_or(&0))
        };
    }

    pub fn start_slot(&self) -> Slot {
        /* Must be implemented, used by replay stage. */
        unsafe { fd_ext_poh_reset_slot() - 1 }
    }

    pub fn reached_leader_slot(&self, _pubkey: &Pubkey) -> PohLeaderStatus {
        /* Must be implemented, used by replay stage.
        The pubkey currently used here is always the
        leader pubkey only, so it can be ignored. */
        let mut leader_slot: u64 = 0;
        let mut reset_slot: u64 = 0;
        let is_leader =
            unsafe { fd_ext_poh_reached_leader_slot(&mut leader_slot, &mut reset_slot) };

        if is_leader != 0 {
            PohLeaderStatus::Reached {
                poh_slot: leader_slot,
                parent_slot: reset_slot - 1,
                missing_blocks_in_grace: is_leader == 2,
            }
        } else {
            PohLeaderStatus::NotReached
        }
    }

    pub fn set_bank(
        &mut self,
        bank_with_scheduler: BankWithScheduler,
        _track_transaction_indexes: bool,
    ) {
        /* Must be implemented, used by replay stage. */
        let bank = bank_with_scheduler.clone_without_scheduler();
        let slot = bank.slot();
        let epoch = bank.epoch();
        let hashes_per_tick = bank.hashes_per_tick().unwrap_or(1);

        let cus_vote_cost_limit = solana_cost_model::block_cost_limits::MAX_VOTE_UNITS;
        let cus_block_limit = bank.read_cost_tracker().unwrap().get_block_limit();
        let cus_account_cost_limit = bank.read_cost_tracker().unwrap().get_account_limit();
        let block_txn_limit = bank
            .read_cost_tracker()
            .unwrap()
            .get_block_txn_count_limit();
        let block_txn_byte_limit = bank.read_cost_tracker().unwrap().get_block_txn_byte_limit();
        let leader_bank: *const Bank = Arc::into_raw(bank);
        unsafe {
            fd_ext_poh_begin_leader(
                leader_bank as *const c_void,
                slot,
                epoch,
                hashes_per_tick,
                cus_block_limit,
                cus_vote_cost_limit,
                cus_account_cost_limit,
                block_txn_limit,
                block_txn_byte_limit,
            )
        };
    }

    pub fn reset(&mut self, reset_bank: Arc<Bank>, _next_leader_slot: Option<(Slot, Slot)>) {
        /* Must be implemented, used by replay stage. */
        let reset_bank_slot = reset_bank.slot();
        let reset_bank_blockhash = reset_bank.last_blockhash();
        let hashes_per_tick = reset_bank.hashes_per_tick().unwrap_or(1);

        let block_id = reset_bank.block_id().unwrap_or_default();
        let block_id_ptr = if let Some(_block_id) = reset_bank.block_id() {
            /* _block_id scope ends here. We can't use _block_id.as_ref().as_ptr()
            as it points to memory with an incorrect value.
            We must use block_id, that lives beyond this scope. */
            block_id.as_ref().as_ptr()
        } else {
            std::ptr::null()
        };

        /* There is a subset of FD_POH_RECORDER_FEATURES_OF_INTEREST
        activation slots that the shred tile needs to be aware of.
        Due to the fact that their computation requires the bank,
        we are forced (so far) to implement it here, sending them
        to the poh tile as an intermediary (before forwarding them
        to the shred tile). This is not elegant, and it should be
        revised in the future (TODO), but it provides a "temporary"
        working solution to handle features activation. */
        let mut features_activation_slot: [u64; FD_POH_RECORDER_FEATURES_OF_INTEREST_CNT] =
            [u64::MAX; FD_POH_RECORDER_FEATURES_OF_INTEREST_CNT];
        for (i, pubkey) in FD_POH_RECORDER_FEATURES_OF_INTEREST.iter().enumerate() {
            features_activation_slot[i] = match reset_bank.feature_set.activated_slot(pubkey) {
                None => u64::MAX,
                Some(feature_slot) => {
                    let epoch_schedule = reset_bank.epoch_schedule();
                    epoch_schedule.get_epoch(feature_slot) as Slot
                }
            }
        }
        self.leader_slots = reset_bank.get_leader_slots(reset_bank.epoch());
        unsafe {
            fd_ext_poh_reset(
                reset_bank_slot,
                reset_bank_blockhash.as_ref().as_ptr(),
                hashes_per_tick,
                block_id_ptr,
                features_activation_slot.as_ref().as_ptr(),
            )
        };
    }
}
