use anyhow::Context;
use log::info;
use serde::{Deserialize, Serialize};
use std::fs;
use std::path::Path;

#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct Config {
    /// Fogo
    #[serde(default)]
    pub fogo: FogoConfig,
}

#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct FogoConfig {
    /// Threads
    #[serde(default)]
    pub threads: ThreadsConfig,
    /// Affinity
    #[serde(default)]
    pub affinity: AffinityConfig,
    /// Rpc
    #[serde(default)]
    pub rpc: RpcConfig,
}

#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct ThreadsConfig {
    /// Rpc threads
    #[serde(default)]
    pub rpc: Option<usize>,
    /// RPC blocking
    #[serde(default)]
    pub rpc_blocking: Option<usize>,
}

#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct ThreadPoolConfig {
    /// Prefix
    #[serde(default)]
    pub prefix: Vec<String>,
    /// Core count
    #[serde(default)]
    pub core_count: Option<usize>,
}

#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct AffinityConfig {
    /// Affinity manager on
    #[serde(default)]
    pub use_affinity_manager: bool,
    /// Thread pools
    #[serde(default)]
    pub thread_pool: Vec<ThreadPoolConfig>,
}

#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct RpcConfig {
    /// Enable block
    #[serde(default)]
    pub enable_block_pubsub: Option<bool>,
    /// Enable vote
    #[serde(default)]
    pub enable_vote_pubsub: Option<bool>,
    /// Pubsub limit
    #[serde(default)]
    pub max_active_subscriptions: Option<usize>,
    /// Scan limit bytes
    #[serde(default)]
    pub scan_limit_bytes: Option<usize>,
    /// Max request body size
    #[serde(default)]
    pub max_request_body_size: Option<usize>,
}

pub(crate) fn load_config(path: &Path) -> anyhow::Result<Config> {
    info!("[fogo_validator] Loading config from: {:?}", path);
    let contents = fs::read_to_string(path).context("loading config")?;
    let contents = toml::from_str(&contents).context("parsing config")?;
    info!("[fogo_validator] Loaded configs: {:#?}", contents);

    Ok(contents)
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn test_load_example_config() {
        let config = toml::from_str::<Config>(include_str!("../config_example.toml")).unwrap();
        assert_eq!(config.fogo.threads.rpc, Some(10));
        assert_eq!(config.fogo.threads.rpc_blocking, Some(10));
        assert_eq!(config.fogo.affinity.use_affinity_manager, true);
        assert_eq!(config.fogo.affinity.thread_pool.len(), 2);
        assert_eq!(
            config.fogo.affinity.thread_pool[0].prefix,
            vec!["solAdminRpc*", "solRpc*"]
        );
        assert_eq!(config.fogo.affinity.thread_pool[0].core_count, Some(4));
        assert_eq!(
            config.fogo.affinity.thread_pool[1].prefix,
            vec!["solReplay*"]
        );
        assert_eq!(config.fogo.affinity.thread_pool[1].core_count, Some(4));
    }
}
