//! Bindings and rust abstractions for the firedancer netconnector.
//!
//! The net_connector tile receives UDP packets from the XDP net tile and
//! routes them to ring buffers based on packet type (gossip/repair) and
//! staked/unstaked status. This module provides safe Rust wrappers around
//! the C FFI functions to poll packets from those ring buffers.
//!
//! Ring buffer layout:
//! - Gossip staked (ring 0): Priority gossip packets from staked validators
//! - Gossip unstaked (ring 1): Gossip packets from unstaked sources
//! - Repair staked (ring 2): Priority repair packets from staked validators
//! - Repair unstaked (ring 3): Repair packets from unstaked sources
//!
//! Traffic isolation: Staked rings drop packets when full, unstaked rings
//! overwrite oldest packets. This ensures high unstaked traffic cannot
//! affect delivery of staked packets.
//!
//! This crate has a feature called "firedancer", if its compiled with that feature, the actual
//! FFI bindings will be used. Otherwise it falls back to solana_streamer.

use solana_perf::packet::{PacketBatch, PacketBatchRecycler};
use solana_streamer::streamer::{ChannelSend, StreamerReceiveStats};
use std::net::UdpSocket;
use std::sync::atomic::AtomicBool;
use std::sync::Arc;
use std::thread::{Builder, JoinHandle};
use std::time::Duration;

#[macro_use]
extern crate log;

#[cfg(feature = "firedancer")]
mod firedancer;

pub mod protocol {
    pub const GOSSIP: i32 = 1;
    pub const REPAIR: i32 = 2;
    pub const ANCESTOR_HASHES: i32 = 3;

    pub fn to_string(proto: i32) -> &'static str {
        match proto {
            GOSSIP => "gossip",
            REPAIR => "repair",
            ANCESTOR_HASHES => "ancestor_hashes",
            _ => "unknown",
        }
    }
}

#[allow(clippy::too_many_arguments)]
pub fn receiver<const PROTOCOL: i32>(
    thread_name: String,
    socket: Arc<UdpSocket>,
    exit: Arc<AtomicBool>,
    packet_batch_sender: impl ChannelSend<PacketBatch>,
    recycler: PacketBatchRecycler,
    stats: Arc<StreamerReceiveStats>,
    coalesce: Option<Duration>,
    use_pinned_memory: bool,
    in_vote_only_mode: Option<Arc<AtomicBool>>,
    is_staked_service: bool,
) -> JoinHandle<()> {
    let res = socket.set_read_timeout(Some(Duration::new(1, 0)));
    assert!(res.is_ok(), "streamer::receiver set_read_timeout error");
    Builder::new()
        .name(thread_name)
        .spawn(move || {
            #[cfg(feature = "firedancer")]
            let _ = firedancer::recv_loop::<PROTOCOL>(
                &socket,
                &exit,
                &packet_batch_sender,
                &recycler,
                &stats,
                coalesce,
                use_pinned_memory,
                in_vote_only_mode,
                is_staked_service,
            );

            #[cfg(not(feature = "firedancer"))]
            let _ = recv_loop(
                &socket,
                &exit,
                &packet_batch_sender,
                &recycler,
                &stats,
                coalesce,
                use_pinned_memory,
                in_vote_only_mode,
                is_staked_service,
            );
        })
        .unwrap()
}
