//! FFI bindings to Firedancer's net_connector tile for receiving packets.
//!
//! The net_connector tile receives UDP packets from the XDP net tile and
//! routes them to ring buffers based on packet type (gossip/repair) and
//! staked/unstaked status. This module provides safe Rust wrappers around
//! the C FFI functions to poll packets from those ring buffers.
//!
//! Ring buffer layout:
//! - Gossip staked (ring 0): Priority gossip packets from staked validators
//! - Gossip unstaked (ring 1): Gossip packets from unstaked sources
//! - Repair staked (ring 2): Priority repair packets from staked validators
//! - Repair unstaked (ring 3): Repair packets from unstaked sources
//!
//! Traffic isolation: Staked rings drop packets when full, unstaked rings
//! overwrite oldest packets. This ensures high unstaked traffic cannot
//! affect delivery of staked packets.

use crate::protocol;
use crossbeam_channel::{SendError, TrySendError};
use solana_packet::Packet;
use solana_perf::packet::{PacketBatch, PacketBatchRecycler, PinnedPacketBatch, PACKETS_PER_BATCH};
use solana_streamer::streamer::{ChannelSend, StreamerError, StreamerReceiveStats};
use std::net::{Ipv4Addr, SocketAddr, SocketAddrV4, UdpSocket};
use std::sync::atomic::{AtomicBool, Ordering};
use std::sync::Arc;
use std::thread::sleep;
use std::time::Duration;

extern "C" {
    /// Blocks until the net_connector tile is initialized.
    /// Call once during startup before polling.
    fn fd_ext_netconn_wait_ready();

    /// Polls gossip ring buffers for the next available packet.
    /// Returns payload size (0 if no packet available).
    /// Prioritizes staked packets over unstaked.
    fn fd_ext_netconn_poll_gossip(
        out_payload: *mut u8,
        out_src_ip: *mut u32,
        out_src_port: *mut u16,
    ) -> u64;

    /// Polls repair ring buffers for the next available packet.
    /// Returns payload size (0 if no packet available).
    /// Prioritizes staked packets over unstaked.
    fn fd_ext_netconn_poll_repair(
        out_payload: *mut u8,
        out_src_ip: *mut u32,
        out_src_port: *mut u16,
    ) -> u64;

    /// Polls ancestor ring buffers for the next available packet.
    /// Returns payload size (0 if no packet available).
    /// Prioritizes staked packets over unstaked.
    fn fd_ext_netconn_poll_ancestor(
        out_payload: *mut u8,
        out_src_ip: *mut u32,
        out_src_port: *mut u16,
    ) -> u64;

    /// Returns total number of gossip packets available.
    fn fd_ext_netconn_gossip_avail() -> u64;

    /// Returns total number of repair packets available.
    fn fd_ext_netconn_repair_avail() -> u64;

    /// Returns total number of repair packets available.
    fn fd_ext_netconn_ancestor_avail() -> u64;
}

/// Safe wrapper around the net_connector FFI.
///
/// This struct provides a safe interface to poll packets from the
/// Firedancer net_connector tile's ring buffers.
pub struct NetConnector;

impl NetConnector {
    /// Creates a new NetConnector and waits for the tile to be ready.
    ///
    /// This function blocks until the net_connector tile has initialized
    /// its ring buffers. Call this once during startup.
    pub fn new() -> Self {
        unsafe {
            fd_ext_netconn_wait_ready();
        }

        Self
    }

    /// Polls for the next packet from the given protocol.
    ///
    /// Returns `Some(packet)` if a packet is available, `None` otherwise.
    /// Staked packets are prioritized over unstaked packets.
    pub fn poll<const PROTOCOL: i32>(&self) -> Option<Packet> {
        if self.available::<PROTOCOL>() == 0 {
            return None;
        }

        let mut src_ip: u32 = 0;
        let mut src_port: u16 = 0;

        let mut packet = Packet::default();

        let size = unsafe {
            match PROTOCOL {
                protocol::GOSSIP => fd_ext_netconn_poll_gossip(
                    (packet.buffer_mut()).as_mut_ptr(),
                    &mut src_ip,
                    &mut src_port,
                ),
                protocol::REPAIR => fd_ext_netconn_poll_repair(
                    (packet.buffer_mut()).as_mut_ptr(),
                    &mut src_ip,
                    &mut src_port,
                ),
                protocol::ANCESTOR_HASHES => fd_ext_netconn_poll_ancestor(
                    (packet.buffer_mut()).as_mut_ptr(),
                    &mut src_ip,
                    &mut src_port,
                ),
                _ => unreachable!(),
            }
        };

        if size == 0 {
            return None;
        }

        // Convert network byte order IP to Ipv4Addr
        let ip = Ipv4Addr::from(u32::from_be(src_ip));
        // Convert network byte order port
        let port = u16::from_be(src_port);
        let from = SocketAddr::V4(SocketAddrV4::new(ip, port));
        packet.meta_mut().size = size as usize;
        packet.meta_mut().set_socket_addr(&from);

        Some(packet)
    }

    /// Returns the number of gossip packets available.
    pub fn available<const PROTOCOL: i32>(&self) -> usize {
        match PROTOCOL {
            protocol::GOSSIP => unsafe { fd_ext_netconn_gossip_avail() as usize },
            protocol::REPAIR => unsafe { fd_ext_netconn_repair_avail() as usize },
            protocol::ANCESTOR_HASHES => unsafe { fd_ext_netconn_ancestor_avail() as usize },
            _ => unreachable!(),
        }
    }
}

impl Default for NetConnector {
    fn default() -> Self {
        Self::new()
    }
}

/// How long to wait when no packets are available before sleeping
const NET_CONNECTOR_POLL_INTERVAL: Duration = Duration::from_millis(1);

pub fn recv_loop<const PROTOCOL: i32>(
    _: &UdpSocket,
    exit: &AtomicBool,
    packet_batch_sender: &impl ChannelSend<PacketBatch>,
    recycler: &PacketBatchRecycler,
    stats: &StreamerReceiveStats,
    _: Option<Duration>,
    use_pinned_memory: bool,
    in_vote_only_mode: Option<Arc<AtomicBool>>,
    is_staked_service: bool,
) -> solana_streamer::streamer::Result<()> {
    let net_connector = NetConnector::new();
    warn!(
        "net_connector {} receiver started",
        protocol::to_string(PROTOCOL)
    );

    loop {
        // Check for exit signal, even if socket is busy
        // (for instance the leader transaction socket)
        if exit.load(Ordering::Relaxed) {
            return Ok(());
        }

        if let Some(ref in_vote_only_mode) = in_vote_only_mode {
            if in_vote_only_mode.load(Ordering::Relaxed) {
                sleep(Duration::from_millis(1));
                continue;
            }
        }

        let mut packet_batch = if use_pinned_memory {
            PinnedPacketBatch::new_with_recycler(recycler, PACKETS_PER_BATCH, stats.name)
        } else {
            PinnedPacketBatch::with_capacity(PACKETS_PER_BATCH)
        };

        // Drain available packets up to batch size
        while packet_batch.len() < PACKETS_PER_BATCH {
            if let Some(packet) = net_connector.poll::<PROTOCOL>() {
                // warn!("fd_netcon got agave packet {}", PROTOCOL);
                packet_batch.push(packet);
            } else {
                // out of packets
                break;
            }
        }

        if packet_batch.is_empty() {
            // No packets available, sleep briefly
            sleep(NET_CONNECTOR_POLL_INTERVAL);
            continue;
        }

        // send batch
        let StreamerReceiveStats {
            packets_count,
            packet_batches_count,
            full_packet_batches_count,
            max_channel_len,
            ..
        } = stats;

        let len = packet_batch.len();
        packets_count.fetch_add(len, Ordering::Relaxed);
        packet_batches_count.fetch_add(1, Ordering::Relaxed);
        max_channel_len.fetch_max(packet_batch_sender.len(), Ordering::Relaxed);
        if len == PACKETS_PER_BATCH {
            full_packet_batches_count.fetch_add(1, Ordering::Relaxed);
        }
        packet_batch
            .iter_mut()
            .for_each(|p| p.meta_mut().set_from_staked_node(is_staked_service));
        match packet_batch_sender.try_send(packet_batch.into()) {
            Ok(_) => {}
            Err(TrySendError::Full(_)) => {
                stats.num_packets_dropped.fetch_add(len, Ordering::Relaxed);
            }
            Err(TrySendError::Disconnected(err)) => {
                return Err(StreamerError::Send(SendError(err)))
            }
        }
    }
}
