//! defines block cost related limits
//!

// Cluster data, method of collecting at https://github.com/solana-labs/solana/issues/19627
// Dashboard: https://metrics.solana.com/d/monitor-edge/cluster-telemetry?orgId=1

/// Cluster averaged compute unit to micro-sec conversion rate
pub const COMPUTE_UNIT_TO_US_RATIO: u64 = 30;
/// Number of compute units for one signature verification.
pub const SIGNATURE_COST: u64 = COMPUTE_UNIT_TO_US_RATIO * 24;
/// Number of compute units for one secp256k1 signature verification.
pub const SECP256K1_VERIFY_COST: u64 = COMPUTE_UNIT_TO_US_RATIO * 223;
/// Number of compute units for one ed25519 signature verification.
pub const ED25519_VERIFY_COST: u64 = COMPUTE_UNIT_TO_US_RATIO * 76;
/// Number of compute units for one ed25519 strict signature verification.
pub const ED25519_VERIFY_STRICT_COST: u64 = COMPUTE_UNIT_TO_US_RATIO * 80;
/// Number of compute units for one secp256r1 signature verification.
pub const SECP256R1_VERIFY_COST: u64 = COMPUTE_UNIT_TO_US_RATIO * 160;
/// Number of compute units for one write lock
pub const WRITE_LOCK_UNITS: u64 = COMPUTE_UNIT_TO_US_RATIO * 10;
/// Number of data bytes per compute units
pub const INSTRUCTION_DATA_BYTES_COST: u64 = 140 /*bytes per us*/ / COMPUTE_UNIT_TO_US_RATIO;

/// Number of compute units that a block is allowed. A block's compute units are
/// accumulated by Transactions added to it; A transaction's compute units are
/// calculated by cost_model, based on transaction's signatures, write locks,
/// data size and built-in and SBF instructions.
pub const MAX_BLOCK_UNITS: u64 = MAX_BLOCK_UNITS_SIMD_0207;
pub const MAX_BLOCK_UNITS_SIMD_0207: u64 = 5_000_000;
pub const MAX_BLOCK_UNITS_SIMD_0256: u64 = 6_000_000;
pub const MAX_BLOCK_UNITS_FOGO: u64 = 10_000_000;

/// Number of compute units that a writable account in a block is allowed. The
/// limit is to prevent too many transactions write to same account, therefore
/// reduce block's parallelism.
pub const MAX_WRITABLE_ACCOUNT_UNITS: u64 = 1_200_000;
pub const MAX_WRITABLE_ACCOUNT_UNITS_FOGO: u64 = 2_500_000;

/// Number of compute units that a block can have for vote transactions,
/// set to less than MAX_BLOCK_UNITS to leave room for non-vote transactions
pub const MAX_VOTE_UNITS: u64 = 3_600_000;

/// The maximum allowed size, in bytes, that accounts data can grow, per block.
/// This can also be thought of as the maximum size of new allocations per block.
pub const MAX_BLOCK_ACCOUNTS_DATA_SIZE_DELTA: u64 = 100_000_000;

/// Fogo limits the number of txns in a block.
/// We do this to protect the validator from too
/// many small txns which use almost no CU, yet still
/// have verification overhead. This limit is set to 4000
/// because in testing this yields 100K TPS of self transfers
/// which the validator is stable at. Beyond this rate
/// the validator is not stable. In practice we can only hit
/// 4000 txns per block with self-transfers, any real work load
/// should not hit this limit
pub const MAX_TXN_IN_BLOCK: u64 = 4_000;

/// Fogo limits the number txn bytes per block.
/// We do this because each txn byte in the block puts presssure
/// on rocksdb and store tile. We might have relatively low CU
/// txns with high payloads which could cause validator instability
/// due to disk I/O from having to store all these bytes in the blockstore.
/// 2 million bytes was derived from testing the limits of the validator
/// by spamming it with 1K payloads at different TPS. At 55K TPS with 1K
/// payloads, the validator becomes unstable. The value of 2 million bytes
/// per block translates into 41K TPS with 1K payloads. This is a conservative
/// value at which the validator is stable.
pub const MAX_TXN_BYTES_IN_BLOCK: u64 = 2_000_000;

/// Return the block limits that will be used upon activation of SIMD-0256.
/// Returns as
/// (account_limit, block_limit, vote_limit)
// ^ Above order is used to be consistent with the order of
//   `CostTracker::set_limits`.
pub const fn simd_0256_block_limits() -> (u64, u64, u64) {
    (
        MAX_WRITABLE_ACCOUNT_UNITS_FOGO,
        MAX_BLOCK_UNITS_SIMD_0256,
        MAX_VOTE_UNITS,
    )
}

pub const fn get_raised_writeable_account_units_2_5() -> u64 {
    MAX_WRITABLE_ACCOUNT_UNITS_FOGO
}
