#ifndef HEADER_fd_src_discof_rpcserver_fd_methods_h
#define HEADER_fd_src_discof_rpcserver_fd_methods_h

#include "json_lex.h"

// Data structure describing a "path" to a value in json data. This is
// basically a path through the syntax tree. A path element can be one
// of the following:
//   object member:  (JSON_TOKEN_LBRACE<<16) | keyword_id
//   array member:   (JSON_TOKEN_LBRACKET<<16) | index
//   string value:   (JSON_TOKEN_STRING<<16)
//   int value:      (JSON_TOKEN_INTEGER<<16)
//   float value:    (JSON_TOKEN_FLOAT<<16)
//   boolean value:  (JSON_TOKEN_BOOL<<16)
//   null value:     (JSON_TOKEN_NULL<<16)
// keyword ids are generated by json_keyword(...) in keywords.h
#define JSON_MAX_PATH 8
struct json_path {
    uint len;
    uint elems[JSON_MAX_PATH];
};

// Represents the result of parsing a json data structure. Each leaf
// value (string, number, boolean, etc) gets an entry in the values
// list. The complete paths to those values are provided. This
// structure is optimized for quickly finding values at predetermined
// paths. It is compact and efficient.
#define JSON_MAX_PATHS 32
struct json_values {
    // Number of leaf values
    uint num_values;
    struct {
        // Path to data
        struct json_path path;
        // Offset and size of data value in buffer
        ulong data_offset;
        ulong data_sz;
    } values[JSON_MAX_PATHS];
    // Dynamic buffer containing all data
    char* buf;
    ulong buf_sz;
    ulong buf_alloc;
    char buf_init[2048];
};

// Initialize a json_values
void json_values_new(struct json_values* values);

// Destroy a json_values
void json_values_delete(struct json_values* values);

// Add a parsed value to a json_values
void json_add_value(struct json_values* values, struct json_path* path, const void* data, ulong data_sz, fd_spad_t * spad);

// Retrieve a value at a given path. A NULL is returned if the path
// isn't found
const void* json_get_value(struct json_values* values, const uint* path, uint path_sz, ulong* data_sz);

// Version that allows iterative retrieval. *pos should be initialized to zero.
const void* json_get_value_multi(struct json_values* values, const uint* path, uint path_sz, ulong* data_sz, uint * pos);

// Dump the values and paths to stdout
void json_values_printout(struct json_values* values);

// Parse a block of json. Returns 1 on success.
int json_values_parse(json_lex_state_t* lex, struct json_values* values, struct json_path* path);

#endif /* HEADER_fd_src_discof_rpcserver_fd_methods_h */
