use fxhash::FxHashMap;
use fxhash::{FxBuildHasher, FxHashSet};
use solana_accounts_db::ancestors::Ancestors;
use solana_runtime::banks_ancestors::HasAncestors;
#[allow(dead_code)]
use std::collections::{HashMap, HashSet};
use std::sync::atomic::{AtomicU64, Ordering};
use std::sync::Arc;

fn main() {
    println!("Hello, world!");
}

pub type Slot = u64;

struct MockBankForks {
    pub banks: HashMap<Slot, MockBank>,
    pub root: Arc<AtomicU64>,
}

impl MockBankForks {
    /// Create a map of bank slot id to the set of ancestors for the bank slot.
    pub fn ancestors(&self) -> HashMap<Slot, HashSet<Slot>> {
        let root = self.root();
        self.banks
            .iter()
            .map(|(slot, bank)| {
                let ancestors = bank.proper_ancestors().filter(|k| *k >= root);
                (*slot, ancestors.collect())
            })
            .collect()
    }

    pub fn root(&self) -> Slot {
        self.root.load(Ordering::Relaxed)
    }
}

struct MockBank {
    /// Bank slot (i.e. block)
    pub slot: Slot,
    /// The set of parents including this bank
    pub ancestors: Ancestors,
}

impl MockBank {
    /// Returns all ancestors excluding self.slot.
    pub(crate) fn proper_ancestors(&self) -> impl Iterator<Item = Slot> + '_ {
        self.ancestors
            .keys()
            .into_iter()
            .filter(move |slot| *slot != self.slot)
    }
}

impl HasAncestors for MockBank {
    fn get_ancestors_ref(&self) -> &Ancestors {
        &self.ancestors
    }

    fn proper_ancestors_iter<'s>(&'s self) -> impl Iterator<Item = Slot> + 's {
        self.proper_ancestors()
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use solana_pubkey::Pubkey;
    use solana_runtime::banks_ancestors::BanksAncestors;
    use std::hint::black_box;
    use std::time::Instant;

    /// creates a lineral forks thingy
    fn prep_test(start: u64, end: u64) -> MockBankForks {
        let mut banks = HashMap::new();

        for ii in start..=end {
            let anc = (start..=ii).collect::<Vec<_>>();
            let bank = MockBank {
                slot: ii,
                ancestors: Ancestors::from(anc),
            };

            banks.insert(ii, bank);
        }

        MockBankForks {
            banks,
            root: Arc::new(AtomicU64::new(start)),
        }
    }

    #[test]
    fn test_fogo_ancestors_og() {
        let start_slot = 10;
        let mut end_slot = 1000;

        let start_ver = Instant::now();
        for ii in 0..1000 {
            end_slot += 1;
            let forks = prep_test(start_slot, end_slot);
            let start = Instant::now();
            let ancestors = forks.ancestors();
            black_box(ancestors);
            let elapsed = start.elapsed();
            println!(
                "{} ({}->{}): {}ms",
                ii,
                start_slot,
                end_slot,
                elapsed.as_millis()
            );
        }
        let elapsed = start_ver.elapsed();
        println!("({}->{}): {}ms", start_slot, end_slot, elapsed.as_millis());
    }

    #[test]
    fn test_fogo_ancestors_cached() {
        let start_slot = 10;
        let mut end_slot = 1000;

        let start_ver = Instant::now();
        let mut cache = BanksAncestors::<MockBank>::new();
        for ii in 0..10000 {
            end_slot += 1;
            let forks = prep_test(start_slot, end_slot);
            let start = Instant::now();
            let ancestors = cache.clone_new_ancestors(&forks.banks, forks.root());
            black_box(ancestors);
            let elapsed = start.elapsed();
            println!(
                "{} ({}->{}): {}ms",
                ii,
                start_slot,
                end_slot,
                elapsed.as_millis()
            );
        }
        let elapsed = start_ver.elapsed();
        println!("({}->{}): {}ms", start_slot, end_slot, elapsed.as_millis());
    }

    #[test]
    fn test_fogo_sanity() {
        let start_slot = 10;
        let mut end_slot = 20;

        for ii in 0..10 {
            end_slot *= 2;
            let start = Instant::now();
            let mut slots = FxHashMap::with_capacity_and_hasher(
                (end_slot - start_slot) as usize,
                FxBuildHasher::default(),
            );

            for s in start_slot..end_slot {
                let mut sset = FxHashSet::with_capacity_and_hasher(
                    (s - start_slot) as usize,
                    FxBuildHasher::default(),
                );
                for ss in start_slot..s {
                    sset.insert(ss);
                }

                slots.insert(s, sset);
            }

            let elapsed = start.elapsed();
            println!(
                "{} ({}->{}): {}ms",
                ii,
                start_slot,
                end_slot,
                elapsed.as_millis()
            );
        }
    }

    #[test]
    fn test_fogo_hashing() {
        let start_slot = 10;
        let mut end_slot = 20;

        for ii in 0..10 {
            end_slot *= 2;
            let forks = prep_test(start_slot, end_slot);
            let start = Instant::now();

            let mut hashes = HashMap::with_capacity(forks.banks.len());
            for (slot, b) in &forks.banks {
                hashes.insert(*slot, b.ancestors.hash());
            }

            println!(
                "{} {} {}",
                hashes.get(&start_slot).unwrap(),
                hashes.get(&end_slot).unwrap(),
                hashes.get(&(start_slot + 1)).unwrap()
            );

            let elapsed = start.elapsed();
            println!(
                "{} ({}->{}): {}ms",
                ii,
                start_slot,
                end_slot,
                elapsed.as_millis()
            );
        }
    }

    #[test]
    fn test_dump_keys() {
        let zone_program_id = Pubkey::from_str_const("zpgQUDf86gkFLUFCLCNDxa3gmTYMo7YF7pQc8wMRjL2");
        let pubkey_zone = Pubkey::find_program_address(&[b"CUR_ZONES"], &zone_program_id).0;
        let pubkey_zone_validators =
            Pubkey::find_program_address(&[b"CUR_ZONE_VALIDATORS"], &zone_program_id).0;

        dbg!(pubkey_zone);
        dbg!(pubkey_zone_validators);
    }
}
