use {
    crate::{
        snapshot_bank_utils,
        snapshot_utils::{self, ArchiveFormat, SnapshotVersion, ZstdConfig},
    },
    solana_clock::Slot,
    std::{fs, num::NonZeroUsize, path::PathBuf},
    thiserror::Error,
};

/// Snapshot configuration and runtime information
#[derive(Clone, Debug)]
pub struct SnapshotConfig {
    /// Specifies the ways thats snapshots are allowed to be used
    pub usage: SnapshotUsage,

    /// Generate a new full snapshot archive every this many slots
    pub full_snapshot_archive_interval_slots: Slot,

    /// Generate a new incremental snapshot archive every this many slots
    pub incremental_snapshot_archive_interval_slots: Slot,

    /// Path to the directory where full snapshot archives are stored
    pub full_snapshot_archives_dir: PathBuf,

    /// Path to the directory where incremental snapshot archives are stored
    pub incremental_snapshot_archives_dir: PathBuf,

    /// Path to the directory where bank snapshots are stored
    pub bank_snapshots_dir: PathBuf,

    /// The archive format to use for snapshots
    pub archive_format: ArchiveFormat,

    /// Snapshot version to generate
    pub snapshot_version: SnapshotVersion,

    /// Maximum number of full snapshot archives to retain
    pub maximum_full_snapshot_archives_to_retain: NonZeroUsize,

    /// Maximum number of incremental snapshot archives to retain
    /// NOTE: Incremental snapshots will only be kept for the latest full snapshot
    pub maximum_incremental_snapshot_archives_to_retain: NonZeroUsize,

    // Thread niceness adjustment for snapshot packager service
    pub packager_thread_niceness_adj: i8,

    /// Max snapshot config
    pub max_snapshot: MaxSnapshotSlotConfig,
}

impl Default for SnapshotConfig {
    fn default() -> Self {
        Self {
            usage: SnapshotUsage::LoadAndGenerate,
            full_snapshot_archive_interval_slots:
                snapshot_bank_utils::DEFAULT_FULL_SNAPSHOT_ARCHIVE_INTERVAL_SLOTS,
            incremental_snapshot_archive_interval_slots:
                snapshot_bank_utils::DEFAULT_INCREMENTAL_SNAPSHOT_ARCHIVE_INTERVAL_SLOTS,
            full_snapshot_archives_dir: PathBuf::default(),
            incremental_snapshot_archives_dir: PathBuf::default(),
            bank_snapshots_dir: PathBuf::default(),
            archive_format: ArchiveFormat::TarZstd {
                config: ZstdConfig::default(),
            },
            snapshot_version: SnapshotVersion::default(),
            maximum_full_snapshot_archives_to_retain:
                snapshot_utils::DEFAULT_MAX_FULL_SNAPSHOT_ARCHIVES_TO_RETAIN,
            maximum_incremental_snapshot_archives_to_retain:
                snapshot_utils::DEFAULT_MAX_INCREMENTAL_SNAPSHOT_ARCHIVES_TO_RETAIN,
            packager_thread_niceness_adj: 0,
            max_snapshot: MaxSnapshotSlotConfig::None,
        }
    }
}

impl SnapshotConfig {
    /// A new snapshot config used for only loading at startup
    pub fn new_load_only() -> Self {
        Self {
            usage: SnapshotUsage::LoadOnly,
            ..Self::default()
        }
    }

    /// A new snapshot config used to disable snapshot generation and loading at
    /// startup
    pub fn new_disabled() -> Self {
        Self {
            usage: SnapshotUsage::Disabled,
            ..Self::default()
        }
    }

    /// Should snapshots be generated?
    pub fn should_generate_snapshots(&self) -> bool {
        self.usage == SnapshotUsage::LoadAndGenerate
    }

    /// Should snapshots be loaded?
    pub fn should_load_snapshots(&self) -> bool {
        self.usage == SnapshotUsage::LoadAndGenerate || self.usage == SnapshotUsage::LoadOnly
    }
}

/// Specify the ways that snapshots are allowed to be used
#[derive(Debug, Clone, Eq, PartialEq)]
pub enum SnapshotUsage {
    /// Snapshots are never generated or loaded at startup,
    /// instead start from genesis.
    Disabled,
    /// Snapshots are only used at startup, to load the accounts and bank
    LoadOnly,
    /// Snapshots are used everywhere; both at startup (i.e. load) and steady-state (i.e.
    /// generate).  This enables taking snapshots.
    LoadAndGenerate,
}

#[derive(Error, Debug)]
pub enum MaxSnapshotSlotError {
    #[error("snapshot backup directory {0} does not exist or is not valid.")]
    BackupDirectoryDoesNotExistOrInvalid(PathBuf),

    #[error("snapshot backup directory specified without max_snapshot_slot")]
    BackupSpecifiedWithoutMaxSnapshotSlot,
}

/// Max snapshot slot configuration
#[derive(Debug, Clone, Eq, PartialEq)]
pub enum MaxSnapshotSlotConfig {
    /// No max slot, work as normal
    None,
    /// Max snapshot slot, fail if there are newer snapshots
    FailAfter(Slot),
    /// Max snapshot slot with backup folder to move snapshots to
    BackupAfter {
        slot: Slot,
        backup_directory: PathBuf,
    },
}

impl MaxSnapshotSlotConfig {
    pub fn from_args(
        slot: Option<Slot>,
        backup_directory: Option<PathBuf>,
    ) -> Result<Self, MaxSnapshotSlotError> {
        match (slot, backup_directory) {
            (Some(slot), None) => Ok(Self::FailAfter(slot)),
            (Some(slot), Some(backup_directory)) => {
                // validate directory
                let meta = fs::metadata(&backup_directory).map_err(|_| {
                    MaxSnapshotSlotError::BackupDirectoryDoesNotExistOrInvalid(
                        backup_directory.clone(),
                    )
                })?;
                if !meta.is_dir() {
                    return Err(MaxSnapshotSlotError::BackupDirectoryDoesNotExistOrInvalid(
                        backup_directory.clone(),
                    ));
                }

                Ok(Self::BackupAfter {
                    slot,
                    backup_directory,
                })
            }
            (None, None) => Ok(Self::None),
            (None, Some(_)) => Err(MaxSnapshotSlotError::BackupSpecifiedWithoutMaxSnapshotSlot),
        }
    }
}
