use crate::installed_scheduler_pool::BankWithScheduler;
use solana_accounts_db::ancestors::Ancestors;
use solana_clock::Slot;
use std::collections::{HashMap, HashSet};
use std::marker::PhantomData;

pub type ClonedBankForksAncestors = HashMap<Slot, HashSet<Slot>>;

pub trait HasAncestors: 'static + Sized {
    fn get_ancestors_ref(&self) -> &Ancestors;

    fn proper_ancestors_iter<'s>(&'s self) -> impl Iterator<Item = Slot> + 's;
}

impl HasAncestors for BankWithScheduler {
    fn get_ancestors_ref(&self) -> &Ancestors {
        &self.ancestors
    }

    fn proper_ancestors_iter<'s>(&'s self) -> impl Iterator<Item = Slot> + 's {
        self.proper_ancestors()
    }
}

pub struct BanksAncestors<T: HasAncestors> {
    inner: Option<BanksAncestorsInner>,
    _bank: PhantomData<T>,
}

pub struct BanksAncestorsInner {
    last_value: ClonedBankForksAncestors,
    last_ancestors_hashes: HashMap<Slot, u64>,
    last_root: Slot,
}

impl<T: HasAncestors> Default for BanksAncestors<T> {
    fn default() -> Self {
        Self::new()
    }
}

impl<TBank: HasAncestors> BanksAncestors<TBank> {
    pub fn new() -> Self {
        Self {
            inner: None,
            _bank: Default::default(),
        }
    }

    fn clone_bank(bank: &TBank, slot: Slot, root: Slot) -> (Slot, HashSet<Slot>) {
        let ancestors = bank.proper_ancestors_iter().filter(|k| *k >= root);
        (slot, ancestors.collect())
    }

    fn initialize(&mut self, banks: &HashMap<Slot, TBank>, root: Slot) -> ClonedBankForksAncestors {
        self.inner = Some(BanksAncestorsInner {
            last_value: banks
                .iter()
                .map(|(slot, bank)| Self::clone_bank(bank, *slot, root))
                .collect(),
            last_ancestors_hashes: banks
                .iter()
                .map(|(slot, bank)| (*slot, bank.get_ancestors_ref().hash()))
                .collect(),
            last_root: root,
        });

        self.inner.as_ref().unwrap().last_value.clone()
    }

    pub fn clone_new_ancestors(
        &mut self,
        banks: &HashMap<Slot, TBank>,
        root: Slot,
    ) -> ClonedBankForksAncestors {
        let Some(inner) = &mut self.inner else {
            return self.initialize(banks, root);
        };

        // root go backwars, no good
        if root < inner.last_root {
            return self.initialize(banks, root);
        }

        // prune saved values
        inner.last_value.retain(|slot, _| banks.contains_key(slot));
        inner
            .last_ancestors_hashes
            .retain(|slot, _| banks.contains_key(slot));

        for (bank_slot, bank) in banks {
            let ancestor_hash = bank.get_ancestors_ref().hash();
            let Some(existing_value) = inner.last_value.get_mut(bank_slot) else {
                let clone = Self::clone_bank(bank, *bank_slot, root);
                inner.last_value.insert(clone.0, clone.1);
                inner.last_ancestors_hashes.insert(clone.0, ancestor_hash);
                continue;
            };

            let Some(hash) = inner.last_ancestors_hashes.get(bank_slot) else {
                panic!("Hash can't be missing if value isn't")
            };

            // if nothing changes
            if *hash == ancestor_hash && root == inner.last_root {
                continue;
            }

            // ancestors in the bank changed
            if *hash != ancestor_hash {
                *existing_value = Self::clone_bank(bank, *bank_slot, root).1;
                inner
                    .last_ancestors_hashes
                    .insert(*bank_slot, ancestor_hash);
            }

            // loot is larger then last root
            if root != inner.last_root {
                // prune all slots before root
                existing_value.retain(|slot| *slot >= root);
            }
        }

        inner.last_root = root;
        // all the values are updated, we are done
        inner.last_value.clone()
    }
}
