/// Helper for building non-overlapping core groups
pub struct CoreGroupBuilder<'a> {
    available_cores: &'a [usize],
    offset: usize,
}

impl<'a> CoreGroupBuilder<'a> {
    pub fn new(available_cores: &'a [usize]) -> Self {
        Self {
            available_cores,
            offset: 0,
        }
    }

    /// Take the next N cores
    pub fn take(&mut self, count: usize) -> Vec<usize> {
        let cores = self.fixed(count, self.offset);
        self.offset += cores.len();
        cores
    }

    /// Take a percentage of total available cores
    pub fn take_percent(&mut self, percent: f32) -> Vec<usize> {
        let count = (self.available_cores.len() as f32 * percent).ceil() as usize;
        self.take(count.max(1))
    }

    /// Take all remaining cores
    pub fn take_rest(&mut self) -> Vec<usize> {
        let cores = self.fixed(
            self.available_cores.len().saturating_sub(self.offset),
            self.offset,
        );
        self.offset = self.available_cores.len();
        cores
    }

    /// Skip N cores
    pub fn skip(&mut self, count: usize) -> &mut Self {
        self.offset += count;
        self
    }

    /// Get current offset
    pub fn offset(&self) -> usize {
        self.offset
    }

    /// Get remaining cores available
    pub fn remaining(&self) -> usize {
        self.available_cores.len().saturating_sub(self.offset)
    }

    // === Utility functions for creating core groups ===

    /// Get N cores starting at offset
    pub fn fixed(&self, count: usize, offset: usize) -> Vec<usize> {
        self.available_cores
            .iter()
            .skip(offset)
            .take(count)
            .copied()
            .collect()
    }

    /// Get a percentage of available cores starting at offset
    pub fn percent(&self, percent: f32, offset: usize) -> Vec<usize> {
        let count = (self.available_cores.len() as f32 * percent).ceil() as usize;
        let count = count.max(1);
        self.fixed(count, offset)
    }

    /// Get a range of cores [start..end]
    pub fn range(&self, start: usize, end: usize) -> Vec<usize> {
        self.available_cores
            .get(start..end)
            .map(|slice| slice.to_vec())
            .unwrap_or_default()
    }

    /// Split available cores into N equal groups
    pub fn split(&self, num_groups: usize) -> Vec<Vec<usize>> {
        if num_groups == 0 {
            return vec![];
        }

        let per_group = self.available_cores.len() / num_groups;
        let remainder = self.available_cores.len() % num_groups;

        let mut groups = Vec::new();
        let mut start = 0;

        for i in 0..num_groups {
            let extra = if i < remainder { 1 } else { 0 };
            let size = per_group + extra;
            let end = start + size;

            groups.push(self.available_cores[start..end].to_vec());
            start = end;
        }

        groups
    }

    /// Get all available cores
    pub fn all(&self) -> Vec<usize> {
        self.available_cores.to_vec()
    }

    /// Get the number of available cores
    pub fn count(&self) -> usize {
        self.available_cores.len()
    }
}
