use anyhow::Context;
use log::info;
use serde::{Deserialize, Serialize};
use std::fs;
use std::path::Path;

#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct Config {
    /// MOTD
    pub motd: String,
    /// Threads
    #[serde(default)]
    pub threads: ThreadsConfig,
    /// Affinity
    #[serde(default)]
    pub affinity: AffinityConfig,
    /// Rpc
    #[serde(default)]
    pub rpc: RpcConfig,
}

#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct ThreadsConfig {
    /// Rpc threads
    #[serde(default)]
    pub rpc: Option<usize>,
    /// RPC blocking
    #[serde(default)]
    pub rpc_blocking: Option<usize>,
}

#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct AffinityConfig {
    /// Affinity manager on
    #[serde(default)]
    pub use_affinity_manager: bool,

    /// Rpc group core count
    #[serde(default)]
    pub rpc_core_count: Option<usize>,
}

#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct RpcConfig {
    /// Enable block
    #[serde(default)]
    pub enable_block_pubsub: Option<bool>,
    /// Enable vote
    #[serde(default)]
    pub enable_vote_pubsub: Option<bool>,
    /// Pubsub limit
    #[serde(default)]
    pub max_active_subscriptions: Option<usize>,
    /// Scan limit bytes
    #[serde(default)]
    pub scan_limit_bytes: Option<usize>,
}

pub(crate) fn load_config(path: &Path) -> anyhow::Result<Config> {
    info!("[fogo_validator] Loading config from: {:?}", path);
    let contents = fs::read_to_string(path).context("loading config")?;
    let contents = toml::from_str(&contents).context("parsing config")?;
    info!("[fogo_validator] Loaded configs: {:#?}", contents);

    Ok(contents)
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn test_load_example_config() {
        let config = toml::from_str::<Config>(include_str!("../config_example.toml")).unwrap();
        assert_eq!(config.threads.rpc_blocking, Some(10));
    }
}
