use libc::{cpu_set_t, sched_getaffinity, sched_setaffinity, CPU_ISSET, CPU_SET, CPU_ZERO};
use log::info;
use std::collections::HashMap;
use std::fs;
use std::mem;
use std::str::FromStr;
use std::sync::{Arc, RwLock};
use std::thread;
use std::time::Duration;

const MAX_CPUS: usize = 256;
const SLEEP_TIME_MS: u64 = 5_000;

pub struct AffinityManager {
    rules: Arc<RwLock<HashMap<String, Vec<usize>>>>,
    available_cores: Vec<usize>,
    running: Arc<std::sync::atomic::AtomicBool>,
}

impl AffinityManager {
    pub fn new() -> Self {
        let available_cores = Self::get_process_affinity();
        info!(
            "[fogo_aff] Available cores from firedancer: {:?}",
            available_cores
        );

        Self {
            rules: Arc::new(RwLock::new(HashMap::new())),
            available_cores,
            running: Arc::new(std::sync::atomic::AtomicBool::new(false)),
        }
    }

    /// Get the cores that the C parent process allowed us to use
    fn get_process_affinity() -> Vec<usize> {
        unsafe {
            let mut cpuset: cpu_set_t = mem::zeroed();

            if sched_getaffinity(0, mem::size_of::<cpu_set_t>(), &mut cpuset) == 0 {
                (0..MAX_CPUS) // CPU_SETSIZE is usually 1024, but 128 is reasonable
                    .filter(|&i| CPU_ISSET(i, &cpuset))
                    .collect()
            } else {
                // Fallback: assume all cores available
                (0..num_cpus::get()).collect()
            }
        }
    }

    /// Add rule for a given pattern and list of cores
    pub fn add_rule(&self, name_pattern: &str, core_ids: &Vec<usize>) {
        if !core_ids.is_empty() {
            info!("[fogo_aff]Rule '{}': cores {:?}", name_pattern, core_ids);
            self.rules
                .write()
                .unwrap()
                .insert(name_pattern.to_string(), core_ids.clone());
        }
    }

    pub fn available_cores(&self) -> &[usize] {
        &self.available_cores
    }

    pub fn start(&self) {
        self.running
            .store(true, std::sync::atomic::Ordering::Relaxed);
        let rules = self.rules.clone();
        let running = self.running.clone();

        thread::Builder::new()
            .name("affinity-manager".to_string())
            .spawn(move || {
                while running.load(std::sync::atomic::Ordering::Relaxed) {
                    apply_affinity_rules(&rules);
                    thread::sleep(Duration::from_millis(SLEEP_TIME_MS));
                }
            })
            .unwrap();
    }

    pub fn stop(&self) {
        self.running
            .store(false, std::sync::atomic::Ordering::Relaxed);
    }
}

fn apply_affinity_rules(rules: &RwLock<HashMap<String, Vec<usize>>>) {
    let rules = rules.read().unwrap();
    if rules.is_empty() {
        return;
    }

    let Ok(entries) = fs::read_dir("/proc/self/task") else {
        return;
    };

    for entry in entries.flatten() {
        let tid = entry.file_name().to_string_lossy().to_string();
        let ntid = i32::from_str(tid.as_str()).expect("thread id non number");

        let comm_path = format!("/proc/self/task/{}/comm", tid);
        let Ok(name) = fs::read_to_string(&comm_path) else {
            continue;
        };
        let name = name.trim();
        let aff = get_thread_affinity(ntid);

        for (pattern, core_ids) in rules.iter() {
            if matches_pattern(name, pattern) {
                if core_ids == &aff {
                    break;
                }

                set_thread_affinity(&tid, core_ids);
                break;
            }
        }
    }
}

fn matches_pattern(name: &str, pattern: &str) -> bool {
    if pattern.contains('*') {
        let regex_pattern = pattern.replace('.', "\\.").replace('*', ".*");
        regex::Regex::new(&format!("^{}$", regex_pattern))
            .map(|re| re.is_match(name))
            .unwrap_or(false)
    } else {
        name == pattern
    }
}

pub fn get_thread_affinity(tid: i32) -> Vec<usize> {
    unsafe {
        let mut cpuset: cpu_set_t = mem::zeroed();

        if sched_getaffinity(tid, mem::size_of::<cpu_set_t>(), &mut cpuset) == 0 {
            (0..MAX_CPUS).filter(|&i| CPU_ISSET(i, &cpuset)).collect()
        } else {
            vec![]
        }
    }
}

fn set_thread_affinity(tid: &str, core_ids: &[usize]) {
    let Ok(tid_num) = tid.parse::<i32>() else {
        return;
    };

    unsafe {
        let mut cpuset: cpu_set_t = mem::zeroed();
        CPU_ZERO(&mut cpuset);

        for &core_id in core_ids {
            CPU_SET(core_id, &mut cpuset);
        }

        sched_setaffinity(tid_num, mem::size_of::<cpu_set_t>(), &cpuset);
    }
}
