#!/usr/bin/env bash
set -e

cd "$(dirname "$0")/.."

if [[ -n $APPVEYOR ]]; then
  # Bootstrap rust build environment
  source ci/env.sh
  source ci/rust-version.sh

  appveyor DownloadFile https://win.rustup.rs/ -FileName rustup-init.exe
  export USERPROFILE="D:\\"
  ./rustup-init -yv --default-toolchain "$rust_stable" --default-host x86_64-pc-windows-msvc
  export PATH="$PATH:/d/.cargo/bin"
  rustc -vV
  cargo -vV
fi

DRYRUN=
if [[ -z $CI_BRANCH ]]; then
  DRYRUN="echo"
  CHANNEL=unknown
fi

eval "$(ci/channel-info.sh)"

TAG=
if [[ -n "$CI_TAG" ]]; then
  CHANNEL_OR_TAG=$CI_TAG
  TAG="$CI_TAG"
else
  CHANNEL_OR_TAG=$CHANNEL
fi

if [[ -z $CHANNEL_OR_TAG ]]; then
  echo +++ Unable to determine channel or tag to publish into, exiting.
  exit 0
fi

case "$CI_OS_NAME" in
osx)
  _cputype="$(uname -m)"
  if [[ $_cputype = arm64 ]]; then
    _cputype=aarch64
  fi
  TARGET=${_cputype}-apple-darwin
  ;;
linux)
  TARGET=x86_64-unknown-linux-gnu
  ;;
windows)
  TARGET=x86_64-pc-windows-msvc
  # Enable symlinks used by some build.rs files
  # source: https://stackoverflow.com/a/52097145/10242004
  (
    set -x
    git --version
    git config core.symlinks true
    find . -type l -delete
    git reset --hard
    # patched crossbeam doesn't build on windows
    sed -i 's/^crossbeam-epoch/#crossbeam-epoch/' Cargo.toml
  )
  ;;
*)
  echo CI_OS_NAME unset
  exit 1
  ;;
esac

RELEASE_BASENAME="${RELEASE_BASENAME:=solana-release}"
TARBALL_BASENAME="${TARBALL_BASENAME:="$RELEASE_BASENAME"}"

echo --- Creating release tarball
(
  set -x
  rm -rf "${RELEASE_BASENAME:?}"/
  mkdir "${RELEASE_BASENAME}"/

  COMMIT="$(git rev-parse HEAD)"

  (
    echo "channel: $CHANNEL_OR_TAG"
    echo "commit: $COMMIT"
    echo "target: $TARGET"
  ) > "${RELEASE_BASENAME}"/version.yml

  # Make CHANNEL available to include in the software version information
  export CHANNEL

  source ci/rust-version.sh stable
  scripts/cargo-install-all.sh stable "${RELEASE_BASENAME}"

  tar cvf "${TARBALL_BASENAME}"-$TARGET.tar "${RELEASE_BASENAME}"
  bzip2 "${TARBALL_BASENAME}"-$TARGET.tar
  cp "${RELEASE_BASENAME}"/bin/agave-install-init agave-install-init-$TARGET
  cp "${RELEASE_BASENAME}"/version.yml "${TARBALL_BASENAME}"-$TARGET.yml
)

# Maybe tarballs are platform agnostic, only publish them from the Linux build
MAYBE_TARBALLS=
if [[ "$CI_OS_NAME" = linux ]]; then
  (
    set -x
    platform-tools-sdk/sbf/scripts/package.sh
    [[ -f sbf-sdk.tar.bz2 ]]
  )
  MAYBE_TARBALLS="sbf-sdk.tar.bz2"
fi

source ci/upload-ci-artifact.sh

for file in "${TARBALL_BASENAME}"-$TARGET.tar.bz2 "${TARBALL_BASENAME}"-$TARGET.yml agave-install-init-"$TARGET"* $MAYBE_TARBALLS; do
  if [[ -n $DO_NOT_PUBLISH_TAR ]]; then
    upload-ci-artifact "$file"
    echo "Skipped $file due to DO_NOT_PUBLISH_TAR"
    continue
  fi

  if [[ -n $BUILDKITE ]]; then
    echo --- GCS Store: "$file"
    upload-gcs-artifact "/solana/$file" gs://anza-release/"$CHANNEL_OR_TAG"/"$file"

    echo Published to:
    $DRYRUN ci/format-url.sh https://release.anza.xyz/"$CHANNEL_OR_TAG"/"$file"

    if [[ -n $TAG ]]; then
      ci/upload-github-release-asset.sh "$file"
    fi
  elif [[ -n $GITHUB_ACTIONS ]]; then
    mkdir -p github-action-s3-upload/"$CHANNEL_OR_TAG"
    cp -v "$file" github-action-s3-upload/"$CHANNEL_OR_TAG"/

    if [[ -n $TAG ]]; then
      mkdir -p github-action-release-upload/
      cp -v "$file" github-action-release-upload/
    fi
  elif [[ -n $APPVEYOR ]]; then
    # Add artifacts for .appveyor.yml to upload
    appveyor PushArtifact "$file" -FileName "$CHANNEL_OR_TAG"/"$file"
  fi
done

echo --- ok
