#ifndef HEADER_fd_src_flamenco_runtime_fd_runtime_h
#define HEADER_fd_src_flamenco_runtime_fd_runtime_h

#include "stdarg.h"

#include "../fd_flamenco_base.h"
#include "fd_runtime_err.h"
#include "fd_runtime_init.h"
#include "fd_rocksdb.h"
#include "fd_acc_mgr.h"
#include "fd_hashes.h"
#include "../features/fd_features.h"
#include "fd_rent_lists.h"
#include "../../ballet/poh/fd_poh.h"
#include "../leaders/fd_leaders.h"
#include "context/fd_exec_slot_ctx.h"
#include "context/fd_capture_ctx.h"
#include "context/fd_exec_txn_ctx.h"
#include "info/fd_runtime_block_info.h"
#include "info/fd_instr_info.h"
#include "../gossip/fd_gossip.h"
#include "../repair/fd_repair.h"
#include "../../disco/pack/fd_microblock.h"
#include "info/fd_microblock_info.h"
#include "../../ballet/bmtree/fd_wbmtree.h"
#include "../../ballet/sbpf/fd_sbpf_loader.h"
#include "fd_runtime_public.h"

/* Various constant values used by the runtime. */

#define MICRO_LAMPORTS_PER_LAMPORT (1000000UL)

#define DEFAULT_HASHES_PER_TICK  (12500)
#define UPDATED_HASHES_PER_TICK2 (17500)
#define UPDATED_HASHES_PER_TICK3 (27500)
#define UPDATED_HASHES_PER_TICK4 (47500)
#define UPDATED_HASHES_PER_TICK5 (57500)
#define UPDATED_HASHES_PER_TICK6 (62500)

#define FD_RUNTIME_TRACE_NONE   (0)
#define FD_RUNTIME_TRACE_SAVE   (1)
#define FD_RUNTIME_TRACE_REPLAY (2)

#define FD_RUNTIME_OFFLINE_NUM_ROOT_BLOCKS (6UL) /* 6 root blocks for offline replay */

#define FD_RENT_EXEMPT_RENT_EPOCH (ULONG_MAX)

#define SECONDS_PER_YEAR ((double)(365.242199 * 24.0 * 60.0 * 60.0))

/* TODO: increase this to default once we have enough memory to support a 95G status cache. */
#define MAX_CACHE_TXNS_PER_SLOT (FD_TXNCACHE_DEFAULT_MAX_TRANSACTIONS_PER_SLOT / 8)

struct fd_execute_txn_task_info {
  fd_spad_t * *       spads;
  fd_spad_t *         spad;
  fd_exec_txn_ctx_t * txn_ctx;
  fd_txn_p_t *        txn;
  int                 exec_res;
};
typedef struct fd_execute_txn_task_info fd_execute_txn_task_info_t;

struct fd_raw_block_txn_iter {
  fd_block_entry_batch_t const * curr_batch;
  uchar const * orig_data;
  ulong remaining_batches;
  ulong remaining_microblocks;
  ulong remaining_txns;
  ulong curr_offset;

  ulong curr_txn_sz;
};

typedef struct fd_raw_block_txn_iter fd_raw_block_txn_iter_t;

struct fd_poh_verifier {
  union {
    fd_microblock_hdr_t const * hdr;
    uchar * raw;
  } microblock;
  fd_hash_t const * in_poh_hash;
  ulong microblk_max_sz;
  fd_spad_t * spad;
  int success;
};
typedef struct fd_poh_verifier fd_poh_verifier_t;

/* The below logic is used to size out the memory footprint generated by the
   runtime during transaction execution. */

/* The prevailing layout we have in the runtime is the meta followed by
   the account's data. This struct encodes that layout and asserts that
   the alignment requirements of the constituents are satisfied. */
// TODO: Use this struct at allocation sites so it's clear we use this layout
struct __attribute__((packed)) fd_account_rec {
  fd_account_meta_t meta;
  uchar data[];
};
typedef struct fd_account_rec fd_account_rec_t;
#define FD_ACCOUNT_REC_ALIGN      (8UL)
#define FD_ACCOUNT_REC_DATA_ALIGN (8UL)
FD_STATIC_ASSERT( FD_ACCOUNT_REC_ALIGN>=FD_ACCOUNT_META_ALIGN,     account_rec_meta_align );
FD_STATIC_ASSERT( FD_ACCOUNT_REC_ALIGN>=FD_ACCOUNT_REC_DATA_ALIGN, account_rec_data_align );
FD_STATIC_ASSERT( (offsetof(fd_account_rec_t, meta)%FD_ACCOUNT_META_ALIGN)==0,     account_rec_meta_offset );
FD_STATIC_ASSERT( (offsetof(fd_account_rec_t, data)%FD_ACCOUNT_REC_DATA_ALIGN)==0, account_rec_data_offset );

#define MAX_PERMITTED_DATA_INCREASE (10240UL) // 10KB
#define FD_BPF_ALIGN_OF_U128        (8UL    )
FD_STATIC_ASSERT( FD_BPF_ALIGN_OF_U128==FD_ACCOUNT_REC_DATA_ALIGN, input_data_align );
#define FD_RUNTIME_INPUT_REGION_ALLOC_ALIGN_UP (16UL)

/******** These macros bound out memory footprint ********/

/* The tight upper bound on borrowed account footprint over the
   execution of a single transaction. */
#define FD_RUNTIME_BORROWED_ACCOUNT_FOOTPRINT (MAX_TX_ACCOUNT_LOCKS * FD_ULONG_ALIGN_UP( FD_ACC_TOT_SZ_MAX, FD_ACCOUNT_REC_ALIGN ))

/* The tight-ish upper bound on input region footprint over the
   execution of a single transaction. See input serialization code for
   reference: fd_bpf_loader_serialization.c

   This bound is based off of the transaction MTU. We consider the
   question of what kind of transaction one would construct to
   maximally bloat the input region.
   The worst case scenario is when every nested instruction references
   all unique accounts in the transaction. A transaction can lock a max
   of MAX_TX_ACCOUNT_LOCKS accounts. Then all remaining input account
   references are going to be duplicates, which cost 1 byte to specify
   offset in payload, and which cost 8 bytes during serialization. Then
   there would be 0 bytes of instruction data, because they exist byte
   for byte in the raw payload, which is not a worthwhile bloat factor.
 */
#define FD_RUNTIME_INPUT_REGION_UNIQUE_ACCOUNT_FOOTPRINT(direct_mapping)                                                                                      \
                                                        (1UL                         /* dup byte          */                                                + \
                                                         sizeof(uchar)               /* is_signer         */                                                + \
                                                         sizeof(uchar)               /* is_writable       */                                                + \
                                                         sizeof(uchar)               /* executable        */                                                + \
                                                         sizeof(uint)                /* original_data_len */                                                + \
                                                         sizeof(fd_pubkey_t)         /* key               */                                                + \
                                                         sizeof(fd_pubkey_t)         /* owner             */                                                + \
                                                         sizeof(ulong)               /* lamports          */                                                + \
                                                         sizeof(ulong)               /* data len          */                                                + \
                                                         (direct_mapping ? FD_BPF_ALIGN_OF_U128 : FD_ULONG_ALIGN_UP( FD_ACC_SZ_MAX, FD_BPF_ALIGN_OF_U128 )) + \
                                                         MAX_PERMITTED_DATA_INCREASE                                                                        + \
                                                         sizeof(ulong))              /* rent_epoch        */

#define FD_RUNTIME_INPUT_REGION_INSN_FOOTPRINT(account_lock_limit, direct_mapping)                                                                       \
                                              (FD_ULONG_ALIGN_UP( (sizeof(ulong)         /* acct_cnt       */                                          + \
                                                                   account_lock_limit*FD_RUNTIME_INPUT_REGION_UNIQUE_ACCOUNT_FOOTPRINT(direct_mapping) + \
                                                                   sizeof(ulong)         /* instr data len */                                          + \
                                                                                         /* No instr data  */                                            \
                                                                   sizeof(fd_pubkey_t)), /* program id     */                                            \
                                                                   FD_RUNTIME_INPUT_REGION_ALLOC_ALIGN_UP ) + FD_BPF_ALIGN_OF_U128)

#define FD_RUNTIME_INPUT_REGION_TXN_FOOTPRINT(account_lock_limit, direct_mapping)                                                                           \
                                             ((FD_MAX_INSTRUCTION_STACK_DEPTH*FD_RUNTIME_INPUT_REGION_INSN_FOOTPRINT(account_lock_limit, direct_mapping)) + \
                                              ((FD_TXN_MTU-FD_TXN_MIN_SERIALIZED_SZ-account_lock_limit)*8UL)) /* We can have roughly this much duplicate offsets */

/* Bincode valloc footprint over the execution of a single transaction.
   As well as other footprint specific to each native program type.

   N.B. We know that bincode valloc footprint is bounded, because
   whenever we alloc something, we advance our pointer into the binary
   buffer, so eventually we are gonna reach the end of the buffer.
   This buffer is usually backed by and ultimately bounded in size by
   either accounts data or the transaction MTU.

   That being said, it's not obvious what the tight upper bound would
   be for allocations across all possible execution paths of all native
   programs, including possible CPIs from native programs.  The
   footprint estimate here is based on a manual review of our native
   program implementation.  Note that even if the possible paths remain
   steady at the Solana protocol level, the footprint is subject to
   change when we change our implementation.

   ### Native programs
   ALUT (migrated to BPF)
   Loader
     - rodata for bpf program relocation and validation
   Compute budget (0 allocations)
   Config (migrated to BPF)
   Precompile (0 allocations)
   Stake
     - The instruction with the largest footprint is deactivate_delinquent
       - During instruction decode, no allocations
       - During execution, this is (vote account get_state() + vote convert_to_current()) times 2, once for delinquent_vote_account, and once for reference_vote_account
   System
     - system_program_instruction_decode seed
   Vote
     - The instruction with the largest footprint is compact vote state update
       - During instruction decode, this is 9*lockouts_len bytes, MTU bounded
       - During execution, this is vote account get_state() + vote convert_to_current() + 12*lockouts_len bytes + lockouts_len ulong + deq_fd_landed_vote_t_alloc(lockouts_len)
   Zk Elgamal (0 allocations)

   The largest footprint is hence deactivate_delinquent, in which the
   two get_state() calls dominate the footprint.  In particular, the
   authorized_voters treaps bloat 40 bytes (epoch+pubkey) in a vote
   account to 72 bytes (sizeof(fd_vote_authorized_voter_t)) in memory.
 */
#define FD_RUNTIME_BINCODE_AND_NATIVE_FOOTPRINT (2UL*FD_ACC_SZ_MAX*72UL/40UL)

/* Misc other footprint. */
#define FD_RUNTIME_SYSCALL_TABLE_FOOTPRINT (FD_MAX_INSTRUCTION_STACK_DEPTH*FD_ULONG_ALIGN_UP(FD_SBPF_SYSCALLS_FOOTPRINT, FD_SBPF_SYSCALLS_ALIGN))

#ifdef FD_DEBUG_SBPF_TRACES
#define FD_RUNTIME_VM_TRACE_EVENT_MAX      (1UL<<30)
#define FD_RUNTIME_VM_TRACE_EVENT_DATA_MAX (2048UL)
#define FD_RUNTIME_VM_TRACE_FOOTPRINT (FD_MAX_INSTRUCTION_STACK_DEPTH*fd_ulong_align_up( fd_vm_trace_footprint( FD_RUNTIME_VM_TRACE_EVENT_MAX, FD_RUNTIME_VM_TRACE_EVENT_DATA_MAX ), fd_vm_trace_align() ))
#else
#define FD_RUNTIME_VM_TRACE_FOOTPRINT (0UL)
#endif

#define FD_RUNTIME_MISC_FOOTPRINT (FD_RUNTIME_SYSCALL_TABLE_FOOTPRINT+FD_RUNTIME_VM_TRACE_FOOTPRINT)

/* Now finally, we bound out the footprint of transaction execution. */
#define FD_RUNTIME_TRANSACTION_EXECUTION_FOOTPRINT(account_lock_limit, direct_mapping)                                         \
                                                  (FD_RUNTIME_BORROWED_ACCOUNT_FOOTPRINT                                     + \
                                                   FD_RUNTIME_INPUT_REGION_TXN_FOOTPRINT(account_lock_limit, direct_mapping) + \
                                                   FD_RUNTIME_BINCODE_AND_NATIVE_FOOTPRINT                                   + \
                                                   FD_RUNTIME_MISC_FOOTPRINT)

/* Convenience macros for common use cases.

   TODO: If account lock limits are increased to 128, this macro will need to be updated. */
#define FD_RUNTIME_TRANSACTION_EXECUTION_FOOTPRINT_FUZZ    FD_RUNTIME_TRANSACTION_EXECUTION_FOOTPRINT(64UL, 0)
#define FD_RUNTIME_TRANSACTION_EXECUTION_FOOTPRINT_DEFAULT FD_RUNTIME_TRANSACTION_EXECUTION_FOOTPRINT(64UL, 0)

/* Footprint here is dominated by vote account decode.  See above for
   why 72/40. */
#define FD_RUNTIME_TRANSACTION_FINALIZATION_FOOTPRINT      (FD_ACC_SZ_MAX*72UL/40UL)

/* The below macros aren't used anywhere, but since the spads are used for PoH tick verification,
   we ensure that the default spad size is large enough for the wbmtree and leaves */

#define FD_RUNTIME_MERKLE_LEAF_CNT_MAX (FD_TXN_MAX_PER_SLOT * FD_TXN_ACTUAL_SIG_MAX)

#define FD_RUNTIME_MERKLE_VERIFICATION_FOOTPRINT FD_RUNTIME_MERKLE_LEAF_CNT_MAX * sizeof(fd_wbmtree32_leaf_t)     /* leaves */   \
                                                + sizeof(fd_wbmtree32_t) + sizeof(fd_wbmtree32_node_t)*(FD_RUNTIME_MERKLE_LEAF_CNT_MAX + (FD_RUNTIME_MERKLE_LEAF_CNT_MAX/2)) /* tree footprint */ \
                                                + FD_RUNTIME_MERKLE_LEAF_CNT_MAX * (sizeof(fd_ed25519_sig_t) + 1) /* sig mbuf */ \
                                                + 1UL + FD_WBMTREE32_ALIGN + alignof(fd_wbmtree32_leaf_t)         /* aligns */

FD_STATIC_ASSERT( FD_RUNTIME_MERKLE_VERIFICATION_FOOTPRINT <= FD_RUNTIME_TRANSACTION_EXECUTION_FOOTPRINT_DEFAULT, merkle verify footprint exceeds txn execution footprint );

/* Helpers for runtime public frame management. */

/* Helpers for runtime spad frame management. */
struct fd_runtime_spad_verify_handle_private {
  fd_spad_t *         spad;
  fd_exec_txn_ctx_t * txn_ctx;
};
typedef struct fd_runtime_spad_verify_handle_private fd_runtime_spad_verify_handle_private_t;

static inline void
fd_runtime_spad_private_frame_end( fd_runtime_spad_verify_handle_private_t * _spad_handle ) {
  /* fd_spad_verify() returns 0 if everything looks good, and non-zero
     otherwise.

     Since the fast spad alloc API doesn't check for or indicate an OOM
     situation and is going to happily permit an OOB alloc, we need
     some way of detecting that. Moreover, we would also like to detect
     unbalanced frame push/pop or usage of more frames than allowed.
     While surrounding the spad with guard regions will help detect the
     former, it won't necessarily catch the latter.

     On compliant transactions, fd_spad_verify() isn't all that
     expensive.  Nonetheless, We invoke fd_spad_verify() only at the
     peak of memory usage, and not gratuitously everywhere. One peak
     would be right before we do the most deeply nested spad frame pop.
     However, we do pops through compiler-inserted cleanup functions
     that take only a single pointer, so we define this helper function
     to access the needed context info.  The end result is that we do
     super fast spad calls everywhere in the runtime, and every now and
     then we invoke verify to check things. */
  /* -1UL because spad pop is called after instr stack pop. */
  if( FD_UNLIKELY( _spad_handle->txn_ctx->instr_stack_sz>=FD_MAX_INSTRUCTION_STACK_DEPTH-1UL && fd_spad_verify( _spad_handle->txn_ctx->spad ) ) ) {
    uchar const * txn_signature = (uchar const *)fd_txn_get_signatures( _spad_handle->txn_ctx->txn_descriptor, _spad_handle->txn_ctx->_txn_raw->raw );
    FD_BASE58_ENCODE_64_BYTES( txn_signature, sig );
    FD_LOG_ERR(( "spad corrupted or overflown on transaction %s", sig ));
  }
  fd_spad_pop( _spad_handle->spad );
}

#define FD_RUNTIME_TXN_SPAD_FRAME_BEGIN(_spad, _txn_ctx) do {                                                        \
  fd_runtime_spad_verify_handle_private_t _spad_handle __attribute__((cleanup(fd_runtime_spad_private_frame_end))) = \
    (fd_runtime_spad_verify_handle_private_t) { .spad = _spad, .txn_ctx = _txn_ctx };                                \
  fd_spad_push( _spad_handle.spad );                                                                                 \
  do

#define FD_RUNTIME_TXN_SPAD_FRAME_END while(0); } while(0)

FD_PROTOTYPES_BEGIN

/* Runtime Helpers ************************************************************/

/*
   Returns 0 on success, and non zero otherwise.  On failure, the
   out values will not be modified.
 */
int
fd_runtime_compute_max_tick_height( ulong   ticks_per_slot,
                                    ulong   slot,
                                    ulong * out_max_tick_height /* out */ );

void
fd_runtime_update_leaders( fd_bank_t * bank,
                           ulong       slot,
                           fd_spad_t * runtime_spad );

/* TODO: Invoked by fd_executor: layering violation. Rent logic is deprecated
   and will be torn out entirely very soon. */
ulong
fd_runtime_collect_rent_from_account( fd_epoch_schedule_t const * schedule,
                                      fd_rent_t const *           rent,
                                      double                      slots_per_year,
                                      fd_txn_account_t *          acc,
                                      ulong                       epoch );

void
fd_runtime_update_slots_per_epoch( fd_bank_t * bank,
                                   ulong       slots_per_epoch );

/* Block Level Execution Prep/Finalize ****************************************/

#define FD_BLOCK_OK                          (0UL)
#define FD_BLOCK_ERR_INCOMPLETE              (1UL)
#define FD_BLOCK_ERR_INVALID_ENTRY_HASH      (2UL)
#define FD_BLOCK_ERR_INVALID_LAST_TICK       (3UL)
#define FD_BLOCK_ERR_TOO_FEW_TICKS           (4UL)
#define FD_BLOCK_ERR_TOO_MANY_TICKS          (5UL)
#define FD_BLOCK_ERR_INVALID_TICK_HASH_COUNT (6UL)
#define FD_BLOCK_ERR_TRAILING_ENTRY          (7UL)
#define FD_BLOCK_ERR_DUPLICATE_BLOCK         (8UL)

/*
   https://github.com/anza-xyz/agave/blob/v2.1.0/ledger/src/blockstore_processor.rs#L1096
   This function assumes a full block.
   This needs to be called after epoch processing to get the up to date
   hashes_per_tick.

   Provide scratch memory >= the max size of a batch to use. This is because we can only
   assemble shreds by batch, so we iterate and assemble shreds by batch in this function
   without needing the caller to do so.
 */
ulong
fd_runtime_block_verify_ticks( fd_blockstore_t * blockstore,
                               ulong             slot,
                               uchar *           block_data_mem,
                               ulong             block_data_sz,
                               ulong             tick_height,
                               ulong             max_tick_height,
                               ulong             hashes_per_tick );

/* The following microblock-level functions are exposed and non-static due to also being used for fd_replay.
   The block-level equivalent functions, on the other hand, are mostly static as they are only used
   for offline replay */

/* extra fine-grained streaming tick verification */
int
fd_runtime_microblock_verify_ticks( fd_blockstore_t *           blockstore,
                                    ulong                       slot,
                                    fd_microblock_hdr_t const * hdr,
                                    bool               slot_complete,
                                    ulong              tick_height,
                                    ulong              max_tick_height,
                                    ulong              hashes_per_tick );

/*
   fd_runtime_microblock_verify_read_write_conflicts verifies that a
   list of txns (e.g., those in a microblock) do not have read-write
   or write-write conflits. FD_TXN_CONFLICT_MAP_MAX_NACCT defines a
   conservative estimation of the number of accounts touched by txns
   in one slot. Given the conservative estimation, the footprint of
   the account map (fd_conflict_detect_map) is about 2112MB. One can
   certainly use a better estimation leading to a smaller footprint.

   this function corresponds to try_lock_accounts in Agave which
   detects transaction conflicts:
   https://github.com/anza-xyz/agave/blob/v2.2.3/runtime/src/bank.rs
   #L3145

   Specifically, from the replay stage of Agave, the control flow is
   (1) replay_blockstore_into_bank: https://github.com/anza-xyz/agave/
   blob/v2.2.3/core/src/replay_stage.rs#L2232
   (2) confirm_slot: https://github.com/anza-xyz/agave/blob/v2.2.3/
   ledger/src/blockstore_processor.rs#L1561
   (3) confirm_slot_entries: https://github.com/anza-xyz/agave/blob/
   v2.2.3/ledger/src/blockstore_processor.rs#L1609
   (4) process_entries: https://github.com/anza-xyz/agave/blob/v2.2.3/
   ledger/src/blockstore_processor.rs#L704
   (5) queue_batches_with_lock_retry: https://github.com/anza-xyz/agave/
   blob/v2.2.3/ledger/src/blockstore_processor.rs#L789
   (6) bank.try_lock_accounts is called in queue_batches_with_lock_retry
   (7) this try_lock_accounts eventually calls another try_lock_accounts,
       (see https://github.com/anza-xyz/agave/blob/v2.2.3/accounts-db/src
       /account_locks.rs#L24), which acquires the locks and returns
       TransactionError::AccountInUse if r-w or w-w conflict is detected
   (8) when calling try_lock_accounts, function accounts_with_is_writable
   is used to decide whether an account is writable (see https://github.
   com/anza-xyz/agave/blob/v2.2.3/accounts-db/src/accounts.rs#L605) which
   internally calls the is_writable function depending on whether the txn
   is legacy or V0:

   is_writable for legacy: https://github.com/anza-xyz/solana-sdk/blob/
   message%40v2.2.1/message/src/sanitized.rs#L75

   is_writable for V0: https://github.com/anza-xyz/solana-sdk/blob/
   message%40v2.2.1/message/src/versions/v0/loaded.rs#L152

   In both cases, Agave does the following check in addition to whether
   an account has been specified as writable in the transaction message
   (https://github.com/anza-xyz/solana-sdk/blob/message%40v2.2.1/message
   /src/versions/v0/loaded.rs#L146). This additional check is handled by
   function fd_txn_account_is_writable_idx_flat in our code.

   txns is an array containing txn_cnt transactions in fd_txn_p_t type;
   acct_map is used to detect conflicts and acct_arr is used to clear the
   map before the function returns; funk and funk_txn are used to read
   Solana accounts for address lookup tables; slot and slot_hashes are
   needed for checking certain bounds in the address lookup table system
   program; features is needed in fd_txn_account_is_writable_idx_flat to
   decide whether a writable account is demoted to read-only.

   If an error occurs in runtime, the function returns the runtime error.
   If there's no conflict, the return value is FD_RUNTIME_EXECUTE_SUCCESS
   and out_conflict_detected will be 0. If there's a conflict, the return
   value is FD_RUNTIME_TXN_ERR_ACCOUNT_IN_USE and out_conflict_detected
   will be 1 (read-write) or 2 (write-write), and out_conflict_addr_opt,
   if not NULL, will hold the account address causing the conflict.
 */
struct fd_conflict_detect_ele {
  fd_acct_addr_t key;
  uchar          writable;
};
typedef struct fd_conflict_detect_ele fd_conflict_detect_ele_t;
#define FD_TXN_CONFLICT_MAP_SEED        (0UL)
#define FD_TXN_CONFLICT_MAP_MAX_NACCT   (FD_SHRED_DATA_PAYLOAD_MAX_PER_SLOT / FD_TXN_MIN_SERIALIZED_SZ * FD_TXN_ACCT_ADDR_MAX)

static const fd_acct_addr_t fd_acct_addr_null = {.b={0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF}};
#define MAP_NAME              fd_conflict_detect_map
#define MAP_KEY_T             fd_acct_addr_t
#define MAP_T                 fd_conflict_detect_ele_t
#define MAP_HASH_T            ulong
#define MAP_KEY_NULL          (fd_acct_addr_null)
#define MAP_KEY_EQUAL(k0,k1)  (0==memcmp((k0).b,(k1).b,32))
#define MAP_KEY_HASH(key)     fd_hash( FD_TXN_CONFLICT_MAP_SEED, key.b, 32 )
#define MAP_KEY_INVAL(k)      (0==memcmp(&fd_acct_addr_null, (k).b, 32))
#define MAP_KEY_EQUAL_IS_SLOW 0
#define MAP_MEMOIZE           0

#include "../../util/tmpl/fd_map_dynamic.c"

#define FD_RUNTIME_NO_CONFLICT_DETECTED          0
#define FD_RUNTIME_READ_WRITE_CONFLICT_DETECTED  1
#define FD_RUNTIME_WRITE_WRITE_CONFLICT_DETECTED 2

int
fd_runtime_microblock_verify_read_write_conflicts( fd_txn_p_t *               txns,
                                                   ulong                      txn_cnt,
                                                   fd_conflict_detect_ele_t * acct_map,
                                                   fd_acct_addr_t *           acct_arr,
                                                   fd_funk_t *                funk,
                                                   fd_funk_txn_t *            funk_txn,
                                                   ulong                      slot,
                                                   fd_slot_hash_t *           slot_hashes,
                                                   fd_features_t *            features,
                                                   int *                      out_conflict_detected,
                                                   fd_acct_addr_t *           out_conflict_addr_opt );

/* Load the accounts in the address lookup tables of txn into out_accts_alt */
int
fd_runtime_load_txn_address_lookup_tables( fd_txn_t const * txn,
                                           uchar const *    payload,
                                           fd_funk_t *      funk,
                                           fd_funk_txn_t *  funk_txn,
                                           ulong            slot,
                                           fd_slot_hash_t * hashes,
                                           fd_acct_addr_t * out_accts_alt );

/* fd_runtime_poh_verify is responsible for verifying poh hashes while
   streaming in microblocks. */

void
fd_runtime_poh_verify( fd_poh_verifier_t * poh_info );

int
fd_runtime_block_execute_prepare( fd_exec_slot_ctx_t * slot_ctx,
                                  fd_spad_t *          runtime_spad );

void
fd_runtime_block_execute_finalize_start( fd_exec_slot_ctx_t *             slot_ctx,
                                         fd_spad_t *                      runtime_spad,
                                         fd_accounts_hash_task_data_t * * task_data,
                                         ulong                            lt_hash_cnt );

int
fd_runtime_block_execute_finalize_finish( fd_exec_slot_ctx_t *             slot_ctx,
                                          fd_capture_ctx_t *               capture_ctx,
                                          fd_runtime_block_info_t const *  block_info,
                                          fd_spad_t *                      runtime_spad,
                                          fd_accounts_hash_task_data_t *   task_data,
                                          ulong                            lt_hash_cnt );

int
fd_runtime_block_execute_finalize_para( fd_exec_slot_ctx_t *             slot_ctx,
                                        fd_capture_ctx_t *               capture_ctx,
                                        fd_runtime_block_info_t const *  block_info,
                                        ulong                            worker_cnt,
                                        fd_spad_t *                      runtime_spad,
                                        fd_exec_para_cb_ctx_t *          exec_para_ctx );

/* Transaction Level Execution Management *************************************/

/* fd_runtime_prepare_txns_start and fd_runtime_pre_execute_check are only
   publicly exposed for the fuzzing harnesses. These functions are responsible
   for various transaction sanitization checks. */

int
fd_runtime_prepare_txns_start( fd_exec_slot_ctx_t *         slot_ctx,
                               fd_execute_txn_task_info_t * task_info,
                               fd_txn_p_t *                 txns,
                               ulong                        txn_cnt,
                               fd_spad_t *                  runtime_spad );

void
fd_runtime_pre_execute_check( fd_execute_txn_task_info_t * task_info );

/* fd_runtime_process_txns is responsible for end-to-end preparing, executing,
   and finalizing a list of transactions. It will execute all of the
   transactions on a single core. */
int
fd_runtime_process_txns( fd_exec_slot_ctx_t * slot_ctx,
                         fd_capture_ctx_t *   capture_ctx,
                         fd_txn_p_t *         txns,
                         ulong                txn_cnt,
                         fd_spad_t *          exec_spad,
                         fd_spad_t *          runtime_spad );

void
fd_runtime_finalize_txn( fd_funk_t *                  funk,
                         fd_funk_txn_t *              funk_txn,
                         fd_execute_txn_task_info_t * task_info,
                         fd_spad_t *                  finalize_spad,
                         fd_bank_t *                  bank );

/* Epoch Boundary *************************************************************/

uint
fd_runtime_is_epoch_boundary( fd_exec_slot_ctx_t * slot_ctx,
                              ulong                curr_slot,
                              ulong                prev_slot );

/*
   This is roughly Agave's process_new_epoch() which gets called from
   new_from_parent() for every slot.
   https://github.com/anza-xyz/agave/blob/v1.18.26/runtime/src/bank.rs#L1483
   This needs to be called after funk_txn_prepare() because the accounts
   that we modify when processing a new epoch need to be hashed into
   the bank hash.
 */
void
fd_runtime_block_pre_execute_process_new_epoch( fd_exec_slot_ctx_t * slot_ctx,
                                                fd_spad_t * *        exec_spads,
                                                ulong                exec_spad_cnt,
                                                fd_spad_t *          runtime_spad,
                                                int *                is_epoch_boundary );

/* This function is responsible for inserting fresh entries or updating existing entries in the program cache.
   When the client boots up, the program cache is empty. As programs get invoked, this function is responsible
   for verifying and inserting these programs into the cache. Additionally, this function performs reverification
   checks for every existing program that is invoked after an epoch boundary once per program per epoch, and invalidates
   any programs that fail reverification.

   When the cluster's feature set changes at an epoch, there is a possibility that existing programs
   fail new SBPF / ELF header checks. Therefore, after every epoch, we should reverify all programs
   and update our program cache so that users cannot invoke those old programs. Since iterating through
   all programs every single epoch is expensive, we adopt a lazy approach where we reverify programs as they
   are referenced in transactions, since only a small subset of all programs are actually referenced at any
   time. We also make sure each program is only verified once per epoch, so repeated executions of a
   program within the same epoch will only trigger verification once at the very first invocation.

   Note that ALUTs must be resolved because programs referenced in ALUTs can be invoked via CPI. */
void
fd_runtime_update_program_cache( fd_exec_slot_ctx_t * slot_ctx,
                                 fd_txn_p_t const *   txn_p,
                                 fd_spad_t *          runtime_spad );

/* Debugging Tools ************************************************************/

void
fd_runtime_checkpt( fd_capture_ctx_t *   capture_ctx,
                    fd_exec_slot_ctx_t * slot_ctx,
                    ulong                slot );

/* Offline Replay *************************************************************/

int
fd_runtime_block_execute( fd_exec_slot_ctx_t *            slot_ctx,
                          fd_capture_ctx_t *              capture_ctx,
                          fd_runtime_block_info_t const * block_info,
                          fd_spad_t *                     runtime_spad );

int
fd_runtime_process_txns_in_microblock_stream_sequential( fd_exec_slot_ctx_t * slot_ctx,
                                                         fd_capture_ctx_t *   capture_ctx,
                                                         fd_txn_p_t *         txns,
                                                         ulong                txn_cnt,
                                                         fd_spad_t *          runtime_spad,
                                                         fd_cost_tracker_t *  cost_tracker_opt );

int
fd_runtime_block_execute_finalize_sequential( fd_exec_slot_ctx_t *             slot_ctx,
                                              fd_capture_ctx_t *               capture_ctx,
                                              fd_runtime_block_info_t const *  block_info,
                                              fd_spad_t *                      runtime_spad );

void
fd_runtime_read_genesis( fd_exec_slot_ctx_t * slot_ctx,
                         char const *         genesis_filepath,
                         uchar                is_snapshot,
                         fd_capture_ctx_t *   capture_ctx,
                         fd_spad_t *          spad );

FD_PROTOTYPES_END

#endif /* HEADER_fd_src_flamenco_runtime_fd_runtime_h */
