use {
    crate::compute_commit,
    serde_derive::{Deserialize, Serialize},
    solana_sanitize::Sanitize,
    std::{convert::TryInto, fmt},
};

// Older version structure used earlier 1.3.x releases
#[cfg_attr(feature = "frozen-abi", derive(AbiExample))]
#[derive(Serialize, Deserialize, Clone, Debug, PartialEq, Eq)]
pub struct LegacyVersion1 {
    major: u16,
    minor: u16,
    patch: u16,
    commit: Option<u32>, // first 4 bytes of the sha1 commit hash
}

impl Sanitize for LegacyVersion1 {}

#[cfg_attr(feature = "frozen-abi", derive(AbiExample))]
#[derive(Serialize, Deserialize, Clone, PartialEq, Eq)]
pub struct LegacyVersion2 {
    pub major: u16,
    pub minor: u16,
    pub patch: u16,
    pub commit: Option<u32>, // first 4 bytes of the sha1 commit hash
    pub feature_set: u32,    // first 4 bytes of the FeatureSet identifier
}

extern "C" {
    pub(crate) static fdctl_major_version: u64;
    pub(crate) static fdctl_minor_version: u64;
    pub(crate) static fdctl_patch_version: u64;
    pub(crate) static fdctl_commit_ref: u32;
}

impl Default for LegacyVersion2 {
    fn default() -> Self {
        let feature_set =
            u32::from_le_bytes(agave_feature_set::ID.as_ref()[..4].try_into().unwrap());
        let _ = compute_commit(None);
        Self {
            major: unsafe { fdctl_major_version as u16 },
            minor: unsafe { fdctl_minor_version as u16 },
            patch: unsafe { fdctl_patch_version as u16 },
            commit: Some(unsafe { fdctl_commit_ref }),
            feature_set,
        }
    }
}

impl fmt::Debug for LegacyVersion2 {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(
            f,
            "{}.{}.{} (src:{}; feat:{})",
            self.major,
            self.minor,
            self.patch,
            match self.commit {
                None => "devbuild".to_string(),
                Some(commit) => format!("{commit:08x}"),
            },
            self.feature_set,
        )
    }
}

impl Sanitize for LegacyVersion2 {}
