use crate::affinity_manager::AffinityManager;
use crate::config::Config;
use crate::core_group_builder::CoreGroupBuilder;
use clap::{value_t_or_exit, ArgMatches};
use log::info;
use once_cell::sync::OnceCell;
use solana_core::validator::ValidatorConfig;
use std::path::PathBuf;

pub mod affinity_manager;
pub mod config;
pub mod core_group_builder;

pub static AFFINITY_MANAGER: OnceCell<AffinityManager> = OnceCell::new();

/// Initialzies the name based affinity manager
pub fn init_cores(config: &Config) {
    if !config.affinity.use_affinity_manager {
        info!("[fogo_aff] Not affinity manager");
        return;
    }

    info!("[fogo_aff] Loading affinity manager");
    let manager = AffinityManager::new();
    let mut groups = CoreGroupBuilder::new(manager.available_cores());

    if let Some(rpc_cores) = config.affinity.rpc_core_count {
        let rpc_threads = groups.take(rpc_cores);
        manager.add_rule("solAdminRpc*", &rpc_threads);
        manager.add_rule("solRpc*", &rpc_threads);
    }

    let remaining = groups.take_rest();
    manager.add_rule("sol*", &remaining);

    manager.start();
    AFFINITY_MANAGER
        .set(manager)
        .map_err(|_| "Failed to set affinity manager")
        .unwrap();
}

/// Applies the fogo overrides
pub fn apply_overrides(arg_matches: &ArgMatches, validator_config: &mut ValidatorConfig) {
    let config = if arg_matches.is_present("fogo_config") {
        config::load_config(&value_t_or_exit!(arg_matches, "fogo_config", PathBuf))
            .expect("Failed to load config")
    } else {
        Default::default()
    };

    info!("[fogo_validator] Fogo config: {} hi!", config.motd);

    // apply config here
    if let Some(threads) = config.threads.rpc {
        validator_config.rpc_config.rpc_threads = threads;
    }

    if let Some(threads) = config.threads.rpc_blocking {
        validator_config.rpc_config.rpc_blocking_threads = threads;
    }

    // rpc config application
    if let Some(enable) = config.rpc.enable_block_pubsub {
        validator_config.pubsub_config.enable_block_subscription = enable;
    }
    if let Some(enable) = config.rpc.enable_vote_pubsub {
        validator_config.pubsub_config.enable_vote_subscription = enable;
    }
    if let Some(val) = config.rpc.max_active_subscriptions {
        validator_config.pubsub_config.max_active_subscriptions = val;
    }
    if let Some(val) = config.rpc.scan_limit_bytes {
        validator_config
            .accounts_db_config
            .as_mut()
            .expect("no account db")
            .index
            .as_mut()
            .expect("no index config")
            .scan_results_limit_bytes = Some(val);
    }

    init_cores(&config);
}
